/* chcontext -- change security context of a pathname */

#include <config.h>
#include <stdio.h>
#include <sys/types.h>
#include <grp.h>
#include <getopt.h>
#include <fs_secure.h>

#include "system.h"
#include "xstrtoul.h"
#include "closeout.h"
#include "error.h"
#include "savedir.h"
#include "group-member.h"

enum Change_status
{
  CH_SUCCEEDED,
  CH_FAILED,
  CH_NO_CHANGE_REQUESTED
};

enum Verbosity
{
  /* Print a message for each file that is processed.  */
  V_high,

  /* Print a message for each file whose attributes we change.  */
  V_changes_only,

  /* Do not be verbose.  This is the default. */
  V_off
};

static int change_dir_context PARAMS ((const char *dir, int sid,
				       const struct stat *statp));

/* The name the program was run with. */
char *program_name;

/* If nonzero, and the systems has support for it, change the context
   of symbolic links rather than any files they point to.  */
static int change_symlinks;

/* If nonzero, change the context of directories recursively. */
static int recurse;

/* If nonzero, force silence (no error messages). */
static int force_silent;

/* Level of verbosity.  */
static enum Verbosity verbosity = V_off;

/* The name of the context the file is being given. */
static const char *contextname;

/* The argument to the --reference option.  Use the context SID of this file.
   This file must exist.  */
static char *reference_file;

/* If nonzero, display usage information and exit.  */
static int show_help;

/* If nonzero, print the version on standard output and exit.  */
static int show_version;

static struct option const long_options[] =
{
  {"recursive", no_argument, 0, 'R'},
  {"changes", no_argument, 0, 'c'},
  {"no-dereference", no_argument, 0, 'h'},
  {"silent", no_argument, 0, 'f'},
  {"quiet", no_argument, 0, 'f'},
  {"reference", required_argument, 0, CHAR_MAX + 1},
  {"sid", required_argument, 0, CHAR_MAX + 2},
  {"verbose", no_argument, 0, 'v'},
  {"help", no_argument, &show_help, 1},
  {"version", no_argument, &show_version, 1},
  {0, 0, 0, 0}
};

/* Tell the user how/if the context of FILE has been changed.
   CHANGED describes what (if anything) has happened. */

static void
describe_change (const char *file, enum Change_status changed)
{
  const char *fmt;
  switch (changed)
    {
    case CH_SUCCEEDED:
      fmt = _("context of %s changed to %s\n");
      break;
    case CH_FAILED:
      fmt = _("failed to change context of %s to %s\n");
      break;
    case CH_NO_CHANGE_REQUESTED:
      fmt = _("context of %s retained as %s\n");
      break;
    default:
      abort ();
    }
  printf (fmt, file, contextname);
}

/* Change the context of FILE to SID CONTEXT.
   If it is a directory and -R is given, recurse.
   Return 0 if successful, 1 if errors occurred. */

static int
change_file_context (const char *file, int sid)
{
  struct stat file_stats;
  security_id_t file_sid;
  int errors = 0;

  if (lstat_secure (file, &file_stats, &file_sid))
    {
      if (force_silent == 0)
	error (0, errno, "%s", file);
      return 1;
    }

  if (sid != file_sid)
    {
      int fail;

      if (change_symlinks)
      	fail = lchsid (file, sid);
      else
      	fail = chsid (file, sid);

      if (verbosity == V_high || (verbosity == V_changes_only && !fail))
	describe_change (file, (fail ? CH_FAILED : CH_SUCCEEDED));

      if (fail)
	{
	  errors = 1;
	  if (force_silent == 0)
	    {
	      error (0, errno, "%s", file);
	    }
	}
    }
  else if (verbosity == V_high)
    {
      describe_change (file, CH_NO_CHANGE_REQUESTED);
    }

  if (recurse && S_ISDIR (file_stats.st_mode))
    errors |= change_dir_context (file, sid, &file_stats);

  return errors;
}

/* Recursively change context of the files in directory DIR
   to SID CONTEXT.
   STATP points to the results of lstat on DIR.
   Return 0 if successful, 1 if errors occurred. */

static int
change_dir_context (const char *dir, int sid, const struct stat *statp)
{
  char *name_space, *namep;
  char *path;			/* Full path of each entry to process. */
  unsigned dirlength;		/* Length of `dir' and '\0'. */
  unsigned filelength;		/* Length of each pathname to process. */
  unsigned pathlength;		/* Bytes allocated for `path'. */
  int errors = 0;

  errno = 0;
  name_space = savedir (dir, (unsigned int) statp->st_size);
  if (name_space == NULL)
    {
      if (errno)
        {
	  if (force_silent == 0)
	    error (0, errno, "%s", dir);
	  return 1;
	}
      else
	error (1, 0, _("virtual memory exhausted"));
    }

  dirlength = strlen (dir) + 1;	/* + 1 is for the trailing '/'. */
  pathlength = dirlength + 1;
  /* Give `path' a dummy value; it will be reallocated before first use. */
  path = xmalloc (pathlength);
  strcpy (path, dir);
  path[dirlength - 1] = '/';

  for (namep = name_space; *namep; namep += filelength - dirlength)
    {
      filelength = dirlength + strlen (namep) + 1;
      if (filelength > pathlength)
	{
	  pathlength = filelength * 2;
	  path = xrealloc (path, pathlength);
	}
      strcpy (path + dirlength, namep);
      errors |= change_file_context (path, sid);
    }
  free (path);
  free (name_space);
  return errors;
}

static void
usage (int status)
{
  if (status != 0)
    fprintf (stderr, _("Try `%s --help' for more information.\n"),
	     program_name);
  else
    {
      printf (_("\
Usage: %s [OPTION]... CONTEXT FILE...\n\
  or:  %s [OPTION]... --reference=RFILE FILE...\n\
  or:  %s [OPTION]... --sid=SID FILE...\n\
"),
	program_name, program_name, program_name);
      printf (_("\
Change the security context of each FILE to CONTEXT.\n\
\n\
  -c, --changes          like verbose but report only when a change is made\n\
  -h, --no-dereference   affect symbolic links instead of any referenced file\n\
                         (available only on systems with lchown system call)\n\
  -f, --silent, --quiet  suppress most error messages\n\
      --reference=RFILE  use RFILE's group instead of using a CONTEXT value\n\
      --sid=SID          use context corresponding to SID for CONTEXT value\n\
  -R, --recursive        change files and directories recursively\n\
  -v, --verbose          output a diagnostic for every file processed\n\
      --help             display this help and exit\n\
      --version          output version information and exit\n\
"));
      close_stdout ();
    }
  exit (status);
}

int
main (int argc, char **argv)
{
  int sid = -1;
  int errors = 0;
  int optc;
  
  program_name = argv[0];
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  recurse = force_silent = 0;
  
  while ((optc = getopt_long (argc, argv, "Rcfhv", long_options, NULL)) != -1)
  {
	  switch (optc)
	  {
	  case 0:
		  break;
	  case CHAR_MAX + 1:
		  reference_file = optarg;
		  break;
	  case CHAR_MAX +2:
	          sid = (int)strtol( optarg, (char **)NULL, 10);
		  /* We use an int to represent sids so we can use -1 for *
		   * an error condition.  All valid sids are positive.    *
		   * Check for an invalid sid.                            */
                  if( ( errno == ERANGE ) || ( sid <= 0 ) ) {  
		    error( 1, 0, _("invalid sid"));
		  }
	          break;
	  case 'R':
		  recurse = 1;
		  break;
 	  case 'c':
		  verbosity = V_changes_only;
		  break;
	  case 'f':
		  force_silent = 1;
		  break;
	  case 'h':
		  change_symlinks = 1;
		  break;
	  case 'v':
		  verbosity = V_high;
		  break;
	  default:
		  usage (1);
	  }
  }

  if (show_version)
  {
     printf ("chcon (%s) %s\n", GNU_PACKAGE, VERSION);
     close_stdout ();
     exit (0);
  }

  if (show_help)
    usage (0);

  if (argc - optind + ( (reference_file || ( sid > 0 ) ) ? 1 : 0) <= 1)
  {
     error (0, 0, _("too few arguments"));
     usage (1);
  }
  
  if (reference_file)
    {
      struct stat ref_stats;
      security_id_t ref_sid;
      if (stat_secure (reference_file, &ref_stats, &ref_sid))
	error (1, errno, "%s", reference_file);

      sid = ref_sid;
    }
  else if( sid <= 0 )  /* sid > 0 means sid already set by --sid above */
   {
     contextname = argv[optind++];

     if (*contextname == '\0')
       error (1, 0, _("can not change to null context"));

     if (security_context_to_sid (contextname, strlen (contextname) + 1, &sid))
       error (1, errno, "%s", contextname);
   }

  for (; optind < argc; ++optind)
     errors |= change_file_context (argv[optind], sid);

  if (verbosity != V_off)
    close_stdout ();
  exit (errors);
}
