/* mkdir -- make directories
   Copyright (C) 90, 95, 96, 97, 1998 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* David MacKenzie <djm@ai.mit.edu>  */

#include <config.h>
#include <stdio.h>
#include <getopt.h>
#include <sys/types.h>

#include "system.h"
#include "modechange.h"
#include "makepath.h"
#include "closeout.h"
#include "error.h"

#ifdef FLASK_LINUX
#include <fs_secure.h>
#include <flask_util.h>          /* for is_flask_enabled() */
#define CTXTLEN 256
char *scontext = NULL;
int ctxtlen = CTXTLEN;
char *calloc();
#endif FLASK_LINUX

/* The name this program was run with. */
char *program_name;

/* If nonzero, ensure that all parents of the specified directory exist.  */
static int path_mode;

/* If nonzero, display usage information and exit.  */
static int show_help;

/* If nonzero, print the version on standard output and exit.  */
static int show_version;

static struct option const longopts[] =
{
#ifdef FLASK_LINUX
  {"sid",     required_argument, NULL, 's'},
  {"context", required_argument, NULL, 'c'},
#endif FLASK_LINUX
  {"mode", required_argument, NULL, 'm'},
  {"parents", no_argument, NULL, 'p'},
  {"help", no_argument, &show_help, 1},
  {"verbose", no_argument, NULL, 2},
  {"version", no_argument, &show_version, 1},
  {NULL, 0, NULL, 0}
};

static void
usage (int status)
{
  if (status != 0)
    fprintf (stderr, _("Try `%s --help' for more information.\n"),
	     program_name);
  else
    {
      printf (_("Usage: %s [OPTION] DIRECTORY...\n"), program_name);
#ifdef FLASK_LINUX
      printf (_("\
Create the DIRECTORY(ies), if they do not already exist.\n\
\n\
  -s, --sid=SID         (Flask) set security ID to SID\n\
  -c, --context=CONTEXT (Flask) set security context to CONTEXT\n\
  -m, --mode=MODE   set permission mode (as in chmod), not rwxrwxrwx - umask\n\
  -p, --parents     no error if existing, make parent directories as needed\n\
      --verbose     print a message for each created directory\n\
      --help        display this help and exit\n\
      --version     output version information and exit\n\
"));
#else  FLASK_LINUX
      printf (_("\
Create the DIRECTORY(ies), if they do not already exist.\n\
\n\
  -m, --mode=MODE   set permission mode (as in chmod), not rwxrwxrwx - umask\n\
  -p, --parents     no error if existing, make parent directories as needed\n\
      --verbose     print a message for each created directory\n\
      --help        display this help and exit\n\
      --version     output version information and exit\n\
"));
#endif FLASK_LINUX
      puts (_("\nReport bugs to <bug-fileutils@gnu.org>."));
      close_stdout ();
    }
  exit (status);
}

main (int argc, char **argv)
{

#ifdef FLASK_LINUX
  security_id_t sid = -1;
  char *scontext = NULL;
  int rv;
#endif FLASK_LINUX
  unsigned int newmode;
  unsigned int parent_mode;
  char *symbolic_mode = NULL;
  const char *verbose_fmt_string = NULL;
  int errors = 0;
  int optc;
  int is_flask_enabled_flag;  /* set iff kernel has extra flask system calls */

  /* Set `is_flask_enabled_flag' iff the kernel has the extra flask syscalls */
  is_flask_enabled_flag = is_flask_enabled();

  program_name = argv[0];
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  path_mode = 0;

#ifdef FLASK_LINUX
  while ((optc = getopt_long (argc, argv, "pm:s:c:", longopts, NULL)) != -1)
#else  FLASK_LINUX
  while ((optc = getopt_long (argc, argv, "pm:", longopts, NULL)) != -1)
#endif FLASK_LINUX
    {
      switch (optc)
	{
	case 0:			/* Long option. */
	  break;
	case 'p':
	  path_mode = 1;
	  break;
	case 'm':
	  symbolic_mode = optarg;
	  break;
	case 2: /* --verbose  */
	  verbose_fmt_string = _("created directory `%s'");
	  break;
#ifdef FLASK_LINUX
	case 's':
	  /* politely decline if we're not on a flask-enabled kernel. */
	  if( !is_flask_enabled_flag ) {
	    fprintf( stderr, "Sorry, --sid (-s) can be used only on "
		             "a flask-enabled kernel.\n" );
	    exit( 1 );
	  }
	  if ( ((int) sid > 0) || (scontext != NULL) ) {
	    (void) fprintf(stderr, "%s: --sid (-s) and --context (-c) are mutually exclusive\n", argv[0]);
	    exit( 1 );
	  }
	  {
	    /* check for typos */
	    char *ep;
	    sid = (security_id_t) strtol(optarg, &ep, 10);
	    if ( *ep ) {
	      (void) fprintf(stderr, "%s: non-numeric SID '%s'\n", argv[0], optarg);
	      exit( 1 );
	    }
	  }
	  /* do sanity check, save result on success */
	  scontext = calloc(1, ctxtlen+1);
	  if ( scontext != NULL ) {
	    if ( security_sid_to_context(sid, scontext, &ctxtlen) ) {
	      if ( errno != ENOSPC ) {
		(void) fprintf(stderr, "%s: security_sid_to_context(%d): '%s'\n", argv[0], (int) sid, strerror(errno));
		exit( 1 );
	      }
	      free(scontext);
	      scontext = calloc(1, ctxtlen+1);
	      /* nonfatal, so if there's an error we punt */
	      if ( scontext != NULL )
		if ( security_sid_to_context(sid, scontext, &ctxtlen) ) {
		  (void) fprintf(stderr, "%s: security_sid_to_context(%d): %s\n", argv[0], (int) sid, strerror(errno));
		  exit( 1 );
		}
	    }
	  }
	  break;
	case 'c':
	  /* politely decline if we're not on a flask-enabled kernel. */
	  if( !is_flask_enabled_flag ) {
	    fprintf( stderr, "Sorry, --context (-c) can be used only on "
		             "a flask-enabled kernel.\n" );
	    exit( 1 );
	  }
	  if ( ((int) sid >= 0) || (scontext != NULL) ) {
	    (void) fprintf(stderr, "%s: --context (-c) and --sid (-s) are mutually exclusive\n", argv[0]);
	    exit( 1 );
	  }
	  scontext = optarg;
	  /* sanity check */
	  rv = security_context_to_sid(scontext, strlen(scontext)+1, &sid);
	  if ( rv ) {
	    (void) fprintf(stderr, "%s: security_context_to_sid(%s): %s\n", argv[0], scontext, strerror(errno));
	    exit( 1 );
	  }
	  break;
#endif FLASK_LINUX
	default:
	  usage (1);
	}
    }

  if (show_version)
    {
#ifdef FLASK_LINUX
      printf ("mkdir (Flask) (%s) %s\n", GNU_PACKAGE, VERSION);
#else
      printf ("mkdir (%s) %s\n", GNU_PACKAGE, VERSION);
#endif
      close_stdout ();
      exit (0);
    }

  if (show_help)
    usage (0);

  if (optind == argc)
    {
      error (0, 0, _("too few arguments"));
      usage (1);
    }

  newmode = 0777 & ~umask (0);
  parent_mode = newmode | 0300;	/* u+wx */
  if (symbolic_mode)
    {
      struct mode_change *change = mode_compile (symbolic_mode, 0);
      if (change == MODE_INVALID)
	error (1, 0, _("invalid mode `%s'"), symbolic_mode);
      else if (change == MODE_MEMORY_EXHAUSTED)
	error (1, 0, _("virtual memory exhausted"));
      newmode = mode_adjust (newmode, change);
    }

  for (; optind < argc; ++optind)
    {
      if (path_mode)
	{
#ifdef FLASK_LINUX
	  /* `sid' will be > 0 iff the --sid (-s) option was specified
	   * on the command line.  The --sid argument processing code
	   * exits politely if the kernel doesn't support the new
	   * flask system calls.  So, if sid > 0 at this point, we
	   * know we're on a flask-enabled kernel.
	   *
	   * make_path_s() calls make_path(), passing `sid' to it via
	   * a global variable `Sid' made specially for that purpose.
	   * The flask-specific code in make_path() calls
	   * flask-specific system calls only if `sid' (aka `Sid') is
	   * > 0.
	   *
	   * Put these two facts together, and we know that make_path_s()
	   * will make flask-specific system calls only if we are on a
	   * flask-enabled kernel.  Consequently, it's safe to call
	   * make_path_s() whether or not we're on a flask-enabled kernel.
	   */
	  errors |= make_path_s (argv[optind], newmode, parent_mode,
				 -1, -1, 1, verbose_fmt_string, sid);
#else  FLASK_LINUX
	  errors |= make_path (argv[optind], newmode, parent_mode,
			       -1, -1, 1, verbose_fmt_string);
#endif FLASK_LINUX
	}
#ifdef FLASK_LINUX
      /* `sid' will be > 0 iff the --sid (-s) option was specified on the  *
       * command line.  The --sid argument processing code exits politely  *
       * if the kernel doesn't support the new flask system calls.  So,    *
       * if sid > 0 at this point, we know we're on a flask-enabled kernel *
       * and can call the "_secure" version of mkdir.                      */
      else if ( (int) sid > 0 ) {
	if ( mkdir_secure (argv[optind], newmode, sid) < 0 ) {
	  error (0, errno, _("Cannot make directory `%s'"), argv[optind]);
	  errors = 1;
	}
      }
#endif FLASK_LINUX
      else if (mkdir (argv[optind], newmode))
	{
	  error (0, errno, _("cannot make directory `%s'"), argv[optind]);
	  errors = 1;
	}
      else if (verbose_fmt_string)
	{
	  error (0, 0, verbose_fmt_string, argv[optind]);
	}
    }

  exit (errors);
}
