/************************************************************************
 *
 * newrole
 *
 * SYNOPSIS:
 *
 * This program allows a user to change their FLASK RBAC role and/or
 * FLASK type (domain) in a manner similar to the way the traditional
 * UNIX su program allows a user to change their identity.
 *
 * USAGE:
 *
 * newrole -r <role> [optional args to shell...]
 * newrole -t <type> [optional args to shell...]
 * newrole -r <role> -t <type> [optional args to shell...]
 *
 * BUILD OPTIONS:
 *
 * option USE_PAM:
 *
 * Set the USE_PAM constant if you want to authenticate users via PAM.
 * If USE_PAM is not set, users will be authenticated via direct
 * access to the shadow password file.
 *
 * If you decide to use PAM must be told how to handle newrole.  A
 * good rule-of-thumb might be to tell PAM to handle newrole in the
 * same way it handles su.  In fact, you might do this with a symbolic
 * link:
 *   su
 *   cd /etc/pam.d
 *   ln -s su newrole
 *
 * If you choose not to use PAM, make sure you have a shadow passwd file
 * in /etc/shadow.  You can use a simlink if your shadow passwd file
 * lives in another directory.  Example:
 *   su
 *   cd /etc
 *   ln -s /etc/auth/shadow shadow
 *
 * If you decide not to use PAM, you will also have to make newrole
 * setuid root, so that it can read the shadow passwd file.
 * 
 *
 * option CANTSPELLGDB:
 *
 * If you set CANTSPELLGDB you will turn on some debugging printfs.
 *
 *
 * WHO TO BLAME:  Tim Fraser
 *
 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>               /* for malloc(), realloc(), free() */
#include <pwd.h>                  /* for getpwuid() */
#include <sys/types.h>            /* to make getuid() and getpwuid() happy */
#include <sys/wait.h>		  /* for wait() */
#include <sys/stat.h>		  /* for struct stat and friends */
#define _GNU_SOURCE
#include <getopt.h>               /* for getopt_long() form of getopt() */
#include <flask_util.h>           /* for is_flask_enabled() */
#include <proc_secure.h>          /* for getsecsid() */
#include <fs_secure.h>
#include <fcntl.h>
#include <ss.h>                   /* for sid<->context routines */
#include <context.h>              /* for context-mangling functions */
#include <get_default_type.h>

/* USAGE_STRING describes the command-line args of this program. */
#define USAGE_STRING "USAGE: newrole -r role [ -t type ] [ shell args ]"

#define DEFAULT_CONTEXT_SIZE 255  /* first guess at context size */

int sid_to_dynamic_context(security_id_t sid, security_context_t* scontext,
				     __u32 *scontext_len)
{
  int context_length;                

  /* Put the context corresponding to `sid' into `context_s'. Getting *
   * the context from the kernel is tricky - we must dynamically      *
   * allocate a buffer (named `context') that is "big enough" to hold *
   * whatever the kernel gives us.  If our intitial guess at "big     *
   * enough" is too small, our initial request for a context will     *
   * fail.  Fortunately, if our initial request does fail, it will be *
   * kind enough to tell us the proper size.  In this case, we will   *
   * realloc the `context_s' buffer to the proper size, and make a    *
   * second call.  Since we know the proper size on the second call,  *
   * we can be confident that it will succeed.                        */


  context_length = DEFAULT_CONTEXT_SIZE;
  if( !(*scontext = (security_context_t)malloc(context_length)) )
    return(-1);

  if( security_sid_to_context(sid,*scontext,&context_length) ) {
    /* our initial guess at `context_length' was too small.  The      *
     * failed call should have updated `context_length' to the proper *
     * length.  Try again.                                            */
    if( !(*scontext = (security_context_t)realloc(scontext, context_length)) )
      return(-1);
    
    if( security_sid_to_context(sid,*scontext,&context_length) )
      return(-1);

  }
  return(0);
}

#ifdef USE_PAM

/************************************************************************
 *
 * All PAM code goes in this section.
 *
 ************************************************************************/

#include <unistd.h>               /* for getuid(), exit(), getopt() */

#include <security/pam_appl.h>    /* for PAM functions */
#include <security/pam_misc.h>    /* for misc_conv PAM utility function */

#define SERVICE_NAME "newrole"    /* the name of this program for PAM */

int authenticate_via_pam( const struct passwd * );

/* authenticate_via_pam()
 *
 * in:     p_passwd_line - struct containing data from our user's line in 
 *                         the passwd file.
 * out:    nothing
 * return: value   condition
 *         -----   ---------
 *           1     PAM thinks that the user authenticated themselves properly
 *           0     otherwise
 *
 * This function uses PAM to authenticate the user running this
 * program.  This is the only function in this program that makes PAM
 * calls.
 *
 */

int authenticate_via_pam( const struct passwd *p_passwd_line ) {

  int result = 0;    /* our result, set to 0 (not authenticated) by default */
  pam_handle_t *pam_handle;      /* opaque handle used by all PAM functions */

  /* This is a jump table of functions for PAM to use when it wants to *
   * communicate with the user.  We'll be using misc_conv(), which is  *
   * provided for us via pam_misc.h.                                   */
  struct pam_conv pam_conversation = {
    misc_conv,
    NULL
  };

  /* Make `p_pam_handle' a valid PAM handle so we can use it when *
   * calling PAM functions.                                       */
  if( PAM_SUCCESS != pam_start( SERVICE_NAME,
				p_passwd_line->pw_name,
				&pam_conversation,
				&pam_handle ) ) {
    fprintf( stderr, "failed to initialize PAM\n" );
    exit( -1 );
  }

  /* Ask PAM to authenticate the user running this program */
  if( PAM_SUCCESS == pam_authenticate(pam_handle,0) ) {
    result = 1;  /* user authenticated OK! */
  }

  /* We're done with PAM.  Free `pam_handle'. */
  pam_end( pam_handle, PAM_SUCCESS );
 
  return( result );

} /* authenticate_via_pam() */

#else /* else !USE_PAM */


/************************************************************************
 *
 * All shadow passwd code goes in this section.
 *
 ************************************************************************/


#define _XOPEN_SOURCE                       /* for crypt() in unistd.h */

#include <unistd.h>                         /* for getuid(), exit(), crypt() */
#include <shadow.h>                         /* for shadow passwd functions */
#include <string.h>                         /* for strlen(), memset() */

#define PASSWORD_PROMPT "Password:"         /* prompt for getpass() */

int authenticate_via_shadow_passwd( const struct passwd * );

/* authenticate_via_shadow_passwd()
 *
 * in:     p_passwd_line - struct containing data from our user's line in 
 *                         the passwd file.
 * out:    nothing
 * return: value   condition
 *         -----   ---------
 *           1     user authenticated themselves properly according to the
 *                 shadow passwd file.
 *           0     otherwise
 *
 * This function uses the shadow passwd file to thenticate the user running
 * this program.
 *
 */

int authenticate_via_shadow_passwd( const struct passwd *p_passwd_line ) {

  struct spwd *p_shadow_line; /* struct derived from shadow passwd file line */
  char *unencrypted_password_s;        /* unencrypted password input by user */
  char *encrypted_password_s; /* user's password input after being crypt()ed */

  /* Make `p_shadow_line' point to the data from the current user's *
   * line in the shadow passwd file.                                */
  setspent();            /* Begin access to the shadow passwd file. */
  p_shadow_line = getspnam( p_passwd_line->pw_name );
  endspent();            /* End access to the shadow passwd file. */
  if( !( p_shadow_line ) ) {
    fprintf( stderr, "Cannot find your entry in the shadow passwd file.\n" );
    exit( -1 );
  }

  /* Ask user to input unencrypted password */
  if( ! ( unencrypted_password_s = getpass( PASSWORD_PROMPT ) ) ) {
    fprintf( stderr, "getpass cannot open /dev/tty\n" );
    exit( -1 );
  }

  /* Use crypt() to encrypt user's input password.  Clear the *
   * unencrypted password as soon as we're done, so it is not * 
   * visible to memory snoopers.                              */
  encrypted_password_s = crypt( unencrypted_password_s,
				p_shadow_line->sp_pwdp );
  memset( unencrypted_password_s, 0, strlen( unencrypted_password_s ) );

  /* Return 1 (authenticated) iff the encrypted version of the user's *
   * input password matches the encrypted password stored in the      *
   * shadow password file.                                            */
  return( !strcmp( encrypted_password_s, p_shadow_line->sp_pwdp ) );

} /* authenticate_via_shadow_passwd() */

#endif /* if/else USE_PAM */

/************************************************************************
 *
 * All code used for both PAM and shadow passwd goes in this section.
 *
 ************************************************************************/

int main( int argc, char *argv[] ) {

  security_id_t new_sid;          /* our target security ID ("sid") */
  security_id_t osid;		     /* our original securiy ID ("osid") */
 
  security_id_t tty_sid;	     /* The current sid of tty file */
  security_id_t new_tty_sid;	     /* The new tty file sid */	

  security_context_t context_s;      /* our security context as a string */
  security_context_t ocontext_s;     /* our original context as a string */
  int context_length;

  context_t context;                 /* manipulatable form of context_s */
  context_t ocontext;		     /* manipulatable form of ocontext_s */

  struct passwd *p_passwd_line;      /* struct derived from passwd file line */

  int clflag;                        /* holds codes for command line flags */
  int flag_index;                    /* flag index in argv[] */
  struct option long_options[] = {   /* long option flags for getopt() */
    { "role", 1, 0, 'r' },
    { "type", 1, 0, 't' },
    { NULL, 0, 0, 0 }
  };
  char *role_s = NULL;               /* role spec'd by user in argv[] */
  char *type_s = NULL;               /* type spec'd by user in argv[] */
  char *ttyn   = NULL;		     /* tty path */
  extern char *optarg;               /* used by getopt() for arg strings */
  extern int opterr;                 /* controls getopt() error messages */
  int done_flag = 0;                 /* set when we're done processing args */
  int last_newrole_arg_index;        /* index of last newrole arg in argv[] */
  pid_t childPid;			     

  struct stat statbuf;		     /* when stat'ing the tty */


  /*
   *
   * Step 1:  Handle command-line arguments.
   *
   */


  /* Verify that we are running on a flask-enabled kernel. */
  if( !is_flask_enabled() ) {
    fprintf( stderr, 
	     "Sorry, newrole may be used only on a flask-enabled kernel.\n" );
    exit(-1);
  }

  /* Process command-line arguments.  We'll process args until we    *
   * reach an argument we don't recognize.  At that point, we'll     *
   * assume the unrecognized argument is meant for the shell we're   *
   * planning on exec'ing, so we'll move on to our next task.  Since *
   * we are expecting to see options that we will not comprehend, we *
   * will turn off the default error reporting behavior of           *
   * getopt_long() by clearing `opterr' to 0.                        */
  opterr = 0;  
  while( !done_flag ) {
    clflag=getopt_long(argc,argv,"r:t:",long_options,&flag_index);
    
    switch( clflag ) {
    case 'r':
      /* If role_s is already set, the user spec'd multiple roles - bad. */
      if( role_s ) {
	fprintf( stderr, "Error: multiple roles specified\n" );
	exit( -1 );
      }
      role_s = optarg;  /* save the role string spec'd by user */
      break;

    case 't':
      /* If type_s is already set, the user spec'd multiple types - bad. */
      if( type_s ) {
	fprintf( stderr, "Error: multiple types specified\n" );
	exit( -1 );
      }
      type_s = optarg;  /* save the type string spec'd by user */
      break;
      
    default:
      /* We don't recognize this argument.  It must be for the shell  *
       * we plan on creating, not for us.  Stop processing arguments. */
      done_flag = 1;
    } /* switch( clflag ) */
  } /* while command-line flags remain for newrole */

  
  /* Verify that the combination of command-line arguments we were *
   * given is a viable one.                                        */
  if( !(role_s || type_s) ) {
    fprintf(stderr, "%s\n",USAGE_STRING);
    exit(-1);
  }

  /* Determine the index of the last argument meant for newrole, and *
   * store it in `last_newrole_arg_index'.  Any arguments after this *
   * index in argv[] are meant for the new shell we're about to kick *
   * off.  Be careful if you decide to replace this code with        *
   * calculations based on getopt()'s `optind'.                      */
  if( role_s && type_s )
    last_newrole_arg_index = 4;
  else
    last_newrole_arg_index = 2;

  /* Fill in a default type if one hasn't been specified */
  if( role_s && !type_s ) {
    if( get_default_type(role_s,strlen(role_s),&type_s) )
      {
	fprintf(stderr,"Couldn't get default type.\n");
	exit(-1);
      }
#ifdef CANTSPELLGDB
  printf( "Your type will be %s.\n", type_s );
#endif  
  }

  /*
   *
   * Step 2:  Authenticate the user.
   *
   */


  /* Make `p_passwd_line' point to a structure containing the data   *
   * from our user's line in the passwd file.  We need to get the    *
   * user's name from this data for authentication.  We also need to *
   * get the user's shell from this data so that we know which shell *
   * to execute when we are done.                                    */

  /* Put our osid into `osid'. */
  if( 0>=(osid=getosecsid()) ) {
    fprintf(stderr,"failed to get osid.\n");
    exit(-1);
  }
  if( sid_to_dynamic_context(osid,&ocontext_s,&context_length) )
    {
      fprintf(stderr,"could not translate osid to context.\n");
      exit(-1);
    }
  ocontext=context_new(ocontext_s);
  if( !(p_passwd_line=getpwnam(context_user_get(ocontext))) ) {
    fprintf(stderr,"cannot find your entry in the passwd file.\n");
    exit(-1);
  }
  context_free( ocontext );

#ifdef CANTSPELLGDB
  printf("Authenticating %s.\n",p_passwd_line->pw_name);
#endif  

  /* Authenticate the user running this program. */
#ifdef USE_PAM
  if( !authenticate_via_pam(p_passwd_line) ) {
#else /* !USE_PAM */
  if( !authenticate_via_shadow_passwd(d p_passwd_line) ) {
#endif /* if/else USE_PAM */
    fprintf(stderr,"newrole: incorrect password\n");
    return(-1);
  }

  /* If we reach here, then we have authenticated the user. */
#ifdef CANTSPELLGDB
  printf( "You are authenticated!\n" );
#endif  

  /*
   *
   * Step 3:  Construct a new SID based on our current SID and the
   *          arguments specified on the command line.
   *
   */

  /* The first step to constructing the new SID for the new shell      *
   * we're planning on execing is to get our current SID.  We'll       *
   * accomplish this by getting our SID in `sid', converting this SID  *
   * into a context string in `context_s', and then finally converting *
   * this context string into an easily-modifiable context structure   *
   * in `context'.                                                     */

  /* Put our current sid into `new_sid'. */
  if( 0>=(new_sid=getsecsid()) ) {
    perror( "failed to get sid" );
    exit(-1);
  }

#ifdef CANTSPELLGDB
  printf( "Your former sid is %d\n", sid );
#endif

  if( sid_to_dynamic_context(new_sid,&context_s,&context_length) )
    {
      fprintf(stderr,"could not translate sid to context.\n");
      exit(-1);
    }

#ifdef CANTSPELLGDB
  printf( "Your former context is %s\n", context_s );
#endif

  /* Convert `context_s' into a more easily manipulatable form in `context' */
  context=context_new(context_s);
  free(context_s);                    /* We're done with this buffer now. */
  context_s=NULL;                     /* We'll be using `context_s' again. */

  /* Get effective user name and shell information */
 if( !(p_passwd_line=getpwuid(getuid()))) {
   fprintf(stderr,"cannot find your entry in the passwd file.\n");
   exit(-1);
  }

#ifdef CANTSPELLGDB
  printf("Your former role is %s\n",context_role_get(context));
  printf("Your former type is %s\n",context_type_get(context));
#endif

  
  /* The second step in constructing a new SID for the new shell we  *
   * plan to exec is to take our current context in `context' as a   *
   * starting point, and modify it according to the options the user *
   * specified on the command line.                                  */

  /* If the user specified a new role on the command line (if `role_s'   *
   * is set), then replace the old role in `context' with this new role. */
  if( role_s ) {
    if( context_role_set(context,role_s)) {
      fprintf(stderr,"failed to set new role %s\n",role_s);
      exit(-1);
    }
#ifdef CANTSPELLGDB
    printf("Your new role is %s\n",context_role_get(context));
#endif
  } /* if user specified new role */


  /* If the user specified a new type on the command line (if `type_s'   *
   * is set), then replace the old type in `context' with this new type. */
  if( type_s ) {
    if( context_type_set(context,type_s)) {
      fprintf(stderr,"failed to set new type %s\n",type_s);
      exit(-1);
    }
#ifdef CANTSPELLGDB
    printf("Your new type is %s\n",context_type_get(context));
#endif
  } /* if user specified new type */


  /* The third step in creating the new SID is to convert our modified *
   * `context' back through the various intermediate representations   *
   * and (finally) into a SID in `sid'.                                */

  /* Make `context_s' point to a string version of `context'.  Note *
   * that we've cleared `context_s' to NULL after we free'd its     *
   * previous buffer.                                               */
  if( !(context_s=context_str(context))) {
    fprintf(stderr,"failed to convert new context to string\n" );
    exit(-1);
  }

#ifdef CANTSPELLGDB
  printf("Your new context is %s\n",context_s);
#endif

  /* Convert our modified context in `context_s' into a new sid in `new_sid' * 
   * Once we are done, `new_sid' will contain the new sid for the new shell  */
  context_length=strlen(context_s)+1;
  if( 0>security_context_to_sid(context_s,context_length,&new_sid) ) {
    fprintf(stderr, "%s is not a valid context\n",context_s);
    exit(-1);
  }
  context_free(context);

#ifdef CANTSPELLGDB
  printf("Your new sid is %d\n",new_sid);
#endif

  /*
   *
   * Step 4:  Execute a new shell with the new SID in `new_sid'. 
   *
   */

#ifdef CANTSPELLGDB
  printf("Your new shell will be %s\n",p_passwd_line->pw_shell);
#endif

  /* Fetch TTY information */
  ttyn=ttyname(0);
  if( ttyn==NULL || *ttyn=='\0' || stat_secure(ttyn,&statbuf,&tty_sid) ) {
      fprintf(stderr, "Could not retrieve tty information.\n");
      exit (-1);
    }

  /* Fork, allowing parent to clean up after shell has executed */
  childPid=fork();
  if( childPid<0 ) {
    int errsv=errno;

    fprintf(stderr,"newrole: failure forking: %s",strerror(errsv));
    exit(-1);
  } else if (childPid) {

    /* PARENT */
    wait(NULL);

    /* Cleanaup TTY Context */
    chsid(ttyn,tty_sid);

    /* Done! */
    exit(0);
  }

  /* CHILD */

  /* Determine new TTY file context */
  if( security_change_sid(new_sid,tty_sid,SECCLASS_CHR_FILE,&new_tty_sid)!=0) {
    fprintf(stderr, "newrole: error: security_change_sid\n");
    exit(-1);
  } 

  /* Relabel it */
  if( chsid(ttyn,new_tty_sid)!=0 ) {
    fprintf(stderr,"newrole: error: chsid");
    exit(-1);
  }

  /* Relabel TTY descriptors by closing and reopening them securely */

  /* Close descriptors 0..2 */
  if( close(0) || close(1) || close(2) )
    {
      fprintf(stderr,"Could not close descriptors.\n");
      exit(-1);
    }

  /* Open them under a new context */
  open_secure(ttyn,O_RDONLY,0400,new_sid,new_tty_sid);  /* Owner Read */
  open_secure(ttyn,O_WRONLY,0220,new_sid,new_tty_sid);  /* Owner and Group Write */
  open_secure(ttyn,O_WRONLY,0220,new_sid,new_tty_sid);  /* Owner and Group Write */
 
  /*  Execute a new shell with the new sid in `sid'.  Make it's *argv *
   *  array start with the name of the shell, followed by any extra  *
   *  arguments that were passed to us.  "Extra" means any argument  *
   *  following *argv[ last_newrole_arg_index ].                      */
  argv[last_newrole_arg_index]=p_passwd_line->pw_shell;
  execv_secure(p_passwd_line->pw_shell,new_sid,&(argv[last_newrole_arg_index]));

  /* If we reach here, then we failed to exec the new shell. */
  perror("failed to exec shell\n");
  return(-1);
} /* main() */

