/*
 * runas [ context | [ -s sid ] | 
 *         ( [ -r role ] [-t type] [ -u user ] [ -l levelrange ] )
 *         command [arg1 [arg2 ...] ]
 *
 * attempt to run the specified command with the specified context.
 * 
 * -s sid   : use the context corresponding to the specified sid
 * -r role  : use the current context with the specified role
 * -t type  : use the current context with the specified type
 * -u user  : use the current context with the specified user
 * -l level : use the current context with the specified level range
 *
 * Contexts are interpreted as follows:
 *
 * Number of       MLS
 * components    system?
 *
 *     1            -         type
 *     2            -         role:type
 *     3            Y         role:type:range
 *     3            N         user:role:type
 *     4            Y         user:role:type:range
 *     4            N         error
 */

#include <unistd.h>
#include <stdio.h>
#include <getopt.h>
#include <context.h>
#include <proc_secure.h>
#include <errno.h>
extern int errno;

char *role    = 0;
char *range   = 0;
char *user    = 0;
char *type    = 0;
char *context = 0;

void
usage(char *str)
{
        fprintf(stderr,"runas: %s\n"
               "usage: runas (context | [-s sid ] | ([-t type]"
               " [-u user] [-r role] [-l levelrange]) cmd [arg1 [arg2 ...]]\n",
               str ? str : "");
        exit(1);
}

int 
main(int argc,char **argv,char **envp )
{
        context_t      con;
        security_id_t  sid = 0;  /* 0 is not a vaild sid */
        while (1) {
                int c;
                int this_option_optind = optind ? optind : 1;
                int option_index = 0;
                static struct option long_options[] = {
                        { "sid",  1, 0, 's' },
                        { "role", 1, 0, 'r' },
                        { "type", 1, 0, 't' },
                        { "user", 1, 0, 'u' },
                        { "range", 1, 0, 'l' },
                        { "help", 0, 0, '?' },
                        { 0, 0, 0, 0 }
                };
                c = getopt_long(argc, argv, "s:r:t:u:l:?", long_options, &option_index);
                if ( c == -1 ) {
                        break;
                }
                switch ( c ) {
                case 's':
                        if ( !( sid == 0 ) ) {
                                fprintf(stderr,"multiple sids\n");
                                exit(1);
                        }
			sid = (int)strtoul( optarg, (char **)NULL, 10);

			/* Check for an invalid sid. */
			if( ( errno == ERANGE ) || ( sid == 0 ) ) {  
			        fprintf( stderr, "invalid sid\n" );
			        exit( 1 );
			}
                        break;
                case 'r':
                        if ( role ) {
                                fprintf(stderr,"multiple roles\n");
                                exit(1);
                        }
                        role = optarg;
                        break;
                case 't':
                        if ( type ) {
                                fprintf(stderr,"multiple types\n");
                                exit(1);
                        }
                        type = optarg;
                        break;
                case 'u':
                        if ( user ) {
                                fprintf(stderr,"multiple users\n");
                                exit(1);
                        }
                        user = optarg;
                        break;
                case 'l':
                        if ( range ) {
                                fprintf(stderr,"multiple levelranges\n");
                                exit(1);
                        }
                        range = optarg;
                        break;
                default:
                        fprintf(stderr,"unrecognised option %c\n",c);
                case '?':
                        usage(0);
                        break;
                }
        }
        if ( ( !(user || role || type || range) ) &&
	     ( sid == 0 ) ) {
                if ( optind >= argc ) {
                        usage("must specify -t, -u, -l, -r, -s or context");
                }
                context = argv[optind++];
        }

        if ( context ) {
                con = context_new(context);
        } else if( sid == 0 ) { /* sid != 0 means sid already set by --sid */
                security_id_t mysid;
                char buff[1024];
                int len = sizeof(buff);
                mysid = getsecsid();
                if ( security_sid_to_context(mysid, 
                                             buff,
                                             &len ) ) {
                        perror("security_sid_to_context");
                        exit(1);
                }
                con = context_new(buff);
                if ( user ) {
                        context_user_set(con,user);
                }
                if ( type ) {
                        context_type_set(con,type);
                }
                if ( range ) {
                        context_range_set(con,range);
                }
                if ( role ) {
                        context_role_set(con,role);
                }
        }
        if ( optind >= argc ) {
                usage("no command found");
        }

#ifdef DEBUG_BROKEN_NEED_TO_HANDLE_SID_CASE
        fprintf(stderr,"Context to use is \"%s\"\n",context_str(con));
        fprintf(stderr,"Command is \"");
        { 
                int tmpoptind = optind;
                while ( tmpoptind < argc ) {
                        fprintf(stderr,"%s",argv[tmpoptind++]);
                        if ( tmpoptind < argc ) {
                                printf(" ");
                        }
                }
                fprintf(stderr,"\"\n");
        }
#endif

	if( sid == 0 ) { /* sid != 0 means sid already set by --sid */
	        if ( security_context_to_sid(context_str(con),
					  strlen(context_str(con))+1,&sid) ) {
	                perror("security_context_to_sid");
	                exit(1);
	        }
	}

#ifdef DEBUG
        fprintf(stderr,"sid = %d\n",sid);
#endif
#ifdef NOT_NOW
        if ( execve_secure(argv[optind],argv+optind,envp,sid) ) {
#endif
        if ( execvp_secure(argv[optind],sid,argv+optind) ) {
                perror("execvp_secure");
                exit(1);
        }
        return 1; /* can't reach this statement.... */
}
