/**************************************************************************** 
** File: ip.c
**
** Author: Mike Borella
**
** Comments: Dump IP header information
**
*****************************************************************************/

#include "global.h"
#include "protocols.h"

/* 
 * Contains the descriptions of IP options
 */

strmap_t ip_option_map[] = 
{
  { IP_OPTION_EOL,                "end of options" },
  { IP_OPTION_NOP,                "no op" },
  { IP_OPTION_RECORDROUTE,        "record route" },
  { IP_OPTION_TIMESTAMP,          "time stamp" },
  { IP_OPTION_TRACEROUTE,         "traceroute" },
  { IP_OPTION_SECURITY,           "security" },
  { IP_OPTION_LSR,                "loose source route" },
  { IP_OPTION_EXTSECURITY,        "extra security" },
  { IP_OPTION_COMSECURITY,        "commercial security" },
  { IP_OPTION_STREAMID,           "stream ID" },
  { IP_OPTION_SSR,                "strict source route" },
  { IP_OPTION_ROUTERALERT,        "router alert" }
};

extern struct arg_t *my_args;
extern void (*ip_proto_func[])(packet_t *);
extern strmap_t ipproto_map[];

void dump_ip_options(packet_t *, u_int8_t);

/*----------------------------------------------------------------------------
**
** in_cksum.c
**
** Do an IP checksum
**
**----------------------------------------------------------------------------
*/

u_int16_t in_cksum(u_int16_t *addr, int len) 
{ 
  int nleft = len; 
  u_int16_t *w = addr; 
  u_int32_t sum = 0; 
  u_int16_t answer = 0; 
     
  /* 
   * Our algorithm is simple, using a 32 bit accumulator (sum), we add 
   * sequential 16 bit words to it, and at the end, fold back all the 
   * carry bits from the top 16 bits into the lower 16 bits. 
   */ 
  while (nleft > 1)  
    { 
      sum += *w++; 
      nleft -= 2; 
    } 
     
  /* 
   * mop up an odd byte, if necessary 
   */ 
   
  if (nleft == 1) 
    { 
      *(u_int8_t *)(&answer) = *(u_int8_t *)w ; 
      sum += answer; 
    } 
     
    /* 
     * add back carry outs from top 16 bits to low 16 bits 
     */ 
   
  sum = (sum >> 16) + (sum & 0xffff); /* add hi 16 to low 16 */ 
  sum += (sum >> 16);         /* add carry */ 
  answer = ~sum;              /* truncate to 16 bits */ 
  return(answer); 
} 

/*----------------------------------------------------------------------------
**
** dump_ip()
**
** Parse IP header and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_ip(packet_t *pkt)
{
  ip_header ip;
  u_int16_t frag_off;
  u_int8_t ver, hlen;
  u_int8_t u_bit, df_bit, mf_bit;
  char holder[64];

#ifdef DEBUG
  printf("\nEntering IP\n");
#endif

  /*
   * Get the IP header
   */

  if (get_packet_bytes((u_int8_t *) &ip, pkt, 20) == 0)
    return;

  /*
   * Conversions
   */

  ver = ip.version;
  hlen = ip.header_length;
  ip.offset = ntohs(ip.offset);
  ip.length = ntohs(ip.length);
  ip.id = ntohs(ip.id);
  frag_off = ip.offset & 0x1fff;
  u_bit = (ip.offset & 0x8000) >> 15;
  df_bit = (ip.offset & 0x4000) >> 14;
  mf_bit = (ip.offset & 0x2000) >> 13;
  ip.checksum = ntohs(ip.checksum);

  /*
   * Dump header
   */

  if (my_args->m && !my_args->n)
    {
      display_minimal((u_int8_t *) &ip.src, 4, DISP_DOTTEDDEC); 
      display_minimal_string("->");
      display_minimal((u_int8_t *) &ip.dst, 4, DISP_DOTTEDDEC); 
      display_minimal_string(" (");
      display_minimal((u_int8_t *) &ip.length, 2, DISP_2BYTEDEC);
      if (df_bit)
	display_minimal_string(",DF");
      display_minimal_string(") ");
    }
  else
    if (!my_args->n)
      {
	/* announcement */
	display_header_banner("IP Header");
	
	/* print fields */
	display("Version", (u_int8_t *) &ver, 1, DISP_1BYTEDEC);
	display("Header length", (u_int8_t *) &hlen, 1, DISP_1BYTEDEC);
	display("TOS", (u_int8_t *) &ip.tos, 1, DISP_HEX);
	display("Total length", (u_int8_t *) &ip.length, 2, DISP_2BYTEDEC);
	display("Identification", (u_int8_t *) &ip.id, 2, DISP_2BYTEDEC);
	display("Fragmentation offset", (u_int8_t *) &frag_off, 2, 
		DISP_2BYTEDEC);
	display("Unused bit", (u_int8_t *) &u_bit, 1, DISP_BINNLZ);
	display("Don't fragment bit", (u_int8_t *) &df_bit, 1, DISP_BINNLZ);
	display("More fragments bit", (u_int8_t *) &mf_bit, 1, DISP_BINNLZ);
	display("Time to live", (u_int8_t *) &ip.ttl, 1, DISP_1BYTEDEC);
	sprintf(holder, "%d (%s)", ip.protocol, 
		map2str(ipproto_map, ip.protocol));
	display("Protocol", holder, strlen(holder), DISP_STRING);
	display("Header checksum", (u_int8_t *) &ip.checksum, 2, 
		DISP_2BYTEDEC);
      
	/* DO the checksum here?
	   memcpy((void *) &ip2, (void *) ip, sizeof(IPHdr));
	   ip2.ip_csum = 0;
	   my_csum = ntohs(in_cksum((u_int16_t *) &ip2, sizeof(IPHdr)));
	   if (my_csum != csum) printf("(error: should be %d)", my_csum);
	   */
	
	/* IP's are not in network byte order?? */
	display("Source address",(u_int8_t *) &ip.src, 4, DISP_DOTTEDDEC); 
	display("Destination address",(u_int8_t *) &ip.dst, 4, 
		DISP_DOTTEDDEC); 
      }

  /*
   * Check for IP options
   */

  if (hlen > 5)
    dump_ip_options(pkt, hlen * 4 - 20);

  /*
   * If this is fragment zero, hand it to the next higher
   * level protocol.
   */
  
  if ((ip.offset & 0x1fff) == 0 && ip_proto_func[ip.protocol]) 
      ip_proto_func[ip.protocol](pkt);

#ifdef DEBUG
  printf("\nLeaving IP\n");
#endif

}

/*----------------------------------------------------------------------------
**
** dump_ip_options()
**
** Dump any IP options
**
**----------------------------------------------------------------------------
*/

void dump_ip_options(packet_t *pkt, u_int8_t length)
{
  u_int8_t bytes_read = 0;
  ip_option_t opt;
  char holder[64];

  while(bytes_read < length)
    {
      /*
       * Get the first byte to make sure that we don't have a 1-byte option
       */

      if (get_packet_bytes((u_int8_t *) &opt.code, pkt, 1) == 0)
	return;
      bytes_read += 1;

      /*
       * Display it
       */
      
	sprintf(holder, "%d (%s)", opt.code, 
		map2str(ip_option_map, opt.code));
      if (!my_args->m)
	display_string("IP option code", holder);
      else
	{
	 display_minimal_string(holder);
	 display_minimal_string(" ");
	}
	
      /*
       * If it is 0, then its just a 1-byte EOL
       */

      if (opt.code == 0)
	continue;

      /*
       * It isn't, so lets read the rest...
       */

      if (get_packet_bytes((u_int8_t *) &opt.length, pkt, 1) == 0)
	return;
      bytes_read += 1;
      if (!my_args->m)
	display("  Length", (u_int8_t *) &opt.length, 1, DISP_1BYTEDEC);

      /*
       * Get the rest of the option, adjusting the length by 2 to count 
       * option type and length fields...
       */
      
      if (get_packet_bytes((u_int8_t *) &holder, pkt, opt.length-2) == 0)
	return;
      bytes_read += (length-2);
      
    }
}
