/**************************************************************************** 
** File: udp.c
**
** Author: Mike Borella
**
** Comments: Dump UDP header information
**
** $Log: udp.c,v $
** Revision 1.4  2000/05/09 00:21:54  mborella
** Got rid of some warnings on FreeBSD compile.
**
** Revision 1.3  2000/05/08 21:18:23  mborella
** Made "minimal mode" into a one-line tcpdump-like mode.
**
** Revision 1.2  2000/05/04 22:47:51  mborella
** Fixed SIP.  Added application function mapping based on port number.
**
** Revision 1.1  2000/05/04 19:11:42  mborella
** Moved a bunch of source files to the src directory.
**
** Revision 1.5  2000/04/29 00:21:07  mborella
** Port to string conversions work, but we're nowhere near complete yet.
**
*****************************************************************************/

#include "global.h"
#include "protocols.h"

extern struct arg_t *my_args;
extern strmap_t port_map[];

/*----------------------------------------------------------------------------
**
** dump_udp()
**
** Parse UDP header and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_udp(packet_t *pkt)
{
  udp_header_t udp;
  char holder[64];
  app_func_t app;

  /*
   * Get the IP header
   */

  if (get_packet_bytes((u_int8_t *) &udp, pkt, 8) == 0)
    return;

  /* 
   * Conversions
   */

  udp.src = ntohs(udp.src);
  udp.dst = ntohs(udp.dst);
  udp.length = ntohs(udp.length);
  udp.checksum = htons(udp.checksum);

  /*
   * Minimal mode
   */

  if (my_args->m && !my_args->t)
    {
      display_minimal((u_int8_t *) &udp.src, 2, DISP_2BYTEDEC);
      display_minimal("->", 2, DISP_STRING);
      display_minimal((u_int8_t *) &udp.dst, 2, DISP_2BYTEDEC);
      display_minimal(" ", 1, DISP_STRING);
    }
  else 
    if (!my_args->t)
      {  
	/* announcement */
	display_header_banner("UDP Header");
	
	/* dump fields */
	sprintf(holder, "%d (%s)", udp.src, map2str(port_map, udp.src));
	display("Source port", (u_int8_t *) holder, strlen(holder), 
		DISP_STRING);
	sprintf(holder, "%d (%s)", udp.dst, map2str(port_map, udp.dst));
	display("Destination port", (u_int8_t *) holder, strlen(holder), 
		DISP_STRING);
	display("Length", (char *) &udp.length, 2, DISP_2BYTEDEC);
	display("Checksum", (char *) &udp.checksum, 2, DISP_2BYTEDEC);
      }

  /*
   * Let's see if we can decode the application from the source port...
   */

  app = port2func(udp.src);
  if (app)
    {
      app(pkt);
      return;
    }

  /*
   * Let's see if we can decode the application from the destination port...
   */

  app = port2func(udp.dst);
  if (app)
    {
      app(pkt);
      return;
    }

  /*
   * Ok, let's just print the payload if there is one and it is turned on
   */

  if (my_args->p && get_packet_bytesleft(pkt)) 
    dump_payload(pkt);

  return;
}
