/**************************************************************************** 
 **
 ** File: isakmp.c
 **
 ** Extensions and additions by: Stuart Stock (stuart@ins.com)
 ** Original Author: Mike Borella
 **
 ** Comments: Dump ISAKMP headers under IPSec DOI 
 **
 ** See RFC 2408 "Internet Security Association and Key Management Protocol"
 ** and RFC 2407 "The Internet IP Security Domain Interpretation for ISAKMP"
 **
 ** and when you can't find the value anywhere else, look in:
 ** draft-ietf-ipsec-ike-01 "The Internet Key Exchange (IKE)"
 **
 ** $Log: isakmp.c,v $
 ** Revision 1.2  2000/05/08 23:36:04  mborella
 ** Got rid of addrtoname.c module.  Functionality is now taken care of in our
 ** own way.
 **
 *****************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include "global.h"
#include "protocols.h"

/*----------------------------------------------------------------------------
**
** dump_isakmp()
**
** Parse ISAKMP packet and dump fields.  
**
**----------------------------------------------------------------------------
*/

void dump_isakmp(u_char *bp, int length)
{
  u_char *p;
  ISAKMPHdr *isakmp;
  void dump_next_payload(int);
  void determine_next_payload(int, u_char *);

  p = bp;

  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP Headers\n");
  printf("-----------------------------------------------------------------\n");

  isakmp = (ISAKMPHdr *) bp;

  printf("Initiator Cookie: %x", EXTRACT_32BITS(isakmp->i_cookie));
  printf("%x\n", EXTRACT_32BITS(isakmp->i_cookie + 4));
  printf("Responder Cookie: %x", EXTRACT_32BITS(isakmp->r_cookie)); 
  if (EXTRACT_32BITS(isakmp->r_cookie) != 0) 
    printf("%x", EXTRACT_32BITS(isakmp->r_cookie + 4)); 
  printf("\n");
  
  printf("Next payload:           %d ", isakmp->next_payload);
  dump_next_payload(isakmp->next_payload);

  printf("Major version:          %d\n", isakmp->maj_version);
  printf("Minor version:          %d\n", isakmp->min_version);

  printf("Exchange type:          %d ", isakmp->exchange_type);
  switch(isakmp->exchange_type)
    {
    case 0:
      printf("(none)");
      break;
    case 1:
      printf("(base)");
      break;
    case 2:
      printf("(identity protection)");
      break;
    case 3:
      printf("(authentication only)");
      break;
    case 4:
      printf("(aggressive)");
      break;
    case 5:
      printf("(informational)");
      break;
    case 32:
      printf("(quick)");
      break;
    case 33:
      printf("(new group)");
      break;
    case 34:
      printf("(Acknowledged Informational)");
      break;
    default:
      if (isakmp->exchange_type >= 6 && isakmp->exchange_type <= 31)
	printf("(future)");
      if (isakmp->exchange_type >= 35 && isakmp->exchange_type <= 239)
	printf("(DOI specific)");
      if (isakmp->exchange_type >= 240 && isakmp->exchange_type <= 255)
	printf("(private)");
    }
  printf("\n");

  printf("Flags:                  %d", isakmp->flags);
  if (isakmp->flags)
    {
      printf(" (");
      if (isakmp->flags & 0x01)
	printf("E");
      if (isakmp->flags & 0x02)
	printf("C");
      if (isakmp->flags & 0x04)
	printf("A");
      printf(")");
    }
  printf("\n");

  printf("Message ID:             %u\n", ntohl(isakmp->msg_id));
  printf("Length:                 %d\n", ntohl(isakmp->length));

  /*
   * Short circuit processing if this is the only header or encryption
   * bit is set
   */

  if (!isakmp->next_payload || isakmp->flags & 0x01)
    return;

  /*
   * Otherwise, advance pointer and keep processing
   */

  bp = bp + sizeof(ISAKMPHdr);
  determine_next_payload(isakmp->next_payload, bp);
}




/*----------------------------------------------------------------------------
**
** dump_next_payload()
**
** Decode and print the next payload type
**
**----------------------------------------------------------------------------
*/

void dump_next_payload(int np)
{

  switch(np)
    {
    case 0:
      printf("(none)");
      break;
    case 1:
      printf("(security association)");
      break;
    case 2:
      printf("(proposal)");
      break;
    case 3:
      printf("(transform)");
      break;
    case 4:
      printf("(key exchange)");
      break;
    case 5:
      printf("(identification)");
      break;
    case 6:
      printf("(certificate)");
      break;
    case 7:
      printf("(certificate request)");
      break;
    case 8: 
      printf("(hash)");
      break;
    case 9:
      printf("(signature)");
      break;
    case 10: 
      printf("(nonce)");
      break;
    case 11:
      printf("(notification)");
      break;
    case 12: 
      printf("(delete)");
      break;
    case 13:
      printf("(vendor ID)");
      break;
    default:
      if (np >= 14 && np <= 127)
	printf("(reserved)");
      else
	printf("(private)");
    }
  printf("\n");

}


/*----------------------------------------------------------------------------
**
** determine_next_payload()
**
** Call routing to parse the next payload type
**
**----------------------------------------------------------------------------
*/

void determine_next_payload(int np, u_char *bp)
{
  void dump_sa_payload(u_char *bp);
  void dump_proposal_payload(u_char *bp);
  void dump_keyexchange_payload(u_char *bp);
  void dump_hash_payload(u_char *bp);
  void dump_notification_payload(u_char *bp);
  void dump_identification_payload(u_char *bp);
  void dump_nonce_payload(u_char *bp);
  void dump_signature_payload(u_char *bp);
  void dump_vendorid_payload(u_char *bp);
  void dump_transform_payload(u_char *bp);
  void dump_delete_payload(u_char *bp); 
  void dump_certificate_payload(u_char *bp, int type);

  switch(np)
    {
    case 0: /* no payload, do nothing */
      break;
    case 1:
      dump_sa_payload(bp);
      break;
    case 2:
      dump_proposal_payload(bp);
      break;
    case 3:
      dump_transform_payload(bp);
      break;
    case 4:
      dump_keyexchange_payload(bp);
      break;
    case 5:
      dump_identification_payload(bp);
      break;
    case 6:
      dump_certificate_payload(bp, 1);
      break;
    case 7:
      dump_certificate_payload(bp, 2);
      break;
    case 8:
      dump_hash_payload(bp); 
      break;
    case 9:
      dump_signature_payload(bp);
      break;
    case 10: 
      dump_nonce_payload(bp);
      break;
    case 11:
      dump_notification_payload(bp);
      break;
    case 12:
      dump_delete_payload(bp); 
      break;
    case 13:
      dump_vendorid_payload(bp);
      break;
    default:
      break;
    }
}

/*---------------------------------------------------------------------
**
** dump_delete_payload()
**
** incomplete routine to dump delete packets
**
**---------------------------------------------------------------------
*/

void dump_delete_payload(u_char *bp)
{

  ISAKMP_generic_hdr *gen;
  u_char *old_bp;
  int doi;

  gen = (ISAKMP_generic_hdr *) bp;
  old_bp = bp;

  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP Delete Header\n");
  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));

  bp = bp + sizeof(ISAKMP_generic_hdr);

  doi = EXTRACT_32BITS(bp);
  bp = bp + 4;
  printf("DOI:                    %d\n", doi);

  /* We want to dump the SPI's and crud here */
  printf("\nNo decode support, yet\n");

  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);
  
}



/*---------------------------------------------------------------------
**
** dump_certificate_payload()
**
** Dumps a certificate payload packet.
**
** Since ISAKMP packet types 6 (cert) and 7 (cert request) are the same
** format, the function takes a second parameter, type, to handle the 
** proper formating.
**
**---------------------------------------------------------------------
*/

void dump_certificate_payload(u_char *bp, int type)
{
  ISAKMP_generic_hdr *gen;
  u_char *old_bp;
  int cert_enc;

  gen = (ISAKMP_generic_hdr *) bp;
  old_bp = bp;

  printf("-----------------------------------------------------------------\n");

  if( type == 1) {  
	  printf("                        ISAKMP Certificate Header\n");
  } 
  else {
  	  printf("                        ISAKMP Certificate Request Header\n");
  }

  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));

  bp = bp + sizeof(ISAKMP_generic_hdr);

  cert_enc = *bp;
  bp ++;
  printf("Certificate Encoding:   %d ", cert_enc);

  switch( cert_enc )
    {
    case 0:
      printf("(NONE)");
      break;
    case 1:
      printf("(PKCS7 Wrapped X.509)");
      break;
    case 2:
      printf("(PGP Certificate)");
      break;
    case 3:
      printf("(DNS Signed Key)");
      break;
    case 4:
      printf("(X.509 - Signature)");
      break;
    case 5:
      printf("(X.509 - Key Exchange)");
      break;
    case 6:
      printf("(Kerberos Tickets)");
      break;
    case 7:
      printf("(Certificate Revocation List)");
      break;
    case 8:
      printf("(Authority Revocation List)");
      break;
    case 9:
      printf("(SPKI Certificate)");
      break;
    case 10:
      printf("(X.509 - Attribute)");
      break;
    default:
      printf("(reserved)");
      break;
    }
  printf("\n");

  printf("Certificate data:             ");
  
  /* XXX Is this right?  4 bytes of ISAKMP crud and 1 byte for cert type */
  display_minimal((u_int8_t *) bp, ntohs(gen->length) - 5, DISP_HEX);
  
  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);
  
}


/*----------------------------------------------------------------------------
**
** dump_sa_payload()
**
** Dump an SA payload.
**
**----------------------------------------------------------------------------
*/

void dump_sa_payload(u_char *bp)
{
  ISAKMP_generic_hdr *gen;
  u_char *old_bp;
  int doi;
  u_int32_t situation;

  gen = (ISAKMP_generic_hdr *) bp;
  old_bp = bp;

  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP SA Header\n");
  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));

  bp = bp + sizeof(ISAKMP_generic_hdr);

  doi = EXTRACT_32BITS(bp);
  bp = bp + 4;
  printf("DOI:                    %d\n", doi);
  printf("Situation:              ");
  if (doi == DOI_IPSEC)
    {
      situation = EXTRACT_32BITS(bp);
      bp = bp + 4;
      printf("%d ", situation);
      switch(situation)
	{
	case SIT_IDENTITY_ONLY:
	  printf("(identity only)");
	  break;
	case SIT_SECRECY:
	  printf("(secrecy)");
	  break;
	case SIT_INTEGRITY:
	  printf("(integrity)");
	  break;
	}
      printf("\n");
    }
  else
    {
      printf("(undefined DOI)\n");
    }

  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);
  
}


/*----------------------------------------------------------------------------
**
** dump_proposal_payload()
**
** Dump a proposal payload.
**
**----------------------------------------------------------------------------
*/

void dump_proposal_payload(u_char *bp)
{
  ISAKMP_generic_hdr *gen;

  u_int8_t protocol_id, spi_size;
  u_int16_t num_trans;
  u_int32_t prop;
  u_char *old_bp;

  old_bp = bp;
  gen = (ISAKMP_generic_hdr *) bp;
  
  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP Proposal Header\n");
  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));

  bp = bp + sizeof(ISAKMP_generic_hdr);

  prop = EXTRACT_32BITS(bp);
  bp = bp + 4;
  printf("Proposal Number:        %d\n", prop);

  protocol_id = *bp;
  bp++;
  printf("Protocol ID:            %d ", protocol_id);
  switch( protocol_id )
    {
    case 0:
      printf("(reserved)");
      break;
    case 1:
      printf("(protocol ISAKMP)");
      break;
    case 2:
      printf("(protocol IPSEC AH)");
      break;
    case 3:
      printf("(protocol IPSEC ESP)");
      break;
    case 4:
      printf("(protocol IPCOMP)");
      break;
    default:
      printf("(unknown)");
      break;
    }
  printf("\n");
  

  spi_size = *bp;
  bp++;
  printf("SPI size:               %d\n", spi_size);

  num_trans = EXTRACT_16BITS(bp);
  bp = bp + 2;
  printf("Number of Transforms:   %d\n", num_trans);

  printf("SPI:                    not shown\n");
  bp = bp + spi_size;

  /*
   * Move pointer to end of this header 
   */

  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);

}


/*----------------------------------------------------------------------------
**
** dump_hash_payload()
**
** Dump a hash payload.
**
**----------------------------------------------------------------------------
*/

void dump_hash_payload(u_char *bp)
{
  ISAKMP_generic_hdr *gen;
  u_char *old_bp;

  gen = (ISAKMP_generic_hdr *) bp;
  old_bp = bp;

  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP Hash Header\n");
  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));

  bp = bp + sizeof(ISAKMP_generic_hdr);

  printf("Hash data:              not shown\n");

  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);

}


/*----------------------------------------------------------------------------
**
** dump_notification_payload()
**
** Dump a notification payload.
**
**----------------------------------------------------------------------------
*/

void dump_notification_payload(u_char *bp)
{
  ISAKMP_generic_hdr *gen;
  u_int8_t protocol_id, spi_size;
  u_int16_t message_type;
  u_int32_t doi;
  u_char *old_bp;

  old_bp = bp;
  gen = (ISAKMP_generic_hdr *) bp;
  
  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP Notification Header\n");
  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));

  bp = bp + sizeof(ISAKMP_generic_hdr);

  doi = EXTRACT_32BITS(bp);
  bp = bp + 4;
  printf("DOI:                    %d\n", doi);

  protocol_id = *bp;
  bp++;
  printf("Protocol ID:            %d ", protocol_id);
  switch( protocol_id )
    {
    case 0:
      printf("(reserved)");
      break;
    case 1:
      printf("(protocol ISAKMP)");
      break;
    case 2:
      printf("(protocol IPSEC_AH)");
      break;
    case 3:
      printf("(protocol IPSEC_ESP)");
      break;
    case 4:
      printf("(protocol IPCOMP)");
      break;
    default:
      printf("(unknown)");
      break;
    }
  printf("\n");
  

  spi_size = *bp;
  bp++;
  printf("SPI size:               %d\n", spi_size);

  message_type = EXTRACT_16BITS(bp);
  bp = bp + 2;
  printf("Message type:           %d ", message_type);
  switch(message_type)
    {
    case 1:
    	printf("(invalid payload type)");
    	break;
    case 2:
    	printf("(DOI not supported)");
    	break;
    case 3:
    	printf("(situation not supported)");
    	break;
    case 4:
    	printf("(invalid cookie)");
    	break;
    case 5:
    	printf("(invalid major version)");
    	break;
    case 6:
    	printf("(invalid minor version)");
    	break;
    case 7:
    	printf("(invalid exchange type)");
    	break;
    case 8:
    	printf("(invalid flags)");
    	break;
    case 9:
    	printf("(invalid message ID)");
    	break;
    case 10:
    	printf("(invalid protocol ID)");
    	break;
    case 11:
    	printf("(invalid SPI)");
    	break;
    case 12:
    	printf("(invalid transform ID)");
    	break;
    case 13:
    	printf("(attributes not supported)");
    	break;
    case 14:
    	printf("(no proposal chosen)");
    	break;
    case 15:
    	printf("(bad proposal syntax)");
    	break;
    case 16:
    	printf("(payload malformed)");
    	break;
    case 17:
    	printf("(invalid key information)");
    	break;
    case 18:
    	printf("(invalid ID information)");
    	break;
    case 19:
    	printf("(invalid cert encoding)");
    	break;
    case 20:
    	printf("(invalid certificate)");
    	break;
    case 21:
    	printf("(cert type unsupported)");
    	break;
    case 22:
    	printf("(invalid cert authority)");
    	break;
    case 23:
    	printf("(invalid hash information)");
    	break;
    case 24:
    	printf("(authentication failed)");
    	break;
    case 25:
    	printf("(invalid signature)");
    	break;
    case 26:
    	printf("(address notification)");
    	break;
    case 27:
    	printf("(notify SA lifetime)");
    	break;
    case 28:
    	printf("(certificate unavailable)");
    	break;
    case 29:
    	printf("(unsupported exchange type)");
    	break;
    case 16384:
    	printf("(connected)");
    	break;
    case 24576:
      printf("(responder lifetime)");
      break;
    case 24577:
      printf("(replay status)");
      break;
    case 24578:
      printf("(initial contact)");
      break;
    default:
      printf("(unknown)");
      break;
    }
  printf("\n");

  printf("SPI:                    not shown\n");
  bp = bp + spi_size;

  /*
   * Here, the notification data length depends on the message type.
   * They really should have defined a length for this field.  We
   * have to get a bit ugly in order to do this right...
   */

  switch(message_type)
    {
    case 24576:
      break;
    case 24577:
      {
	u_int32_t data;
	
	data = EXTRACT_32BITS(bp);
	bp = bp + 4;
	printf("Notification data        %d ", data);
	switch(data)
	  {
	  case 0: 
	    printf("(replay detection disabled)");
	    break;
	  case 1:
	    printf("(replay detection enabled)");
	    break;
	  default:
	    printf("(unknown)");
	    break;
	  }
	printf("\n");
      }
      break;
    case 24578:
      printf("(initial contact)"); /* no data field */
      break;
    default:
      printf("(unknown)");
      break;
    }

  /*
   * Move pointer to end of this header 
   */

  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);

}


/*----------------------------------------------------------------------------
**
** dump_transform_payload()
**
** Dump a transform payload.
**
**----------------------------------------------------------------------------
*/

void dump_transform_payload(u_char *bp)
{
  ISAKMP_generic_hdr *gen;
  u_int8_t trans_id, spi_size;
  u_int32_t trans;
  u_char *old_bp;

  old_bp = bp;
  gen = (ISAKMP_generic_hdr *) bp;
  
  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP Transform Header\n");
  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));

  bp = bp + sizeof(ISAKMP_generic_hdr);

  trans = EXTRACT_32BITS(bp);
  bp = bp + 4;
  printf("Transform Number:       %d\n", trans);

  trans_id = *bp;
  bp++;
  printf("Transform ID:            %d ", trans_id);

  /* NOTE: This should be conditional so that we don't print both the AH 
   * and ESP transform IDs at the same time.  Need to add IPCOMP also.
   */     
  switch( trans_id )
    {
    case 0:
      printf("(reserved)");
      break;
    case 1:
      printf("(AH reserved) or (ESP DES IV64)");
      break;
    case 2:
      printf("(AH MD5) or (ESP DES)");
      break;
    case 3:
      printf("(AH SHA1) or (ESP 3DES)");
      break;
    case 4:
      printf("(AH DES)");
      break;
    case 5:
      printf("(protocol IPCOMP)");
      break;
    default:
      printf("(unknown)");
      break;
    }
  printf("\n");
  

  spi_size = *bp;
  bp++;
  printf("Reserved:               %d\n", spi_size);

  /*
   * Move pointer to end of this header 
   */

  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);

}

/*----------------------------------------------------------------------------
**
** dump_keyexchange_payload()
**
** Dump a key exchange payload.
**
**----------------------------------------------------------------------------
*/

void dump_keyexchange_payload(u_char *bp)
{
  ISAKMP_generic_hdr *gen;
  u_char *old_bp;

  gen = (ISAKMP_generic_hdr *) bp;
  old_bp = bp;

  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP Key Exchange Header\n");
  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));

  bp = bp + sizeof(ISAKMP_generic_hdr);

  printf("Key exchange data:      ");  
  display_minimal(bp, ntohs(gen->length) - 4, DISP_HEX);

  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);

}


/*----------------------------------------------------------------------------
**
** dump_identification_payload()
**
** Dump an identification payload.
**
**----------------------------------------------------------------------------
*/

void dump_identification_payload(u_char *bp)
{
  ISAKMP_generic_hdr *gen;
  u_char *old_bp;
  u_int8_t id_type;
  u_int32_t doi_data;

  gen = (ISAKMP_generic_hdr *) bp;
  old_bp = bp;

  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP Identification Header\n");
  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));

  bp = bp + sizeof(ISAKMP_generic_hdr);

  id_type = *bp;
  bp ++;
  printf("ID type:                %d ", id_type);
  switch(id_type)
    {
    case 0:
      printf("(reserved)");
      break;
    case 1:
      printf("(IPv4 address)");
      break;
    case 2: 
      printf("(FQDN)");
      break;
    case 3:
      printf("(user FQDN)");
      break;
    case 4:
      printf("(IPv4 address subnet)");
      break;
    case 5:
      printf("(IPv6 address)");
      break;
    case 6:
      printf("(IPv6 address subnet)");
      break;
    case 7:
      printf("(IPv4 address range)");
      break;
    case 8:
      printf("(IPv6 address range)");
      break;
    case 9: 
      printf("(DER coding of ASN.1 X.500 dist. name)");
      break;
    case 10:
      printf("(DER coding of ASN.1 X.500 gen. name)");
      break;
    case 11:
      printf("(key ID)");
      break;
    }
  printf("\n");

  /*
   * Read next 3 bytes 
   */

  doi_data = *bp;
  doi_data = doi_data << 8;
  doi_data += *(bp+1);
  doi_data = doi_data << 8;
  doi_data += *(bp+1);
  bp = bp + 3;
  printf("DOI ID data:           %d\n", doi_data);

  switch(id_type)
    {
    case 0:
      break;
    case 1:
      
      break;
    case 2: 
      printf("(FQDN)");
      break;
    case 3:
      printf("(user FQDN)");
      break;
    case 4:
      printf("(IPv4 address subnet)");
      break;
    case 5:
      printf("(IPv6 address)");
      break;
    case 6:
      printf("(IPv6 address subnet)");
      break;
    case 7:
      printf("(IPv4 address range)");
      break;
    case 8:
      printf("(IPv6 address range)");
      break;
    case 9: 
      printf("(DER coding of ASN.1 X.500 dist. name)");
      break;
    case 10:
      printf("(DER coding of ASN.1 X.500 gen. name)");
      break;
    case 11:
      printf("(key ID)");
      break;
    }

  

  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);

}


/*----------------------------------------------------------------------------
**
** dump_nonce_payload()
**
** Dump a nonce payload.
**
**----------------------------------------------------------------------------
*/

void dump_nonce_payload(u_char *bp)
{
  ISAKMP_generic_hdr *gen;
  u_char *old_bp;
  
  gen = (ISAKMP_generic_hdr *) bp;
  old_bp = bp;
  
  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP Nonce Header\n");
  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));
  
  bp = bp + sizeof(ISAKMP_generic_hdr);
  
  printf("Nonce data:             ");
  display_minimal(bp, ntohs(gen->length) - 4, DISP_HEX);
  
  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);
  
}


/*----------------------------------------------------------------------------
**
** dump_signature_payload()
**
** Dump a signature payload.
**
**----------------------------------------------------------------------------
*/

void dump_signature_payload(u_char *bp)
{
  ISAKMP_generic_hdr *gen;
  u_char *old_bp;
  
  gen = (ISAKMP_generic_hdr *) bp;
  old_bp = bp;
  
  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP Signature Header\n");
  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));
  
  bp = bp + sizeof(ISAKMP_generic_hdr);
  
  printf("Signature data:             ");
  display_minimal(bp, ntohs(gen->length) - 4, DISP_HEX);
  
  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);
  
}


/*----------------------------------------------------------------------------
**
** dump_vendorid_payload()
**
** Dump a vendor ID payload.
**
**----------------------------------------------------------------------------
*/

void dump_vendorid_payload(u_char *bp)
{
  ISAKMP_generic_hdr *gen;
  u_char *old_bp;
  
  gen = (ISAKMP_generic_hdr *) bp;
  old_bp = bp;
  
  printf("-----------------------------------------------------------------\n");
  printf("                        ISAKMP Vendor ID Header\n");
  printf("-----------------------------------------------------------------\n");
  printf("Next payload:           %d ", gen->next_payload);
  dump_next_payload(gen->next_payload);
  printf("Reserved:               %d\n", gen->reserved);
  printf("Payload length          %d\n", ntohs(gen->length));
  
  bp = bp + sizeof(ISAKMP_generic_hdr);
  
  printf("Vendor ID:              ");
  display_minimal(bp, ntohs(gen->length) - 4, DISP_HEX);
  
  bp = old_bp + ntohs(gen->length);
  determine_next_payload(gen->next_payload, bp);
  
}
