/**************************************************************************** 
** File: ospf.c
**
** Author: Mike Borella
**
** Comments: Dump OSPF information
**
*****************************************************************************/

#include "global.h"
#include "protocols.h"

/*
 * OSPF type map
 */

strmap_t ospf_type_map [] =
{
  { OSPF_TYPE_HELLO,        "hello" },
  { OSPF_TYPE_DATABASEDESC, "database description" },
  { OSPF_TYPE_LSREQUEST,    "link state request" },
  { OSPF_TYPE_LSUPDATE,     "link state update" },
  { OSPF_TYPE_LSACK,        "link state acknowledgement" },
  { 0, "" }
};

/*
 * OSPF auth type map
 */

strmap_t ospf_auth_map [] =
{
  { OSPF_AUTH_NONE,         "none" },
  { OSPF_AUTH_SIMPLE,       "simple" },
  { OSPF_AUTH_CRYPTO,       "cryptographic" },
  { 0, "" }
};

extern struct arg_t * my_args;

/*----------------------------------------------------------------------------
**
** dump_ospf()
**
** Parse OSPF common header and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_ospf(packet_t *pkt)
{
  ospf_common_header_t ospf;
  char holder[64];
  
  /*
   * Get the header
   */
  
  if (get_packet_bytes((u_int8_t *) &ospf, pkt, 
		       sizeof(ospf_common_header_t)) == 0)
    return;
  
  /*
   * Conversions 
   */
  
  ospf.length = ntohs(ospf.length);
  ospf.router_id = ntohl(ospf.router_id);
  ospf.area_id = ntohl(ospf.area_id);
  ospf.checksum = ntohs(ospf.checksum);
  ospf.auth_type = ntohs(ospf.auth_type);

  /*
   * Dump the fields 
   */
  
  if (my_args->m && !my_args->n)
    {
      /* In minimal mode lets just dump the type and the router ID */
      display_minimal_string("OSPF ");
      display_minimal_string(map2str(ospf_type_map, ospf.type));
      display_minimal_string(" ");      
      display_minimal((u_int8_t *) &ospf.router_id, 4, DISP_DOTTEDDEC);
      display_minimal_string(" ");      
    }
  else
    if (!my_args->n)
      { 
	/* announcement */
	display_header_banner("OSPF Header");

	display("Version", (u_int8_t *) &ospf.version, 1, DISP_1BYTEDEC);
	sprintf(holder, "%d (%s)", ospf.type, 
		map2str(ospf_type_map, ospf.type));
	display_string("Type", holder);
	display("Length", (u_int8_t *) &ospf.length, 2, DISP_2BYTEDEC);
	display("Router ID", (u_int8_t *) &ospf.router_id, 4, DISP_DOTTEDDEC);
	display("Area ID", (u_int8_t *) &ospf.area_id, 4, DISP_DOTTEDDEC);
	display("Checksum", (u_int8_t *) &ospf.checksum, 2, DISP_2BYTEDEC);
	sprintf(holder, "%d (%s)", ospf.auth_type, 
		map2str(ospf_auth_map, ospf.auth_type));
	display_string("Authentication", holder);
	display("Authentication data", (u_int8_t *) &ospf.auth_data, 8, 
		DISP_HEX);
      }

}

