/**************************************************************************** 
** File: rip.c
**
** Author: Mike Borella
**
** Comments: Dump RIP header information. 
**
** $Log: rip.c,v $
** Revision 1.6  2000/06/01 18:36:55  mborella
** Made sure that all string maps are properly terminated, added TODO
** file, minor improvments to ICMPv6.
**
** Revision 1.5  2000/05/30 21:37:46  mborella
** Added dump of protocol to minimal mode.
**
** Revision 1.4  2000/05/25 19:02:53  mborella
** Fixed RIP module to differentiate between RIP v1 and v2.
**
*****************************************************************************/

#include <string.h>
#include "global.h"
#include "protocols.h"

/*
 * RIP command map
 */

strmap_t rip_command_map [] =
{
  { RIP_CMD_RQ,          "request" },
  { RIP_CMD_RP,          "reply" },
  { RIP_CMD_POLL,        "poll" },
  { RIP_CMD_POLL_ENTRY,  "poll entry" },
  { 0, "" }
};

extern struct arg_t * my_args;

/*----------------------------------------------------------------------------
**
** dump_rip()
**
** Parse RIP packet and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_rip(packet_t *pkt)
{
  ripv1_header_t       ripv1;
  ripv1_route_header_t ripv1_route;
  ripv2_header_t       ripv2;
  ripv2_route_header_t ripv2_route;
  ripv2_auth_header_t  ripv2_auth;
  int                  route_cnt;
  char                 holder[64];
  u_int16_t            address_family;

  /* 
   * Look at the version to determine whether to use v1 or v2
   */

  if (look_packet_bytes((u_int8_t *) &ripv1, pkt, sizeof(ripv1_header_t)) == 0)
    return;

  if (ripv1.version == 1)
    {
      /* RIP v1 processing here */

      /*
       * Get the RIP v1 static header
       */
      
      if (get_packet_bytes((u_int8_t *) &ripv1, pkt, sizeof(ripv1_header_t)) 
	  == 0)
	return;

      /*
       * Dump header
       */
      
      if (my_args->m)
	{
	  display_minimal_string("RIPv1 ");
	  display_minimal_string(map2str(rip_command_map, ripv1.command));
	  display_minimal_string(" ");
	}
      else
	{
	  /* announcement */
	  display_header_banner("RIPv1 Header");
	  
	  sprintf(holder, "%d (%s)", ripv1.command, 
		  map2str(rip_command_map, ripv1.command));
	  display_string("Command", holder);
	  display("Version", (u_int8_t *) &ripv1.version, 1, DISP_1BYTEDEC);
	  display("MBZ", (u_int8_t *) &ripv1.mbz, 2, DISP_2BYTEDEC);
	}
      
      /*
       * Do the individual routes.
       */
      
      while(1)
	{
	  if (get_packet_bytes((u_int8_t *) &ripv1_route, pkt, 
			       sizeof(ripv1_route_header_t)) == 0)
	    return;
	  
	  /*
	   * Conversions
	   */
	  
	  ripv1_route.addr_fam = ntohs(ripv1_route.addr_fam);
	  ripv1_route.metric = ntohl(ripv1_route.metric);

	  /*
	   * Dump route header
	   */
	  
	  if (my_args->m)
	    {
	      display_minimal((u_int8_t *) &ripv1_route.ipaddr, 4, 
			      DISP_DOTTEDDEC);
	      display_minimal_string(" ");
	    }
	  else
	    {
	      display("Address family", (u_int8_t *) &ripv1_route.addr_fam,
		      2, DISP_2BYTEDEC);
	      display("MBZ", (u_int8_t *) &ripv1_route.mbz1, 2, 
		      DISP_2BYTEDEC);
	      display("IP address", (u_int8_t *) &ripv1_route.ipaddr, 4, 
		      DISP_DOTTEDDEC);
	      display("MBZ", (u_int8_t *) &ripv1_route.mbz2, 8, 
		      DISP_HEX);
	      display("Metric", (u_int8_t *) &ripv1_route.metric, 4, 
		      DISP_4BYTEDEC);
	    }
	  
	}
      
    }
  else
    {
      /* RIP v2 processing here */
  
      /*
       * Get the RIP v2 static header
       */
      
      if (get_packet_bytes((u_int8_t *) &ripv2, pkt, sizeof(ripv2_header_t)) 
	  == 0)
	return;

      /*
       * Conversions
       */
      
      ripv2.domain = ntohs(ripv2.domain);

      /*
       * Dump header
       */
      
      if (my_args->m)
	{
	  display_minimal_string("RIPv2 ");
	  display_minimal_string(map2str(rip_command_map, ripv2.command));
	  display_minimal_string(" ");
	}
      else
	{
	  /* announcement */
	  display_header_banner("RIPv2 Header");
	  
	  sprintf(holder, "%d (%s)", ripv2.command, 
		  map2str(rip_command_map, ripv2.command));
	  display_string("Command", holder);
	  display("Version", (u_int8_t *) &ripv2.version, 1, DISP_1BYTEDEC);
	  display("Domain", (u_int8_t *) &ripv2.domain, 2, DISP_2BYTEDEC);
	}
      
      /*
       * Do the individual routes.
       * First look at the address family to determine the header type.
       */
      
      while (1)
	{
	  if (look_packet_bytes((u_int8_t *) &address_family, pkt, 2) == 0)
	    return;
      
	  /*
	   * If its an auth header, deal with it - otherwise its a regular
	   * route header 
	   */
	  
	  if (address_family == 0xffff)
	    {
	      if (get_packet_bytes((u_int8_t *) &ripv2_auth, pkt, 
				   sizeof(ripv2_auth_header_t)) == 0)
		return;
	      
	      /*
	       * Conversions
	       */
	      
	      ripv2_auth.addr_fam = ntohs(ripv2_auth.addr_fam);
	      ripv2_auth.type = ntohs(ripv2_auth.type);
	  
	      /*
	       * Dump auth header
	       */
	      
	      if (my_args->m)
		{
		}
	      else
		{
		  display("Address family", (u_int8_t *) &ripv2_auth.addr_fam, 
			  2, DISP_2BYTEDEC);
		  display("Type", (u_int8_t *) &ripv2_auth.type, 2, 
			  DISP_2BYTEDEC);
		  display_string("Password", ripv2_auth.passwd);
		}
	    }
	  else
	    {
	      if (get_packet_bytes((u_int8_t *) &ripv2_route, pkt, 
				   sizeof(ripv2_route_header_t)) == 0)
		return;
	      
	      /*
	       * Conversions
	       */
	      
	      ripv2_route.addr_fam = ntohs(ripv2_route.addr_fam);
	      ripv2_route.route_tag = ntohs(ripv2_route.route_tag);
	      ripv2_route.metric = ntohl(ripv2_route.metric);

	      /*
	       * Dump route header
	       */
	      
	      if (my_args->m)
		{
		  display_minimal((u_int8_t *) &ripv2_route.ipaddr, 4, 
				  DISP_DOTTEDDEC);
		  display_minimal_string(" ");
		}
	      else
		{
		  display("Address family", (u_int8_t *) &ripv2_route.addr_fam,
			  2, DISP_2BYTEDEC);
		  display("Route tag", (u_int8_t *) &ripv2_route.route_tag, 2, 
			  DISP_2BYTEDEC);
		  display("IP address", (u_int8_t *) &ripv2_route.ipaddr, 4, 
			  DISP_DOTTEDDEC);
		  display("Netmask", (u_int8_t *) &ripv2_route.netmask, 4, 
			  DISP_DOTTEDDEC);
		  display("Next hop", (u_int8_t *) &ripv2_route.next_hop, 4, 
			  DISP_DOTTEDDEC);
		  display("Metric", (u_int8_t *) &ripv2_route.metric, 4, 
			  DISP_4BYTEDEC);
		}
	      
	    }
	} /* while */
    }
  
}
