/**************************************************************************** 
** File: arp.c
**
** Author: Mike Borella
**
** Comments: Dump ICMP header information
**
** $Log: arp.c,v $
** Revision 1.6  2000/06/19 15:32:12  mborella
** Modified all files to use new address display API (not that it makes
** a bit of difference to the user...) Also tested RIPng and it seems
** to work ok.
**
** Revision 1.5  2000/06/16 22:28:17  mborella
** Added preliminary RIPng support that probably doesn't work, prettied
** up ARP and IGMP, fixed SNMP so that it compiles on Freebsd.
**
** Revision 1.4  2000/06/01 19:28:19  mborella
** Added minimal mode announcements to arp and icmp
**
** Revision 1.3  2000/06/01 18:36:55  mborella
** Made sure that all string maps are properly terminated, added TODO
** file, minor improvments to ICMPv6.
**
** Revision 1.2  2000/05/09 17:16:43  mborella
** Fixed new minimal more for IPX, IPX RIP and ARP.
**
** Revision 1.1  2000/05/04 19:11:42  mborella
** Moved a bunch of source files to the src directory.
**
** Revision 1.6  2000/04/28 23:43:46  mborella
** Fixed hex printing style.  Began implementing port to string map.
**
** Revision 1.5  2000/04/18 18:32:12  mborella
** Fixed Ethernet timestamp.
** ARP now uses IANA string mechanism.
**
** Revision 1.4  2000/04/17 23:20:00  mborella
** Minor fixes to Ethernet, rudimentary support for IP options.
**
** Revision 1.3  2000/04/17 20:54:01  mborella
** Fixed ARP and cleaned it up.
**
** Revision 1.2  2000/04/16 23:28:46  mborella
** This is a major update, maoving the base IP stack protocols to the new
** generic reading and writing system.  Some stuff is still broken.
**
** Revision 1.1.1.1  2000/04/11 17:26:16  mborella
** Initial checkin of release 0.8.2 code.
**
*****************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include "global.h"
#include "protocols.h"

/*
 * ARP opcode map
 */

strmap_t arp_opcode_map[] =
{
  { ARP_OPCODE_REQUEST,  "ARP request" },
  { ARP_OPCODE_REPLY,    "ARP reply" },
  { RARP_OPCODE_REQUEST, "RARP request" },
  { ARP_OPCODE_REPLY,    "RARP reply" },
  { 0, "" }
};

/*
 * ARP hardware map
 */

strmap_t arp_hardware_map[] =
{
  { ARP_HWTYPE_ETHERNET,    "Ethernet" },
  { ARP_HWTYPE_IEEE802,     "IEEE 802" },
  { ARP_HWTYPE_IEEE1394,    "IEEE 1394" },
  { 0, "" }
};

extern strmap_t ethernet_type_map[];
extern struct arg_t *my_args;

/*----------------------------------------------------------------------------
**
** dump_etherarp()
**
** Parse Ethernet-specifc parts of ARP messages
**
**----------------------------------------------------------------------------
*/

void dump_etherarp(packet_t *pkt)
{
  ether_arp_t etharp;

  /*
   * Get the Ethernet section
   */

  if (get_packet_bytes((u_int8_t *) &etharp, pkt, 20) == 0)
    return;

  /*
   * Do minimal or not-so-minimal mode
   */

  if (my_args->m && !my_args->n)
    {
      display_minimal_ipv4((u_int8_t *) &etharp.target_protoaddr);
    }
  else
    if (!my_args->n)
      {
	display("Sender Ether address", (u_int8_t *) &etharp.sender_hwaddr, 6, 
		DISP_HEXCOLONS);
	display_ipv4("Sender IP address",
		     (u_int8_t *) &etharp.sender_protoaddr); 
	display("Target Ether address", (u_int8_t *) &etharp.target_hwaddr, 6, 
		DISP_HEXCOLONS);
	display_ipv4("Target IP address",
		     (u_int8_t *) &etharp.target_protoaddr); 
      }
}

/*----------------------------------------------------------------------------
**
** dump_arp()
**
** Parse ARP header and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_arp(packet_t *pkt)
{
  arp_header_t arp;
  char holder[64];

  /*
   * Get the ARP fixed header
   */

  if (get_packet_bytes((u_int8_t *) &arp, pkt, 8) == 0)
    return;

  /*
   * Conversions
   */

  arp.hwaddr_format = ntohs(arp.hwaddr_format);
  arp.protoaddr_format = ntohs(arp.protoaddr_format);
  arp.opcode = ntohs(arp.opcode);

  /* 
   * Check for minimal mode first
   */

  if (my_args->m && !my_args->n)
    {
      switch(arp.opcode)
	{
	case ARP_OPCODE_REQUEST:
	case ARP_OPCODE_REPLY:
	  display_minimal_string(map2str(arp_opcode_map, arp.opcode));
	  display_minimal_string(" ");
	  break;
	}
    }
  else
    if (!my_args->n)
      {
	/* 
	 * Banner announcement 
	 */
	
	display_header_banner("ARP Header");
	
	/*
	 * Dump hardware type
	 */
	
	sprintf(holder, "0x%x (%s)", arp.hwaddr_format, 
		map2str(arp_hardware_map, arp.hwaddr_format));
	display("Hardware type", (u_int8_t *) holder, strlen(holder), 
		DISP_STRING);
	
	/*
	 * Dump protocol type
	 */
	
	sprintf(holder, "0x%x (%s)", arp.protoaddr_format, 
		map2str(ethernet_type_map, arp.protoaddr_format));
	display("Protocol type", (u_int8_t *) holder, strlen(holder), 
		DISP_STRING);
	
	/* 
	 * Dump hardware and protocol lengths
	 */
	
	display("Hardware length", (u_int8_t *) &arp.hwaddr_length, 1, 
		DISP_1BYTEDEC);
	display("Protocol length", (u_int8_t *) &arp.protoaddr_length, 1, 
		DISP_1BYTEDEC);
	
	/*
	 * Now do the opcode
	 */
	
	sprintf(holder, "%d (%s)", arp.opcode, 
		map2str(arp_opcode_map, arp.opcode));
	display("Opcode", holder, strlen(holder), DISP_STRING);
      }
  
  /*
   * Check hardware type and layer 3 address format
   */
  
  switch(arp.hwaddr_format)
    {
    case ARP_HWTYPE_ETHERNET: 
      
      switch(arp.protoaddr_format)
	{
	case ETHERNET_TYPE_IP:
	  dump_etherarp(pkt);
	  break;
	  
	default:
	  break;
	} /* inner switch */

    default:

    } /* outer switch */
}



