/**************************************************************************** 
** File: dhcp.c
**
** Author: Mike Borella
**
** Comments: Dump DHCP header information.  See RFCs 2131 and 2132.
**
*****************************************************************************/

#include <string.h>
#include "global.h"
#include "protocols.h"

/*
 * DHCP operation type map
 */

strmap_t dhcp_op_map[] =
{
  { DHCP_OP_BOOTREQUEST, "boot request" },
  { DHCP_OP_BOOTREPLY,   "boot reply" },
  { 0, "" }
};

/*
 * DHCP option map
 */

strmap_t dhcp_option_map[] =
{
  { DHCP_OPTION_PAD,             "pad" },
  { DHCP_OPTION_NETMASK,         "netmask" },
  { DHCP_OPTION_TIMEOFFSET,      "time offset" },
  { DHCP_OPTION_ROUTER,          "router" },
  { DHCP_OPTION_TIMESERVER,      "time server" },
  { DHCP_OPTION_NAMESERVER,      "name server" },
  { DHCP_OPTION_DNS,             "DNS server" },
  { DHCP_OPTION_LOGSERVER,       "log server" },
  { DHCP_OPTION_COOKIESERVER,    "cookie server" },
  { DHCP_OPTION_LPRSERVER,       "LPR server" },
  { DHCP_OPTION_IMPRESSSERVER,   "impress server" },
  { DHCP_OPTION_RESLOCSERVER,    "RLP server" },
  { DHCP_OPTION_HOSTNAME,        "hostname" },
  { DHCP_OPTION_BOOTFILESIZE,    "boot file size" },
  { DHCP_OPTION_MERITDUMP,       "Merit dump" },
  { DHCP_OPTION_DOMAINNAME,      "domain name" },
  { DHCP_OPTION_SWAPSERVER,      "swap server" },
  { DHCP_OPTION_ROOTPATH,        "root path" },
  { DHCP_OPTION_EXTSPATH,        "extension path" },
  { DHCP_OPTION_IPFORWARD,       "IP forward" },
  { DHCP_OPTION_NONLOCALSR,      "source routing" },
  { DHCP_OPTION_POLICYFILTER,    "policy filter" },

  { DHCP_OPTION_REQUESTEDIPADDR, "requested IP address" },
  { DHCP_OPTION_IPADDRLEASE,     "IP address lease" },
  { DHCP_OPTION_OVERLOAD,        "overload" },
  { DHCP_OPTION_MESSAGETYPE,     "message type" },
  { DHCP_OPTION_SERVERID,        "server ID" },
  { DHCP_OPTION_PARAMREQLIST,    "parameter request list" },

  { DHCP_OPTION_VENDORCLASSID,   "vendor class ID" },
  { DHCP_OPTION_CLIENTID,        "client ID" },

  { DHCP_OPTION_END,             "end of options" },
  { 0, "" }
};

/*
 * DHCP message type map
 */

strmap_t dhcp_msgtype_map[] =
{
  { DHCP_MSGTYPE_DISCOVER,    "DHCPDISCOVER" },
  { DHCP_MSGTYPE_OFFER,       "DHCPOFFER" },
  { DHCP_MSGTYPE_REQUEST,     "DHCPREQUEST" },
  { DHCP_MSGTYPE_DECLINE,     "DHCPDECLINE" },
  { DHCP_MSGTYPE_ACK,         "DHCPACK" },
  { DHCP_MSGTYPE_NAK,         "DHCPNAK" },
  { DHCP_MSGTYPE_RELEASE,     "DHCPRELASE" },
  { DHCP_MSGTYPE_INFORM,      "DHCPINFORM" },
  { 0, "" }
};

extern struct arg_t *my_args;

/*----------------------------------------------------------------------------
**
** dump_dhcp()
**
** Parse DHCP packet and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_dhcp(packet_t *pkt)
{
  dhcp_header_t dhcp;
  int j;
  u_int32_t i;
  u_int32_t cookie_holder;
  u_int8_t opt;
  char holder[256]; /* this needs to be as big as the biggest option */

  /*
   * Get the DHCP fixed header
   */

  if (get_packet_bytes((u_int8_t *) &dhcp, pkt, sizeof(dhcp_header_t)) == 0)
    return;

  /*
   * Conversions 
   */

  dhcp.xid = ntohl(dhcp.xid);
  dhcp.secs = ntohs(dhcp.secs);
  dhcp.flags = ntohs(dhcp.flags);

  /* announcement */
  if (!my_args->m)
    display_header_banner("DHCP Header");

  /*
   * Display operation, other info
   */

  sprintf(holder, "%d (%s)", dhcp.op, map2str(dhcp_op_map, dhcp.op));
  if (my_args->m)
    {
      display_minimal_string(map2str(dhcp_op_map, dhcp.op));
      display_minimal_string(" ");
    }
  else
    {
      display("Operation", (u_int8_t *) holder, strlen(holder), DISP_STRING);
      display("Hardware addr type", (u_int8_t *) &dhcp.htype, 1, 
	      DISP_1BYTEDEC);
      display("Hardware addr length", (u_int8_t *) &dhcp.hlen, 1, 
	      DISP_1BYTEDEC);
      display("Hops", (u_int8_t *) &dhcp.hops, 1, DISP_1BYTEDEC); 
      display("Transaction ID", (u_int8_t *) &dhcp.xid, 4, DISP_HEX);
      display("Seconds", (u_int8_t *) &dhcp.secs, 2, DISP_2BYTEDEC);
      display("Flags", (u_int8_t *) &dhcp.flags, 2, DISP_HEX);
    }

  /* 
   * Display IPs 
   */

  if (!my_args->m)
    {
      display_ipv4("Client addr", (u_int8_t *) &dhcp.ciaddr);
      display_ipv4("Your addr", (u_int8_t *) &dhcp.yiaddr);
      display_ipv4("Next server addr", (u_int8_t *) &dhcp.siaddr);
      display_ipv4("Relay agent addr", (u_int8_t *) &dhcp.giaddr);
      printf("Client hardware addr:   \n");
    }

  /*
   * Display names
   */
  
  if (!my_args->m)
    {
      display("Server host name", (u_int8_t *) dhcp.sname, strlen(dhcp.sname),
	      DISP_STRING);
      display("Boot file name", (u_int8_t *) dhcp.file, strlen(dhcp.file),
	      DISP_STRING);
    }
  
  /*
   * Look for BOOTP cookie
   */

  i = htonl(BOOTP_COOKIE);
  if (look_packet_bytes((u_int8_t *) &cookie_holder, pkt, 4) == 0)
    return;
  if (i == cookie_holder && !my_args->m)
    {
      if (skip_packet_bytes(pkt, 4) == 0)
	return;
      display("BOOTP cookie", (u_int8_t *) &cookie_holder, 4, DISP_HEX);
    }

  /*
   * Parse the options list
   */

  while (get_packet_bytesleft(pkt))
    {     
      u_int8_t len=0;
      
      /*
       * Get the option number and dump it
       */
      
      if (get_packet_bytes((u_int8_t *) &opt, pkt, 1) == 0)
	return;
      sprintf(holder, "%d (%s)", opt, map2str(dhcp_option_map, opt));
      if (!my_args->m)
	display("Option", (u_int8_t *) holder, strlen(holder), DISP_STRING);
      
      /*
       * If its a pad, go back and get another
       */
      
      if (opt == DHCP_OPTION_PAD)
	continue;

      /*
       * If its an end-of-options indicator, bail out
       */

      if (opt == DHCP_OPTION_END)
	break;

      /*
       * Otherwise, grab and dump the length
       */

      if (get_packet_bytes((u_int8_t *) &len, pkt, 1) == 0)
	return;
      if (!my_args->m)
	display("  Length", (u_int8_t *) &len, 1, DISP_1BYTEDEC);
      
      switch(opt)
	{
	case DHCP_OPTION_NETMASK: /* 1 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  if (!my_args->m)
	    display_ipv4("  Mask", (u_int8_t *) holder);
	  break;

	case DHCP_OPTION_TIMEOFFSET: /* 2 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  if (!my_args->m)
	    display("  Offset", (u_int8_t *) holder, len, DISP_4BYTEDEC);
	  break;

	case DHCP_OPTION_ROUTER: /* 3 */
	case DHCP_OPTION_TIMESERVER: /* 4 */
	case DHCP_OPTION_NAMESERVER: /* 5 */
	case DHCP_OPTION_DNS: /* 6 */
	case DHCP_OPTION_NETBIOSNAMESERV: /* 44 */
	  j = 0;
	  while (j < len / 4)
	    {
	      if (get_packet_bytes((u_int8_t *) holder, pkt, 4) == 0)
		return;
	      if (!my_args->m)
		display_ipv4("  Address", (u_int8_t *) holder);
	      j++;
	    }
	  break;

	case DHCP_OPTION_HOSTNAME: /* 12 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  holder[len] = '\0';
	  if (!my_args->m)
	    display("  Host name", (u_int8_t *) holder, strlen(holder), 
		    DISP_STRING);
	  break;

	case DHCP_OPTION_DOMAINNAME: /* 15 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  holder[len] = '\0';
	  if (!my_args->m)
	    display("  Domain name", (u_int8_t *) holder, strlen(holder),
		    DISP_STRING);
	  break;

	case DHCP_OPTION_VENDORSPECIFIC: /* 43 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  if (!my_args->m)
	    display("  Parameters", (u_int8_t *) holder, len, DISP_HEX);
	  break;

	case DHCP_OPTION_NETBIOSNODETYPE: /* 46 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  switch(holder[0])
	    {
	    case 0x1: 
	      sprintf(holder, "%x B", holder[0]);
	      break;
	    case 0x2:
	      sprintf(holder, "%x P", holder[0]);
	      break;
	    case 0x4:
	      sprintf(holder, "%x M", holder[0]);
	      break;
	    case 0x8:
	      sprintf(holder, "%x H", holder[0]);
	      break;
	    }
	  break;

	case DHCP_OPTION_REQUESTEDIPADDR: /* 50 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  if (!my_args->m)
	    display_ipv4("  Address", (u_int8_t *) holder);
	  else
	    {
	      display_minimal_ipv4((u_int8_t *) holder);
	      display_minimal_string(" ");
	    }
	  break;

	case DHCP_OPTION_IPADDRLEASE: /* 51 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, 4) == 0)
	    return;
	  if (!my_args->m)
	    display("  Lease time", (u_int8_t *) holder, 4, DISP_4BYTEDEC);
	  break;

	case DHCP_OPTION_MESSAGETYPE: /* 53 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, 1) == 0)
	    return;
	  sprintf(holder, "%d (%s)", holder[0], 
		  map2str(dhcp_msgtype_map, holder[0]));
	  if (!my_args->m)
	    display_string("  Message type", holder);
	  else
	    {
	      display_minimal_string(map2str(dhcp_msgtype_map, holder[0]));
	      display_minimal_string(" ");
	    }
	  break;

	case DHCP_OPTION_SERVERID: /* 54 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  if (!my_args->m)
	    display_ipv4("  Address", (u_int8_t *) holder);
	  break;

	case DHCP_OPTION_PARAMREQLIST: /* 55 */
	  for (i=0; i < len; i++)
	    {
	      if (get_packet_bytes((u_int8_t *) holder, pkt, 1) == 0)
		return;
	      if (!my_args->m)
		display("  Option", (u_int8_t *) holder, 1, DISP_1BYTEDEC);
	    }
	  break;

	case DHCP_OPTION_MAXDHCPMSGSIZE: /* 57 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  if (!my_args->m)
	    display("  Size", (u_int8_t *) holder, 2, DISP_2BYTEDEC);
	  break;

	case DHCP_OPTION_RENEWALTIME: /* 58 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  if (!my_args->m)
	    display("  Renewal time", (u_int8_t *) ntohl((u_int32_t) holder),
		    4, DISP_4BYTEDEC);
	  break;

	case DHCP_OPTION_REBINDINGTIME: /* 59 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  if (!my_args->m)
	    display("  Rebinding time", (u_int8_t *) ntohl((u_int32_t) holder),
		    4, DISP_4BYTEDEC);
	  break;

	case DHCP_OPTION_VENDORCLASSID: /* 60 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  if (!my_args->m)
	    display("  Parameters", (u_int8_t *) holder, len, DISP_HEX);
	  break;

	case DHCP_OPTION_CLIENTID: /* 61 */
	  if (get_packet_bytes((u_int8_t *) holder, pkt, len) == 0)
	    return;
	  if (!my_args->m)
	    display("  Parameters", (u_int8_t *) holder, len, DISP_HEX);
	  break;

	default:
	  /* do nothing for now */
	} /* switch */

    } /* while */
}




