/**************************************************************************** 
** File: display.c
**
** Author: Mike Borella
**
** Comments: Displaying functions
**
** $Log: display.c,v $
** Revision 1.11  2000/06/19 19:47:48  mborella
** Fixed SLP support, v1 seems to be working ok, v2 is not tested.  Also
** fixed bug in displaying routine for DISP_DEC.
**
** Revision 1.10  2000/06/19 16:41:25  mborella
** Made a new API display type: DISP_DEC, which replaces all decimal type
** displaying functions.  Required overcoming the usual hairy byte ordering
** issues, and I'm hoping it isn't going to screw anything up.
**
** Revision 1.9  2000/06/16 23:03:21  mborella
** Added special printing functions for ipv4 and ipv6 addresses.
**
** Revision 1.8  2000/05/13 00:57:17  mborella
** Fixed DNS crashes by disabling anything besides the basic header until
** I get a chance to fix it.  Minimal mode does nothing so far.  Also fixed
** a real ugly crash in the main display routine due to the length of
** a label being too long.
**
** Revision 1.7  2000/05/10 18:26:50  mborella
** Added TCP option support, cleaned up minimal mode, added TCP window
** advert to minimal mode, added timestamp to all lines in minimal mode.
**
** Revision 1.6  2000/05/09 19:41:01  mborella
** Fixed minimal mode for ICMP, fixed IPv6 header printing, added the minimal
** string display mode.
**
** Revision 1.5  2000/05/09 17:16:43  mborella
** Fixed new minimal more for IPX, IPX RIP and ARP.
**
** Revision 1.4  2000/05/08 23:36:04  mborella
** Got rid of addrtoname.c module.  Functionality is now taken care of in our
** own way.
**
** Revision 1.3  2000/05/08 22:55:45  mborella
** Basic support for IPv6.
**
** Revision 1.2  2000/05/08 21:18:23  mborella
** Made "minimal mode" into a one-line tcpdump-like mode.
**
** Revision 1.1  2000/05/04 19:21:55  mborella
** Adding more files to src.
**
** Revision 1.8  2000/04/28 23:43:46  mborella
** Fixed hex printing style.  Began implementing port to string map.
**
** Revision 1.7  2000/04/18 22:11:58  mborella
** Fixed IP encapsulation in ICMP.
**
** Revision 1.6  2000/04/17 23:20:00  mborella
** Minor fixes to Ethernet, rudimentary support for IP options.
**
*****************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include "display.h"

#define LABEL_LENGTH 22

int packet_displayed;

/*----------------------------------------------------------------------------
**
** reverse_byte_order()
**
** Reverses the byte order of an arbitrarily long byte string
**
**----------------------------------------------------------------------------
*/

inline void reverse_byte_order(u_int8_t *s, int len)
{
  u_int8_t * temp;
  int i;

  temp = (u_int8_t *) malloc(len);
  for (i = 0; i < len; i++)
      temp[len-i-1] = s[i];
  memcpy(s, temp, len);
  free(temp);
}

/*----------------------------------------------------------------------------
**
** display_header_line()
**
** Writes out a line of dashes meant to be part of a header
**
**----------------------------------------------------------------------------
*/

inline void display_header_line(void)
{
  printf("----------------------------------------------------------------\n");
}

/*----------------------------------------------------------------------------
**
** display_header_line2()
**
** Writes out a line of equals meant to be part of a header
**
**----------------------------------------------------------------------------
*/

inline void display_header_line2(void)
{
  printf("================================================================\n");
}

/*----------------------------------------------------------------------------
**
** display_header_banner()
**
** Writes out a generic header banner for protocol headers.
**
** ISSUE: Do we want to limit the length of the banner string so that it 
** doesn't run off the screen?
**
**----------------------------------------------------------------------------
*/

inline void display_header_banner(char *banner)
{
  display_header_line();
  printf("                        %s\n", banner);
  display_header_line();
}

/*----------------------------------------------------------------------------
**
** display()
**
** Display a protocol field or some other string according to our rules
**
** Parameters:
**   label - always printed, must be null terminated string
**   content - the stuff to display, usually the value of the field
**   length - the number of bytes to display from the content parameter
**   format - the specific format to use when displaying
**
**----------------------------------------------------------------------------
*/

inline void display(char * label, u_int8_t * content, u_int8_t length, 
	     display_t format)
{
  int i,j;
  int label_len;
  char label_truncated[LABEL_LENGTH+1];

  /*
   * Do sanity checks here
   */

  label_len = strlen(label);
  if (label_len >= LABEL_LENGTH) 
    {
      strncpy(label_truncated, label, LABEL_LENGTH);
      label_truncated[LABEL_LENGTH] = '\0';
      label_len = LABEL_LENGTH;
      printf("%s:", label_truncated); 
    }
  else
    printf("%s:", label);

  /*
   * Figure out how many spaces to pad with
   */

  for (i=0; i < LABEL_LENGTH+1-label_len; i++)
    putchar(' ');

  /*
   * Do the actual displaying
   */

  display_minimal(content, length, format);

  /*
   * Trailing printf()
   */

  printf("\n");
}

/*----------------------------------------------------------------------------
**
** display_minimal()
**
** Display a protocol field or some other string in minimal mode.
** This means that there is no label, no padding, and no carriage return.
**
** Parameters:
**   content - the stuff to display, usually the value of the field
**   length - the number of bytes to display from the content parameter
**   format - the specific format to use when displaying
**
**----------------------------------------------------------------------------
*/

inline void display_minimal(u_int8_t * content, u_int8_t length, 
			    display_t format)
{
  int i,j;
  u_int16_t      * byte2;
  u_int32_t      * byte4;
  u_int8_t       * byte1;
  int              seen_a_one;
  u_int8_t         bit_holder;
  struct in_addr * ip_holder;
  u_int8_t         ipv6_holder;
  u_int8_t       * ptr;
  u_int8_t       * temp;
  double           sum;
  u_int32_t        multiplier;

  /*
   * We have displayed part of this packet, so we need a carriage return.
   */

  packet_displayed = 1;

  /*
   * Depending on the display format, we may do different things
   */

  switch(format)
    {
    case DISP_DEC:
      temp = (u_int8_t *) malloc(length);
      memcpy(temp, content, length);
#if !defined(WORDS_BIGENDIAN)
      reverse_byte_order(temp, length);
#endif
      sum = 0.0;
      multiplier = 1;
      for (i = length-1; i >= 0; i--)
	{
	  ptr = temp+i;
	  sum += (double) (multiplier * *ptr);
	  multiplier = multiplier * 256;
	}
      printf("%.f", sum);
      free(temp);
      break;

    case DISP_4BYTEDEC:
      byte4 = (u_int32_t *) content;
      printf("%u", *byte4);
      break;

    case DISP_2BYTEDEC:
      byte2 = (u_int16_t *) content;
      printf("%u", *byte2);
      break;

    case DISP_1BYTEDEC:
      byte1 = (u_int8_t *) content;
      printf("%u", *byte1);
      break;

    case DISP_DOTTEDDEC:
      ip_holder = (struct in_addr *) content;
      for (i=0; i<4; i++)
	if (i==3)
	  printf("%d", content[i]);
	else
	  printf("%d.", content[i]);
      break;

    case DISP_HEX:
      printf("0x");
      for (i=0; i < length; i++)
	{
	  if (content[i] < 16)
	    printf("0%x", content[i]);
	  else
	    printf("%x", content[i]);
	}
      break;

    case DISP_HEXCOLONS:
      for (i=0; i < length; i++)
	{
	  if (content[i] < 16)
	    printf("0%x", content[i]);
	  else
	    printf("%x", content[i]);
	  if (i != length-1)
	    printf(":");
	}
      break;
      
    case DISP_HEXCOLONS4:
      for (i=0; i < length; i++)
	{
	  printf("%x", content[i]);
	  if (i % 2 == 1 && i != length - 1)
	    printf(":");
	}      
      break;

    case DISP_BINNLZ:
      seen_a_one = 0;
      for (i=0; i < length; i++)
	{
	  bit_holder = 128;
	  for (j=1; j<=8; j++)
	    {
	      if (content[i] & bit_holder)
		{
		  seen_a_one = 1;
		  printf("1");
		}
	      else
		{
		  if (seen_a_one || (i == length -1 && j == 8))
		    printf("0");
		}
	      bit_holder = bit_holder >> 1;
	    }
	}
      break;

    case DISP_STRING:
      /* 
       * Rather than just ass-u-ming that we've been handed a properly
       * null terminated string, we created one of our own
       */

      ptr = (u_int8_t *) malloc(length+1);
      memcpy(ptr, content, length);
      ptr[length] = '\0';

      /* print it then free it */
      printf("%s", ptr);
      free(ptr);
      break;

    default: /* oops, this shouldn't happen... */
    }
}

/*----------------------------------------------------------------------------
**
** display_minimal_string()
**
** Same as display_minimal() but we assume that the content argument is 
** a string.
**
**----------------------------------------------------------------------------
*/

inline void display_minimal_string(u_int8_t * content)
{
  display_minimal(content, strlen(content), DISP_STRING);
}

/*----------------------------------------------------------------------------
**
** display_string()
**
** Same as display() but we assume that the content argument is 
** a string.
**
**----------------------------------------------------------------------------
*/

inline void display_string(u_int8_t * label, u_int8_t * content)
{
  display(label, content, strlen(content), DISP_STRING);
}

/*----------------------------------------------------------------------------
**
** display_ipv4()
**
** Same as display() but we assume that the content argument is 
** a 4-byte IPv4 address.
**
**----------------------------------------------------------------------------
*/

inline void display_ipv4(u_int8_t * label, u_int8_t * addr)
{
  display(label, addr, 4, DISP_DOTTEDDEC);
}

/*----------------------------------------------------------------------------
**
** display_ipv6()
**
** Same as display() but we assume that the content argument is 
** a 16-byte IPv6 address.
**
**----------------------------------------------------------------------------
*/

inline void display_ipv6(u_int8_t * label, u_int8_t * addr)
{
  display(label, addr, 16, DISP_HEXCOLONS4);
}

/*----------------------------------------------------------------------------
**
** display_minimal_ipv4()
**
** Same as display_minimal() but we assume that the content argument is 
** a 4-byte IPv4 address.
**
**----------------------------------------------------------------------------
*/

inline void display_minimal_ipv4(u_int8_t * content)
{
  display_minimal(content, 4, DISP_DOTTEDDEC);
}

/*----------------------------------------------------------------------------
**
** display_minimal_ipv6()
**
** Same as display_minimal() but we assume that the content argument is 
** a 16-byte IPv6 address.
**
**----------------------------------------------------------------------------
*/

inline void display_minimal_ipv6(u_int8_t * content)
{
  display_minimal(content, 16, DISP_HEXCOLONS4);
}

