/**************************************************************************** 
** File: dns.c
**
** Author: Mike Borella
**
** Comments: Dump DNS header information
**
** $Log: dns.c,v $
** Revision 1.11  2000/06/19 15:32:12  mborella
** Modified all files to use new address display API (not that it makes
** a bit of difference to the user...) Also tested RIPng and it seems
** to work ok.
**
** Revision 1.10  2000/06/07 15:31:58  mborella
** Added IPv6 support to DNS.
**
** Revision 1.9  2000/06/01 18:36:55  mborella
** Made sure that all string maps are properly terminated, added TODO
** file, minor improvments to ICMPv6.
**
** Revision 1.8  2000/05/21 22:20:31  mborella
** Fixed RIP.
**
** Revision 1.7  2000/05/21 20:44:33  mborella
** Finally got DNS working.  Not perfect or complete but reasonably good.
**
** Revision 1.6  2000/05/21 19:39:02  mborella
** Added API for DNS label manipulation.  What a mess.  Added to the
** packet manip API as well, so that a place in a packet can be "marked"
** and the distance from that mark can be computed.
**
** Revision 1.5  2000/05/13 00:57:17  mborella
** Fixed DNS crashes by disabling anything besides the basic header until
** I get a chance to fix it.  Minimal mode does nothing so far.  Also fixed
** a real ugly crash in the main display routine due to the length of
** a label being too long.
**
** Revision 1.4  2000/05/10 20:37:26  mborella
** Added HTTP support...hooohaaa! Also managed to create a very nasty
** little memory overwrite bug, which necessitated the creation of
** debug mode from configure.in.
**
** Revision 1.3  2000/05/09 22:49:35  mborella
** Added documentation files.  Enabled DHCP and DNS output.  Trying to fix
** DHCP output.
**
** Revision 1.2  2000/05/09 00:21:54  mborella
** Got rid of some warnings on FreeBSD compile.
**
** Revision 1.1  2000/05/04 19:11:42  mborella
** Moved a bunch of source files to the src directory.
**
** Revision 1.2  2000/04/26 20:53:19  mborella
** Got rid of global packet end indicator.  Fixed SIP output with Cullen's
** patch, then went and broke it again (see comments for details).  Major
** mods to DHCP, made it work with the new architecture.
**
** Revision 1.1.1.1  2000/04/11 17:26:16  mborella
** Initial checkin of release 0.8.2 code.
**
*****************************************************************************/

#include <string.h>
#include "global.h"
#include "protocols.h"
#include "dns_labels.h"

/*
 * QR flag map
 */

strmap_t dns_qrflag_map [] =
{
  { DNS_QRFLAG_QUERY,     "query" },
  { DNS_QRFLAG_RESPONSE,  "response" },
  { 0, "" }
};

/*
 * Opcode flag map
 */

strmap_t dns_opcodeflag_map [] =
{
  { DNS_OPCODEFLAG_STANDARD,    "standard" },
  { DNS_OPCODEFLAG_INVERSE,     "inverse" },
  { DNS_OPCODEFLAG_STATUS,      "status" },
  { 0, "" }
};


/*
 * Rcode (return code) flag map
 */

strmap_t dns_rcodeflag_map [] =
{
  { DNS_RCODEFLAG_NOERROR,          "no error" },
  { DNS_RCODEFLAG_FORMATERROR,      "format error" },
  { DNS_RCODEFLAG_SERVERERROR,      "server error" },
  { DNS_RCODEFLAG_NAMEERROR,        "name error" },
  { DNS_RCODEFLAG_NOTIMPLEMENTED,   "not implemented" },
  { DNS_RCODEFLAG_SERVICEREFUSED,   "service refused" },
  { 0, "" }
};

/*
 * Query type map
 */

strmap_t dns_querytype_map [] =
{
  { DNS_QUERYTYPE_A,           "A - IP address" },
  { DNS_QUERYTYPE_NS,          "NS - name server" },
  { DNS_QUERYTYPE_CNAME,       "CNAME - canonical name" },
  { DNS_QUERYTYPE_SOA,         "SOA - start of auth record" },
  { DNS_QUERYTYPE_PTR,         "PTR - pointer record" },
  { DNS_QUERYTYPE_HINFO,       "HINFO - host info" },
  { DNS_QUERYTYPE_MX,          "MX - mail exchange record" },
  { DNS_QUERYTYPE_AAAA,        "AAAA - IPv6 address" },
  { DNS_QUERYTYPE_AXFR,        "AXFR - zone transfer request" },
  { DNS_QUERYTYPE_ANY,         "ANY - all records request" },
  { 0, "" }
};

/*
 * Query type map
 */

strmap_t dns_queryclass_map [] =
{
  { DNS_QUERYCLASS_IP,         "Internet" },
  { 0, "" }
};

extern struct arg_t *my_args;
void parse_dns_labels(packet_t *pkt, char *name);

#ifdef DEBUG
void dump_ascii(u_char *, u_char *);
#endif

/*----------------------------------------------------------------------------
**
** dump_dns_questions()
**
** Parse DNS questions and display them
**
**----------------------------------------------------------------------------
*/

void dump_dns_questions(packet_t *pkt, u_int8_t num)
{
  char holder[256];
  u_int16_t query_type, query_class;

  while(num > 0)
    {
      /*
       * Parse the name
       */
      
      parse_dns_labels(pkt, holder);
      
      /*
       * Parse the query type and class
       */
      
      if (get_packet_bytes((u_int8_t *) &query_type, pkt, 2) == 0)
	return;
      if (get_packet_bytes((u_int8_t *) &query_class, pkt, 2) == 0)
	return;
      
      /*
       * Conversions
       */
      
      query_type = ntohs(query_type);
      query_class = ntohs(query_class);
      
      /*
       * Dump the info
       */
      
      if (my_args->m)
	{
	  display_minimal_string(holder);
	  display_minimal_string(" ");
	}
      else
	{
	  display_string("Query", holder); 
	  sprintf(holder, "%d (%s)", query_type, 
		  map2str(dns_querytype_map, query_type));
	  display_string("  Query type", holder); 
	  sprintf(holder, "%d (%s)", query_class, 
		  map2str(dns_queryclass_map, query_class));
	  display_string("  Query class", holder); 
	}

      num --;
    }
}

/*----------------------------------------------------------------------------
**
** dump_dns_answers()
**
** Parse DNS answers and display them
**
**----------------------------------------------------------------------------
*/
void dump_dns_answers(packet_t *pkt, u_int8_t num, char *answer_type)
{
  char holder[256];
  u_int16_t query_type, query_class;
  u_int32_t ttl;
  u_int16_t rdl;
  u_int8_t resource_data[64];

  while(num > 0)
    {
      /*
       * Parse the name
       */
      
      parse_dns_labels(pkt, holder);
      
      /*
       * Parse the query type and class
       */
      
      if (get_packet_bytes((u_int8_t *) &query_type, pkt, 2) == 0)
	return;
      if (get_packet_bytes((u_int8_t *) &query_class, pkt, 2) == 0)
	return;
      if (get_packet_bytes((u_int8_t *) &ttl, pkt, 4) == 0)
	return;
      if (get_packet_bytes((u_int8_t *) &rdl, pkt, 2) == 0)
	return;
      
      /*
       * Conversions
       */
      
      query_type = ntohs(query_type);
      query_class = ntohs(query_class);
      ttl = ntohl(ttl);
      rdl = ntohs(rdl);

      /*
       * Get the resource data
       */

      switch (query_type)
	{
	case DNS_QUERYTYPE_NS:
	case DNS_QUERYTYPE_CNAME:
	case DNS_QUERYTYPE_SOA:
	case DNS_QUERYTYPE_PTR:
	  parse_dns_labels(pkt, resource_data);
	  break;

	case DNS_QUERYTYPE_A:
	default:
	  if (get_packet_bytes((u_int8_t *) &resource_data, pkt, 
			       rdl) == 0)
	    return;
	  break;
	  
	}

      /*
       * Dump the info
       */
      
      if (my_args->m)
	{
	  if (query_type == DNS_QUERYTYPE_A && !strcmp(answer_type, "Answers"))
	    {
	      display_minimal_ipv4((u_int8_t *) resource_data);
	      display_minimal_string(" ");
	    }
	  if (query_type == DNS_QUERYTYPE_AAAA && 
	      !strcmp(answer_type, "Answers") && rdl == 16 )
	    {
	      display_minimal_ipv6((u_int8_t *) resource_data);
	      display_minimal_string(" ");
	    }
	}
      else
	{
	  display_string(answer_type, holder); 
	  sprintf(holder, "%d (%s)", query_type, 
		  map2str(dns_querytype_map, query_type));
	  display_string("  Query type", holder); 
	  sprintf(holder, "%d (%s)", query_class, 
		  map2str(dns_queryclass_map, query_class));
	  display_string("  Query class", holder); 
	  display("  TTL", (u_int8_t *) &ttl, 4, DISP_4BYTEDEC);
	  display("  Resource data length", (u_int8_t *) &rdl, 2, 
		  DISP_2BYTEDEC);
	  switch(query_type)
	    {
	    case DNS_QUERYTYPE_A:
	      display_ipv4("  Resource data", (u_int8_t *) resource_data);
	      break;

	    case DNS_QUERYTYPE_AAAA:
	      display_ipv6("  Resource data", (u_int8_t *) resource_data);
	      break;

	    case DNS_QUERYTYPE_NS:
	    case DNS_QUERYTYPE_CNAME:
	    case DNS_QUERYTYPE_SOA:
	    case DNS_QUERYTYPE_PTR:
	      display_string("  Resource data", resource_data);
	      break;
	      
	    default:
	      display("  Resource data", (u_int8_t *) resource_data, rdl, 
		      DISP_HEX);	      
	    }
	}
      
      num --;
    }

}

/*----------------------------------------------------------------------------
**
** dump_dns()
**
** Parse DNS packet and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_dns(packet_t *pkt)
{
  dns_header_t dns;
  u_int8_t f_qr, f_opcode, f_aa, f_tc, f_rd, f_ra, f_zero, f_rcode;
  char holder[64];

  /*
   * Mark the beginning of the DNS portion so that labels can be stored
   * properly
   */

  set_packet_mark(pkt);

  /*
   * Reset the DNS labels structures for a new DNS packet
   */

  reset_dnslabels();

  /*
   * Get the header
   */

  if (get_packet_bytes((u_int8_t *) &dns, pkt, 12) == 0)
    return;
  
  /*
   * Conversions
   */

  dns.id = ntohs(dns.id);
  dns.number_questions = ntohs(dns.number_questions);
  dns.number_answers = ntohs(dns.number_answers);
  dns.number_authority = ntohs(dns.number_authority);
  dns.number_additional = ntohs(dns.number_additional);
  f_qr = dns.flag_qr;
  f_opcode = dns.flag_opcode;
  f_aa = dns.flag_aa;
  f_tc = dns.flag_tc;
  f_rd = dns.flag_rd;
  f_ra = dns.flag_ra;
  f_zero = dns.flag_zero;
  f_rcode = dns.flag_rcode;
  
  /*
   * Print it
   */

  if (my_args->m)
    {
      display_minimal_string("DNS ");
      display_minimal_string(map2str(dns_qrflag_map, f_qr));
      display_minimal_string(" ");
    }
  else
    {
      /* announcement */
      display_header_banner("DNS Header");
	      
      /* identification */
      display("Identification", (u_int8_t *) &dns.id, 2, DISP_2BYTEDEC);

      /* flags */
      sprintf(holder, "%d (%s)", f_qr, map2str(dns_qrflag_map, f_qr));
      display_string("Flag query/response", holder);
      sprintf(holder, "%d (%s)", f_opcode, 
	      map2str(dns_opcodeflag_map, f_opcode));
      display_string("Flag opcode", holder);
      display("Flag auth answer", (u_int8_t *) &f_aa, 1, DISP_BINNLZ);
      display("Flag trunctated", (u_int8_t *) &f_tc, 1, DISP_BINNLZ);
      display("Flag recursion desired", (u_int8_t *) &f_ra, 1, DISP_BINNLZ);
      display("Flag recursion available", (u_int8_t *) &f_ra, 1, DISP_BINNLZ);
      display("Flag zero", (u_int8_t *) &f_zero, 1, DISP_BINNLZ);
      sprintf(holder, "%d (%s)", f_rcode, 
	      map2str(dns_rcodeflag_map, f_rcode));      
      display_string("Flag return code", holder);

      /* numbers of questions and answers */
      display("# of questions", (u_int8_t *) &dns.number_questions, 2, 
	      DISP_2BYTEDEC);
      display("# of answers", (u_int8_t *) &dns.number_answers, 2, 
	      DISP_2BYTEDEC);
      display("# of authorizarion RRs", (u_int8_t *) &dns.number_authority, 2, 
	      DISP_2BYTEDEC);
      display("# of additional RRs", (u_int8_t *) &dns.number_additional, 2, 
	      DISP_2BYTEDEC);
    }

  /*
   * Parse the question and answers
   */

  dump_dns_questions(pkt, dns.number_questions);
  dump_dns_answers(pkt, dns.number_answers, "Answers");
  dump_dns_answers(pkt, dns.number_authority, "Auth answers");
  dump_dns_answers(pkt, dns.number_additional, "Addtl answers");

}


/*----------------------------------------------------------------------------
**
** parse_dns_labels()
**
** Recursively parse a label entry in a DNS packet.  We assume that the
** current byte of the packet points to the first label count.  Store
** the DNS name in the string passed in, and let's assume that this string 
** is big enough.
**
**----------------------------------------------------------------------------
*/

void parse_dns_labels(packet_t *pkt, char *name)
{
  u_int8_t count;
  u_int16_t offset;
  int read_a_label = 0;
  char label[32];
  int first_offset = 0;

  while(1)
    {
      if (get_packet_bytes((u_int8_t *) &count, pkt, 1) == 0)
	return;

      /*
       * A count of 0 indicates an "end of string", so quit
       */

      if (count == 0) 
	break;

      /*
       * This is a new label, so let's mark it as such 
       */

      if (read_a_label)
	set_dnslabel_next();
      new_dnslabel();
      if (!first_offset)
	{
	  first_offset = get_packet_markdistance(pkt)-1;
	}
      set_dnslabel_offset(get_packet_markdistance(pkt)-1);

      /* 
       * A count > 192 indicates a pointer.  What an ugly way of 
       * saving a few bytes
       */

      if (count >= 192)
	{
	  /*
	   * There's a pointer in this label.  Sigh.  Let's grab the 
	   * 14 low-order bits and run with them...
	   */
	  
	  offset = count - 192;
	  offset = offset << 8;
	  if (get_packet_bytes((u_int8_t *) &count, pkt, 1) == 0)
	    return;
	  offset = offset + count;

	  get_dnslabel(offset, name);
	  set_dnslabel_label(name);
	  return;
	}

      /*
       * This is normal behavior, if anything is "normal" for DNS.
       * Grab count bytes and stick them in the label string, follow it
       * up with a "." to separate the hierarchy
       */

      if (get_packet_bytes((u_int8_t *) &label, pkt, count ) == 0)
	return;

      label[count] = '\0';
      set_dnslabel_label(label);

      read_a_label = 1;
    }
  
  get_dnslabel(first_offset, name);
  
  return;
}




