/**************************************************************************** 
** File: ethernet.c
**
** Author: Mike Borella
**
** Comments: Dump ethernet packets
**
** $Log: ethernet.c,v $
** Revision 1.7  2000/06/01 18:36:55  mborella
** Made sure that all string maps are properly terminated, added TODO
** file, minor improvments to ICMPv6.
**
** Revision 1.6  2000/05/09 17:43:47  mborella
** Made some fixes to minimal mode ethernet, so that the ethernet type field
** is always displayed.  However, the results are weird...
**
** Revision 1.5  2000/05/09 00:21:54  mborella
** Got rid of some warnings on FreeBSD compile.
**
** Revision 1.4  2000/05/08 22:55:45  mborella
** Basic support for IPv6.
**
** Revision 1.3  2000/05/08 21:18:23  mborella
** Made "minimal mode" into a one-line tcpdump-like mode.
**
** Revision 1.2  2000/05/04 20:22:55  mborella
** Fixed a handful of configuration issues.  Unfortuantely, the move
** to automake broke a couple of things.  It won't build right now
** until I re-write sip, sdp and mgcp.
**
** Revision 1.1  2000/05/04 19:11:42  mborella
** Moved a bunch of source files to the src directory.
**
** Revision 1.10  2000/04/29 00:21:07  mborella
** Port to string conversions work, but we're nowhere near complete yet.
**
** Revision 1.9  2000/04/28 23:43:46  mborella
** Fixed hex printing style.  Began implementing port to string map.
**
** Revision 1.8  2000/04/18 19:52:04  mborella
** Minor fixes for minimal (-m) mode.
**
** Revision 1.7  2000/04/18 18:32:12  mborella
** Fixed Ethernet timestamp.
** ARP now uses IANA string mechanism.
**
** Revision 1.6  2000/04/18 17:16:27  mborella
** Updates for IANA string printing.
**
*****************************************************************************/

#include "global.h"
#include "protocols.h"

extern struct arg_t *my_args;

/*
 * Ethernet type map
 */

strmap_t ethernet_type_map[] =
{
  { ETHERNET_TYPE_IP,   "IP" },
  { ETHERNET_TYPE_ARP,  "ARP" },
  { ETHERNET_TYPE_RARP, "RARP" },
  { ETHERNET_TYPE_IPX,  "IPX" },
  { ETHERNET_TYPE_IPV6, "IPV6" },
  { 0, "" }
};

/*----------------------------------------------------------------------------
**
** dump_ethernet()
**
** Process packets from the DLT_EN10MB interface type
**
**----------------------------------------------------------------------------
*/

void dump_ethernet(packet_t *pkt)
{
  ether_header_t eth;
  char holder[64];

  /*
   * Read the header
   */

  if (get_packet_bytes((u_int8_t *) &eth, pkt, 14) == 0)
    return;

  /*
   * Conversions
   */

  eth.type = ntohs(eth.type);

  /*
   * Dump header
   */
  if (my_args->m && !my_args->l)
    {
      display_minimal((u_int8_t *) &eth.src, 6, DISP_HEXCOLONS);
      display_minimal("->", 2, DISP_STRING);
      display_minimal((u_int8_t *) &eth.dst, 6, DISP_HEXCOLONS);
      display_minimal(" ", 1, DISP_STRING);
    }
  else
    if (!my_args->l)
      {
	/* announcement */
	sprintf(holder, "%s (%s)", "Ethernet header", pkt->timestamp);
	display_header_banner(holder);
	
	/* fields */
	display("Hardware source", (u_int8_t *) &eth.src, 6, 
		DISP_HEXCOLONS);
	display("Hardware destination", (u_int8_t *) &eth.dst, 6, 
		DISP_HEXCOLONS);
	sprintf(holder, "0x%x (%s)", eth.type, 
		map2str(ethernet_type_map, eth.type));
	display("Protocol", (u_int8_t *) holder, strlen(holder), 
		DISP_STRING);
	display("Length", (u_int8_t *) &pkt->media_length, 2, 
		DISP_2BYTEDEC);
      }
  
  /*
   * Check for IEEE 802 (LLC) encapsulation.  If not, assume regular ethernet
   */

  if (eth.type <= ETHERMTU) 
    {
      /* XXX do something intelligent with LLC, like this? */
      if (my_args->m)
	display_minimal((u_int8_t *) &eth.type, 2, DISP_HEX);
    }
  else 
    {
      switch (eth.type)
	{
        case ETHERNET_TYPE_IP:
	  dump_ip(pkt);
	  return;

        case ETHERNET_TYPE_IPV6:
	  dump_ipv6(pkt);
	  return;

	case ETHERNET_TYPE_ARP:
        case ETHERNET_TYPE_RARP:
	  dump_arp(pkt);
	  return;

	case ETHERNET_TYPE_IPX:
	  dump_ipx(pkt);
	  return;

        default:
	  /* If minimal mode, dump protocol type */
	  if (my_args->m)
	    display_minimal((u_int8_t *) &eth.type, 2, DISP_HEX);
	  return;	  
	}

    } /* else */

}

