/**************************************************************************** 
** File: icmp.c
**
** Author: Mike Borella
**
** Comments: Dump ICMP information
**
*****************************************************************************/

#include "global.h"
#include "protocols.h"

/*
 * Format of ICMP echo request and reply
 */

typedef struct icmp_echo
{
  u_int16_t id;
  u_int16_t seqno;
} icmp_echo_t;

/*
 * Format of ICMP mask request and reply
 */

typedef struct icmp_mask
{
  u_int16_t id;
  u_int16_t seqno;
  u_int32_t mask;
} icmp_mask_t;

/* 
 * Contains the descriptions of ICMP types 
 */

strmap_t icmp_type_map[] = 
{
  { ICMP_TYPE_ECHOREPLY ,      "echo reply" },
  { ICMP_TYPE_DESTUNREACHABLE, "destination unreachable" },
  { ICMP_TYPE_SOURCEQUENCH,    "source quench" },
  { ICMP_TYPE_REDIRECT,        "redirect" },
  { ICMP_TYPE_ECHOREQUEST,     "echo request" },
  { ICMP_TYPE_ROUTERADVERT,    "router advertisement" },
  { ICMP_TYPE_ROUTERSOLICIT,   "router solicitation" },
  { ICMP_TYPE_TIMEEXCEEDED,    "time exceeded" },
  { ICMP_TYPE_PARAMPROBLEM,    "parameter problem" },
  { ICMP_TYPE_TIMESTAMP,       "timestamp" },
  { ICMP_TYPE_TIMESTAMPREPLY,  "timestamp reply" },
  { ICMP_TYPE_INFOREQUEST,     "information request" },
  { ICMP_TYPE_INFOREPLY,       "information reply" },
  { ICMP_TYPE_MASKREQUEST,     "mask request" },
  { ICMP_TYPE_MASKREPLY,       "mask reply" },
  { ICMP_TYPE_TRACEROUTE,      "traceroute" },
  { ICMP_TYPE_CONVERSIONERROR, "datagram conversion error" },
  { 0, "" }
};

/* 
 * Contains the descriptions of ICMP destination unreachable messages
 */

strmap_t icmp_du_map[] = 
{
  { ICMP_DU_NET,               "network unreachable" },
  { ICMP_DU_HOST,              "host unreachable" },
  { ICMP_DU_PROTOCOL,          "protocol unreachable" },
  { ICMP_DU_PORT,              "port unreachable"},
  { ICMP_DU_FRAG,              "fragmentation needed"},
  { ICMP_DU_SRCRTEFAIL,        "source route failed"},
  { ICMP_DU_NETUNKNOWN,        "network unknown"},
  { ICMP_DU_HOSTUNKNOWN,       "host unknown"},
  { ICMP_DU_SRCISOLATED,       "source isolated"},
  { ICMP_DU_NETADMIN,          "network admin prohibited"},
  { ICMP_DU_HOSTADMIN,         "host admin prohibited"},
  { ICMP_DU_NETTOS,            "network unreachable for TOS"},
  { ICMP_DU_HOSTTOS,           "host unreachable for TOS"},
  { ICMP_DU_ADMIN,             "admin prohibited"},
  { ICMP_DU_HOSTPRECVIOL,      "host precedence violation"},
  { ICMP_DU_PRECCUTOFF,        "precendence cutoff"},
  { 0, "" }
};

/* 
 * Contains the descriptions of ICMP destination unreachable messages
 */

strmap_t icmp_timeexceeded_map[] = 
{
  { ICMP_TIMEEXCEEDED_TTL,               "TTL equals 0" },
  { ICMP_TIMEEXCEEDED_REASSMEBLY,        "TTL expired in reassembly"}
};

extern struct arg_t * my_args;

/*----------------------------------------------------------------------------
**
** dump_icmp_echo()
**
** Parse ICMP echo request and echo reply fields
**
**----------------------------------------------------------------------------
*/

void dump_icmp_echo(packet_t *pkt)
{
  icmp_echo_t echo;

  /*
   * Get the Echo Request/Reply fields
   */

  if (get_packet_bytes((u_int8_t *) &echo, pkt, 4) == 0)
    return;

  /*
   * Dump the ID and sequence #
   */

  display("Identifier", (u_int8_t *) &echo.id, 2, DISP_2BYTEDEC);
  display("Sequence number", (u_int8_t *) &echo.seqno, 2, DISP_2BYTEDEC);
}

/*----------------------------------------------------------------------------
**
** dump_icmp_mask()
**
** Parse ICMP mask request and reply fields
**
**----------------------------------------------------------------------------
*/

void dump_icmp_mask(packet_t *pkt)
{
  icmp_mask_t mask;

  /*
   * Get the Mask Request/Reply fields
   */

  if (get_packet_bytes((u_int8_t *) &mask, pkt, 8) == 0)
    return;

  /*
   * Dump the ID and sequence #
   */

  display("Identifier", (u_int8_t *) &mask.id, 2, DISP_2BYTEDEC);
  display("Sequence number", (u_int8_t *) &mask.seqno, 2, DISP_2BYTEDEC);
  display_ipv4("Mask", (u_int8_t *) &mask.mask);
}

/*----------------------------------------------------------------------------
**
** dump_icmp()
**
** Parse ICMP header and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_icmp(packet_t *pkt)
{
  icmp_header_t icmp;
  char holder[64];

  /*
   * Get the header
   */

  if (get_packet_bytes((u_int8_t *) &icmp, pkt, 4) == 0)
    return;

 if (my_args->m && !my_args->n)
    {
      /* In minimal mode lets just dump the type and code */
      display_minimal_string("ICMP ");
      display_minimal_string(map2str(icmp_type_map, icmp.type));
      display_minimal_string(" ");
      switch(icmp.type)
	{
	case ICMP_TYPE_DESTUNREACHABLE:
	  display_minimal_string(map2str(icmp_du_map, icmp.code));
	  break;
	  
	case ICMP_TYPE_TIMEEXCEEDED:
	  display_minimal_string(map2str(icmp_timeexceeded_map, icmp.code));
	    break;
	    
	default:
	    break;
	  }
    }
  else
    if (!my_args->n)
      { 
	/* announcement */
	display_header_banner("ICMP Header");
	
	/* Dump the type */
	sprintf(holder, "%d (%s)", icmp.type, 
		map2str(icmp_type_map, icmp.type));
	display("Type", (u_int8_t *) holder, strlen(holder), DISP_STRING);

	/* 
	 * Based on the type, dump the code, if the type has codes
	 */
	
	switch(icmp.type)
	  {
	  case ICMP_TYPE_DESTUNREACHABLE:
	    sprintf(holder, "%d (%s)", icmp.code, 
		    map2str(icmp_du_map, icmp.code));
	    display("Code", (u_int8_t *) &holder, strlen(holder), DISP_STRING);
	    break;

	  case ICMP_TYPE_TIMEEXCEEDED:
	    sprintf(holder, "%d (%s)", icmp.code, 
		    map2str(icmp_timeexceeded_map, icmp.code));
	    display("Code", (u_int8_t *) &holder, strlen(holder), DISP_STRING);
	    break;
	    
	  default:
	    display("Code", (u_int8_t *) &icmp.code, 1, DISP_1BYTEDEC);
	    break;
	  }
	
	icmp.checksum = ntohs(icmp.checksum);
	display("Checksum", (u_int8_t *) &icmp.checksum, 2, DISP_2BYTEDEC);
  
	/* 
	 * Call an ICMP-type-specific function 
	 */
	
	switch (icmp.type) 
	  {
	  case ICMP_TYPE_ECHOREPLY:
	  case ICMP_TYPE_ECHOREQUEST:
	    dump_icmp_echo(pkt);
	    return; /* bail so that we don't continue reading */
	    
	  case ICMP_TYPE_DESTUNREACHABLE:
	    /* Get the unused bytes */
	    if (get_packet_bytes((u_int8_t *) &holder, pkt, 4) == 0)
	      return;
	    break;
	    
	  case ICMP_TYPE_SOURCEQUENCH:
	    break;
	    
	  case ICMP_TYPE_REDIRECT:
	    break;
	    
	  case ICMP_TYPE_TIMEEXCEEDED:
	    /* Get the unused bytes */
	    if (get_packet_bytes((u_int8_t *) &holder, pkt, 4) == 0)
	      return;
	    break;
	    
	  case ICMP_TYPE_MASKREQUEST:
	  case ICMP_TYPE_MASKREPLY:
	    dump_icmp_mask(pkt);
	    return;
	    
	  default:
	    return;
	    
	  }
	
	/*
	 * In most ICMP packets there will be an IP header after the ICMP stuff.
	 * If it seems to be there, we can parse it out
	 */
	
	if (get_packet_bytesleft(pkt))
	  dump_ip(pkt);
      }

}

