/**************************************************************************** 
** File: ip.c
**
** Author: Mike Borella
**
** Comments: Dump IP header information
**
** $Log: ip.c,v $
** Revision 1.11  2000/06/19 16:41:25  mborella
** Made a new API display type: DISP_DEC, which replaces all decimal type
** displaying functions.  Required overcoming the usual hairy byte ordering
** issues, and I'm hoping it isn't going to screw anything up.
**
** Revision 1.10  2000/06/16 23:03:21  mborella
** Added special printing functions for ipv4 and ipv6 addresses.
**
** Revision 1.9  2000/06/01 18:36:55  mborella
** Made sure that all string maps are properly terminated, added TODO
** file, minor improvments to ICMPv6.
**
** Revision 1.8  2000/05/30 22:15:27  mborella
** Minor documentation additions.
**
*****************************************************************************/

#include "global.h"
#include "protocols.h"

/* 
 * Contains the descriptions of IP options
 */

strmap_t ip_option_map[] = 
{
  { IP_OPTION_EOL,                "end of options" },
  { IP_OPTION_NOP,                "no op" },
  { IP_OPTION_RECORDROUTE,        "record route" },
  { IP_OPTION_TIMESTAMP,          "time stamp" },
  { IP_OPTION_TRACEROUTE,         "traceroute" },
  { IP_OPTION_SECURITY,           "security" },
  { IP_OPTION_LSR,                "loose source route" },
  { IP_OPTION_EXTSECURITY,        "extra security" },
  { IP_OPTION_COMSECURITY,        "commercial security" },
  { IP_OPTION_STREAMID,           "stream ID" },
  { IP_OPTION_SSR,                "strict source route" },
  { IP_OPTION_ROUTERALERT,        "router alert" },
  { 0, "" }
};

extern struct arg_t *my_args;
extern void (*ip_proto_func[])(packet_t *);
extern strmap_t ipproto_map[];

void dump_ip_options(packet_t *, u_int8_t);

/*----------------------------------------------------------------------------
**
** dump_ip()
**
** Parse IP header and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_ip(packet_t *pkt)
{
  ip_header_t ip;
  u_int16_t frag_off;
  u_int8_t ver, hlen;
  u_int8_t u_bit, df_bit, mf_bit;
  char holder[64];

#ifdef DEBUG
  printf("\nEntering IP\n");
#endif

  /*
   * Get the IP header
   */

  if (get_packet_bytes((u_int8_t *) &ip, pkt, 20) == 0)
    return;

  /*
   * Conversions
   */

  ver = ip.version;
  hlen = ip.header_length;
  ip.offset = ntohs(ip.offset);
  ip.length = ntohs(ip.length);
  ip.id = ntohs(ip.id);
  frag_off = ip.offset & 0x1fff;
  u_bit = (ip.offset & 0x8000) >> 15;
  df_bit = (ip.offset & 0x4000) >> 14;
  mf_bit = (ip.offset & 0x2000) >> 13;
  ip.checksum = ntohs(ip.checksum);

  /*
   * Dump header
   */

  if (my_args->m && !my_args->n)
    {
      display_minimal_string("IP ");
      display_minimal_ipv4((u_int8_t *) &ip.src); 
      display_minimal_string("->");
      display_minimal_ipv4((u_int8_t *) &ip.dst); 
      display_minimal_string(" (");
      display_minimal((u_int8_t *) &ip.length, 2, DISP_DEC);
      if (df_bit)
	display_minimal_string(",DF");
      display_minimal_string(") ");
    }
  else
    if (!my_args->n)
      {
	/* announcement */
	display_header_banner("IP Header");
	
	/* print fields */
	display("Version", (u_int8_t *) &ver, 1, DISP_DEC);
	display("Header length", (u_int8_t *) &hlen, 1, DISP_DEC);
	display("TOS", (u_int8_t *) &ip.tos, 1, DISP_HEX);
	display("Total length", (u_int8_t *) &ip.length, 2, DISP_DEC);
	display("Identification", (u_int8_t *) &ip.id, 2, DISP_DEC);
	display("Fragmentation offset", (u_int8_t *) &frag_off, 2, 
		DISP_DEC);
	display("Unused bit", (u_int8_t *) &u_bit, 1, DISP_BINNLZ);
	display("Don't fragment bit", (u_int8_t *) &df_bit, 1, DISP_BINNLZ);
	display("More fragments bit", (u_int8_t *) &mf_bit, 1, DISP_BINNLZ);
	display("Time to live", (u_int8_t *) &ip.ttl, 1, DISP_DEC);
	sprintf(holder, "%d (%s)", ip.protocol, 
		map2str(ipproto_map, ip.protocol));
	display("Protocol", holder, strlen(holder), DISP_STRING);
	display("Header checksum", (u_int8_t *) &ip.checksum, 2, 
		DISP_DEC);
      
	/* IP's are not in network byte order?? */
	display_ipv4("Source address", (u_int8_t *) &ip.src); 
	display_ipv4("Destination address", (u_int8_t *) &ip.dst); 
      }

  /*
   * Check for IP options
   */

  if (hlen > 5)
    dump_ip_options(pkt, hlen * 4 - 20);

  /*
   * If this is fragment zero, hand it to the next higher
   * level protocol.
   */
  
  if ((ip.offset & 0x1fff) == 0 && ip_proto_func[ip.protocol]) 
      ip_proto_func[ip.protocol](pkt);

#ifdef DEBUG
  printf("\nLeaving IP\n");
#endif

}

/*----------------------------------------------------------------------------
**
** dump_ip_options()
**
** Dump any IP options
**
**----------------------------------------------------------------------------
*/

void dump_ip_options(packet_t *pkt, u_int8_t length)
{
  u_int8_t bytes_read = 0;
  ip_option_t opt;
  char holder[64];

  while(bytes_read < length)
    {
      /*
       * Get the first byte to make sure that we don't have a 1-byte option
       */

      if (get_packet_bytes((u_int8_t *) &opt.code, pkt, 1) == 0)
	return;
      bytes_read += 1;

      /*
       * Display it
       */
      
	sprintf(holder, "%d (%s)", opt.code, 
		map2str(ip_option_map, opt.code));
      if (!my_args->m)
	display_string("IP option code", holder);
      else
	{
	 display_minimal_string(holder);
	 display_minimal_string(" ");
	}
	
      /*
       * If it is 0, then its just a 1-byte EOL
       */

      if (opt.code == 0)
	continue;

      /*
       * It isn't, so lets read the rest...
       */

      if (get_packet_bytes((u_int8_t *) &opt.length, pkt, 1) == 0)
	return;
      bytes_read += 1;
      if (!my_args->m)
	display("  Length", (u_int8_t *) &opt.length, 1, DISP_DEC);

      /*
       * Get the rest of the option, adjusting the length by 2 to count 
       * option type and length fields...
       */
      
      if (get_packet_bytes((u_int8_t *) &holder, pkt, opt.length-2) == 0)
	return;
      bytes_read += (length-2);
      
    }
}
