/**************************************************************************** 
** File: packet_manip.c
**
** Author: Mike Borella
**
** Comments: Packet manipulation routines
**
*****************************************************************************/

#include "packet_manip.h"

/*----------------------------------------------------------------------------
**
** get_packet_bytes()
**
** Grab a specified number of bytes from a packet and deposit them in an array.
**
** NOTE: This is NOT a safe function. The caller MUST allocate enough memory 
** in the destination array.
**
** Returns 0 if operation fails and no write occurs.  Returns 1 otherwise.
**----------------------------------------------------------------------------
*/

inline int get_packet_bytes(u_int8_t *dst, packet_t *p, unsigned int n)
{
  /* 
   * First make sure that we can read the bytes without running off the end 
   * of the packet.
   */

  if (p->current + n > p->end + 1)
    return 0;
   
  /* 
   * Do the copy
   */

  memcpy(dst, p->current, n);

  /* 
   * Increment the current pointer
   */

  p->current += n;

  /* 
   * Return successfully
   */

  return 1;
}

/*----------------------------------------------------------------------------
**
** get_packet_line()
**
** Get a \n terminated line of text from the packet.
**
** NOTE: This is NOT a safe function. The caller MUST allocate enough memory 
** in the destination array.
**
** Returns 0 if operation fails and no write occurs.
** Returns number of bytes read otherwise.
**----------------------------------------------------------------------------
*/

inline int get_packet_line(u_int8_t *dst, u_int32_t max, packet_t *p)
{
  u_int8_t * ptr;

  /*
   * Find the next \n
   */
  
  ptr = p->current;
  while (ptr < p->end && (ptr - p->current) < max - 1)
    {
      if (*ptr == '\n')
	break;
      ptr++;
    }

  /*
   * If we reached the end, then there's no line to read, fail.
   */

  if (ptr >= p->end)
    return 0;

  /* 
   * Do the copy (NOTE, we do not copy the \n)
   * We add a \0 in the appropriate place
   */

  memcpy(dst, p->current, ptr - p->current);
  dst[ptr - p->current] = '\0';

  /* 
   * Increment the current pointer past the \n
   */

  p->current += (ptr - p->current + 1);

  /* 
   * Return successfully the length of the text, not including the \n
   */

  return ptr - p->current;
}

/*----------------------------------------------------------------------------
**
** look_packet_bytes()
**
** Copy specified number of bytes from a packet and deposit them in an array.
** Unlike get_packet_bytes(), we will not increment the pointer.
**
** NOTE: This is NOT a safe function. The caller MUST allocate enough memory 
** in the destination array.
**
** Returns 0 if operation fails and no write occurs.  Returns 1 otherwise.
**----------------------------------------------------------------------------
*/

inline int look_packet_bytes(u_int8_t *dst, packet_t *p, unsigned int n)
{
  /* 
   * First make sure that we can read the bytes without running off the end 
   * of the packet.
   */

  if (p->current + n > p->end + 1)
    return 0;
   
  /* 
   * Do the copy
   */

  memcpy(dst, p->current, n);

  /* 
   * Return successfully
   */

  return 1;
}

/*----------------------------------------------------------------------------
**
** skip_packet_bytes()
**
** Jump the pointer ahead the specified number of bytes
**
** Returns 0 if operation fails and no write occurs.  Returns 1 otherwise.
**----------------------------------------------------------------------------
*/

inline int skip_packet_bytes(packet_t *p, unsigned int n)
{
  /* 
   * First make sure that we can read the bytes without running off the end 
   * of the packet.
   */

  if (p->current + n > p->end + 1)
    return 0;
   
  /* 
   * Increment the current pointer
   */

  p->current += n;

  /* 
   * Return successfully
   */

  return 1;
}

/*----------------------------------------------------------------------------
**
** get_packet_bytesleft()
**
** Returns the # of bytes left to read off of the packet.
**
**----------------------------------------------------------------------------
*/

inline u_int32_t get_packet_bytesleft(packet_t *pkt)
{
  u_int32_t len;

  len = pkt->end - pkt->current;
  if (len < 0) 
    return 0;
  else 
    return len;
}

/*----------------------------------------------------------------------------
**
** set_packet_mark()
**
** Mark the current byte of the packet
**
**----------------------------------------------------------------------------
*/

inline void set_packet_mark(packet_t *pkt)
{
  pkt->mark = pkt->current;
}

/*----------------------------------------------------------------------------
**
** get_packet_markdistance()
**
** Get the distance between the current pointer and the mark
**
**----------------------------------------------------------------------------
*/

inline int32_t get_packet_markdistance(packet_t *pkt)
{
  /*
   * Sanity check the mark first
   */

  if (pkt->mark < pkt->contents || pkt->mark > pkt->end)
    return -1;

  return pkt->current - pkt->mark;
}
