/**************************************************************************** 
** File: rip.c
**
** Author: Mike Borella
**
** Comments: Dump RIP header information. 
**
** $Log: rip.c,v $
** Revision 1.9  2000/07/26 18:51:51  mborella
** Changed RIP and RIPng to print only the number of routes seen in minimal
** mode, not the routes themselves.  Major fix to ESP - it was still in 0.8.x
** mode...seesh!
**
** Revision 1.8  2000/06/29 19:04:57  mborella
** Added -Wall and -Wstrict-prototypes to CFLAGS, which nailed a bunch of
** missing include files.  Many fixes later, we're compiling cleanly again.
**
** Revision 1.7  2000/06/19 15:32:12  mborella
** Modified all files to use new address display API (not that it makes
** a bit of difference to the user...) Also tested RIPng and it seems
** to work ok.
**
** Revision 1.6  2000/06/01 18:36:55  mborella
** Made sure that all string maps are properly terminated, added TODO
** file, minor improvments to ICMPv6.
**
** Revision 1.5  2000/05/30 21:37:46  mborella
** Added dump of protocol to minimal mode.
**
** Revision 1.4  2000/05/25 19:02:53  mborella
** Fixed RIP module to differentiate between RIP v1 and v2.
**
*****************************************************************************/

#include <string.h>
#include "global.h"
#include "protocols.h"

/*
 * RIP command map
 */

strmap_t rip_command_map [] =
{
  { RIP_CMD_RQ,          "request" },
  { RIP_CMD_RP,          "reply" },
  { RIP_CMD_POLL,        "poll" },
  { RIP_CMD_POLL_ENTRY,  "poll entry" },
  { 0, "" }
};

extern struct arg_t * my_args;

/*----------------------------------------------------------------------------
**
** dump_rip()
**
** Parse RIP packet and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_rip(packet_t *pkt)
{
  ripv1_header_t       ripv1;
  ripv1_route_header_t ripv1_route;
  ripv2_header_t       ripv2;
  ripv2_route_header_t ripv2_route;
  ripv2_auth_header_t  ripv2_auth;
  char                 holder[64];
  u_int16_t            address_family;
  int route_count;


  /* 
   * Look at the version to determine whether to use v1 or v2
   */

  if (look_packet_bytes((u_int8_t *) &ripv1, pkt, sizeof(ripv1_header_t)) == 0)
    return;

  if (ripv1.version == 1)
    {
      /* RIP v1 processing here */

      /*
       * Get the RIP v1 static header
       */
      
      if (get_packet_bytes((u_int8_t *) &ripv1, pkt, sizeof(ripv1_header_t)) 
	  == 0)
	return;

      /*
       * Dump header
       */
      
      if (my_args->m)
	{
	  display_minimal_string("RIPv1 ");
	  display_minimal_string(map2str(rip_command_map, ripv1.command));
	  display_minimal_string(" ");
	}
      else
	{
	  /* announcement */
	  display_header_banner("RIPv1 Header");
	  
	  sprintf(holder, "%d (%s)", ripv1.command, 
		  map2str(rip_command_map, ripv1.command));
	  display_string("Command", holder);
	  display("Version", (u_int8_t *) &ripv1.version, 1, DISP_1BYTEDEC);
	  display("MBZ", (u_int8_t *) &ripv1.mbz, 2, DISP_2BYTEDEC);
	}
      
      /*
       * Do the individual routes.
       */
      
      route_count = 0;
      while(1)
	{
	  if (get_packet_bytes((u_int8_t *) &ripv1_route, pkt, 
			       sizeof(ripv1_route_header_t)) == 0)
	    break;
	  
	  /*
	   * Conversions
	   */
	  
	  ripv1_route.addr_fam = ntohs(ripv1_route.addr_fam);
	  ripv1_route.metric = ntohl(ripv1_route.metric);

	  /*
	   * Dump route header
	   */
	  
	  if (my_args->m)
	    {
	      /* for minimal mode, we'll just count the routes */
	      route_count++;
	    }
	  else
	    {
	      display("Address family", (u_int8_t *) &ripv1_route.addr_fam,
		      2, DISP_2BYTEDEC);
	      display("MBZ", (u_int8_t *) &ripv1_route.mbz1, 2, 
		      DISP_2BYTEDEC);
	      display_ipv4("IP address", (u_int8_t *) &ripv1_route.ipaddr);
	      display("MBZ", (u_int8_t *) &ripv1_route.mbz2, 8, 
		      DISP_HEX);
	      display("Metric", (u_int8_t *) &ripv1_route.metric, 4, 
		      DISP_4BYTEDEC);
	    }
	  
	}
      
    }
  else
    {
      /* RIP v2 processing here */
  
      /*
       * Get the RIP v2 static header
       */
      
      if (get_packet_bytes((u_int8_t *) &ripv2, pkt, sizeof(ripv2_header_t)) 
	  == 0)
	return;

      /*
       * Conversions
       */
      
      ripv2.domain = ntohs(ripv2.domain);

      /*
       * Dump header
       */
      
      if (my_args->m)
	{
	  display_minimal_string("RIPv2 ");
	  display_minimal_string(map2str(rip_command_map, ripv2.command));
	  display_minimal_string(" ");
	}
      else
	{
	  /* announcement */
	  display_header_banner("RIPv2 Header");
	  
	  sprintf(holder, "%d (%s)", ripv2.command, 
		  map2str(rip_command_map, ripv2.command));
	  display_string("Command", holder);
	  display("Version", (u_int8_t *) &ripv2.version, 1, DISP_1BYTEDEC);
	  display("Domain", (u_int8_t *) &ripv2.domain, 2, DISP_2BYTEDEC);
	}
      
      /*
       * Do the individual routes.
       * First look at the address family to determine the header type.
       */
      
      route_count = 0;
      while (1)
	{
	  if (look_packet_bytes((u_int8_t *) &address_family, pkt, 2) == 0)
	    break;;
      
	  /*
	   * If its an auth header, deal with it - otherwise its a regular
	   * route header 
	   */
	  
	  if (address_family == 0xffff)
	    {
	      if (get_packet_bytes((u_int8_t *) &ripv2_auth, pkt, 
				   sizeof(ripv2_auth_header_t)) == 0)
		return;
	      
	      /*
	       * Conversions
	       */
	      
	      ripv2_auth.addr_fam = ntohs(ripv2_auth.addr_fam);
	      ripv2_auth.type = ntohs(ripv2_auth.type);
	  
	      /*
	       * Dump auth header
	       */
	      
	      if (my_args->m)
		{
		  /* not sure what to do here... */
		}
	      else
		{
		  display("Address family", (u_int8_t *) &ripv2_auth.addr_fam, 
			  2, DISP_2BYTEDEC);
		  display("Type", (u_int8_t *) &ripv2_auth.type, 2, 
			  DISP_2BYTEDEC);
		  display_string("Password", ripv2_auth.passwd);
		}
	    }
	  else
	    {
	      if (get_packet_bytes((u_int8_t *) &ripv2_route, pkt, 
				   sizeof(ripv2_route_header_t)) == 0)
		return;
	      
	      /*
	       * Conversions
	       */
	      
	      ripv2_route.addr_fam = ntohs(ripv2_route.addr_fam);
	      ripv2_route.route_tag = ntohs(ripv2_route.route_tag);
	      ripv2_route.metric = ntohl(ripv2_route.metric);

	      /*
	       * Dump route header
	       */
	      
	      if (my_args->m)
		{
		  /* just count the routes for minimal mode */
		  route_count++;
		}
	      else
		{
		  display("Address family", (u_int8_t *) &ripv2_route.addr_fam,
			  2, DISP_2BYTEDEC);
		  display("Route tag", (u_int8_t *) &ripv2_route.route_tag, 2, 
			  DISP_2BYTEDEC);
		  display_ipv4("IP address", (u_int8_t *) &ripv2_route.ipaddr);
		  display_ipv4("Netmask", (u_int8_t *) &ripv2_route.netmask);
		  display_ipv4("Next hop", (u_int8_t *) &ripv2_route.next_hop);
		  display("Metric", (u_int8_t *) &ripv2_route.metric, 4, 
			  DISP_4BYTEDEC);
		}
	      
	    }
	} /* while */
    }
  
  /* display routes for minimal mode */
  if (my_args->m)
    {
      display_minimal((u_int8_t *) &route_count, 4, DISP_DEC);
      display_minimal_string(" routes ");
    }
}
