/**************************************************************************** 
** File: ripng.c
**
** Author: Mike Borella
**
** Comments: Dump RIPng header information. 
**
** $Log: ripng.c,v $
** Revision 1.3  2000/07/26 18:51:51  mborella
** Changed RIP and RIPng to print only the number of routes seen in minimal
** mode, not the routes themselves.  Major fix to ESP - it was still in 0.8.x
** mode...seesh!
**
** Revision 1.2  2000/06/29 19:04:57  mborella
** Added -Wall and -Wstrict-prototypes to CFLAGS, which nailed a bunch of
** missing include files.  Many fixes later, we're compiling cleanly again.
**
** Revision 1.1  2000/06/19 23:10:34  mborella
** Adding RIPng support.
**
*****************************************************************************/

#include <string.h>
#include "global.h"
#include "protocols.h"

/*
 * RIPng command map
 */

strmap_t ripng_command_map [] =
{
  { RIP_CMD_RQ,          "request" },
  { RIP_CMD_RP,          "reply" },
  { 0, "" }
};

extern struct arg_t * my_args;

/*----------------------------------------------------------------------------
**
** dump_ripng()
**
** Parse RIPng packet and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_ripng(packet_t *pkt)
{
  ripng_header_t       ripng;
  ripng_route_header_t ripng_route;
  char                 holder[64];
  int                  route_count;

  /*
   * Get the RIP v2 static header
   */
      
  if (get_packet_bytes((u_int8_t *) &ripng, pkt, sizeof(ripng_header_t)) == 0)
    return;

  /*
   * Dump header
   */
      
  if (my_args->m)
    {
      display_minimal_string("RIPng ");
      display_minimal_string(map2str(ripng_command_map, ripng.command));
      display_minimal_string(" ");
    }
      else
    {
      /* announcement */
      display_header_banner("RIPng Header");
      
      sprintf(holder, "%d (%s)", ripng.command, 
              map2str(ripng_command_map, ripng.command));
      display_string("Command", holder);
      display("Version", (u_int8_t *) &ripng.version, 1, DISP_1BYTEDEC);
      display("MBZ", (u_int8_t *) &ripng.mbz, 2, DISP_2BYTEDEC);
    }
      
  /*
   * Do the individual routes.
   */
  
  route_count = 0;
  while (1)
    {
      if (get_packet_bytes((u_int8_t *) &ripng_route, pkt, 
			   sizeof(ripng_route_header_t)) == 0)
	break;
	      
      /*
       * Conversions
       */
      
      ripng_route.route_tag = ntohs(ripng_route.route_tag);

      /*
       * Dump route header
       */
      
      if (my_args->m)
	{
	  /* just count routes in minimal mode */
	  route_count++;
	}
      else
	{
	  display_ipv6("Address", (u_int8_t *) &ripng_route.address);
	  display("Route tag", (u_int8_t *) &ripng_route.route_tag, 2, 
		  DISP_2BYTEDEC);
	  display("Netmask", (u_int8_t *) &ripng_route.netmask, 1, 
		  DISP_1BYTEDEC);
	  display("Metric", (u_int8_t *) &ripng_route.metric, 1, 
		  DISP_1BYTEDEC);
	}
      
    } /* while */

  /* display route count in minimal mode */
  if (my_args->m)
    {
      display_minimal((u_int8_t *) &route_count, 4, DISP_DEC);
      display_minimal_string(" routes ");
    }
  
}

