/**************************************************************************** 
**
** File: tcp.c
**
** Author: Mike Borella
**
** Comments: Dump TCP header information. 
**
** $Log: tcp.c,v $
** Revision 1.9  2000/06/19 16:48:24  mborella
** Modified TCP and UDP to use new decimal displaying API.
**
** Revision 1.8  2000/06/01 18:36:55  mborella
** Made sure that all string maps are properly terminated, added TODO
** file, minor improvments to ICMPv6.
**
** Revision 1.7  2000/05/30 21:37:46  mborella
** Added dump of protocol to minimal mode.
**
** Revision 1.6  2000/05/10 20:37:26  mborella
** Added HTTP support...hooohaaa! Also managed to create a very nasty
** little memory overwrite bug, which necessitated the creation of
** debug mode from configure.in.
**
** Revision 1.5  2000/05/10 18:26:50  mborella
** Added TCP option support, cleaned up minimal mode, added TCP window
** advert to minimal mode, added timestamp to all lines in minimal mode.
**
** Revision 1.4  2000/05/09 19:41:01  mborella
** Fixed minimal mode for ICMP, fixed IPv6 header printing, added the minimal
** string display mode.
**
** Revision 1.3  2000/05/09 00:21:54  mborella
** Got rid of some warnings on FreeBSD compile.
**
** Revision 1.2  2000/05/08 21:18:23  mborella
** Made "minimal mode" into a one-line tcpdump-like mode.
**
** Revision 1.1  2000/05/04 19:11:42  mborella
** Moved a bunch of source files to the src directory.
**
** Revision 1.7  2000/05/03 23:45:14  mborella
** Added display map for IP next protocol field.  Cleaned up TCP a little.
** Added more port definitions to ip_protocols.
**
** Revision 1.6  2000/04/29 00:21:07  mborella
** Port to string conversions work, but we're nowhere near complete yet.
**
*****************************************************************************/

#include <string.h>
#include "global.h"
#include "protocols.h"

/* 
 * TCP option map 
 */

strmap_t tcp_option_map [] =
{
  { TCP_OPTION_EOL,     "end of options" },
  { TCP_OPTION_NOP,     "no op" },
  { TCP_OPTION_MSS,     "maximum segment size" },
  { TCP_OPTION_WS,      "window scale" },
  { TCP_OPTION_SACKOK,  "SACK permitted" },
  { TCP_OPTION_SACK,    "SACK" },
  { TCP_OPTION_TS,      "timestamp" },
  { 192,                "cwnd" },
  { 0, "" }
};

extern struct arg_t *my_args;
extern strmap_t port_map[];

/*----------------------------------------------------------------------------
**
** dump_tcp_options()
**
** Parse TCP options 
**
**----------------------------------------------------------------------------
*/

void dump_tcp_options(packet_t *pkt, u_int8_t length)
{
  u_int8_t option_bytes_read;
  char holder[64];
  u_int8_t option_length;
  u_int8_t kind;
  u_int16_t mss;
  u_int32_t ts1, ts2;

#ifdef DEBUG
  printf("\nEntering TCP options\n");
#endif

  option_bytes_read = 0;
  while(option_bytes_read < length)
    {
      /* read the kind of option */
      if (get_packet_bytes((u_int8_t *) &kind, pkt, 1) == 0)
	return;
      option_bytes_read ++;
      
      /* take care of the 1-byte options */
      if (kind == TCP_OPTION_EOL || kind == TCP_OPTION_NOP)
	{
	  if (!my_args->m)
	    {
	      sprintf(holder, "%d (%s)", kind, 
		      map2str(tcp_option_map, kind));
	      display_string("Option", holder);
	    }
	  continue;
	}

      /* display the option's name */
      if (!my_args->m)
	{
	  sprintf(holder, "%d (%s)", kind, 
		  map2str(tcp_option_map, kind));
	  display_string("Option", holder);
	}
      else 
	{
	  display_minimal_string("<");	  
	  display_minimal_string(map2str(tcp_option_map, kind));
	}
      
      /* get and display the length field */
      if (get_packet_bytes((u_int8_t *) &option_length, pkt, 1) == 0)
	return;
      if (!my_args->m)
	display("  Length", (u_int8_t *) &option_length, 1, DISP_DEC);
      option_bytes_read += 1;

      /* read the 'length'-2 bytes because the field counts kind and itself */
      if (get_packet_bytes((u_int8_t *) holder, pkt, option_length-2) == 0)
	return;      
      option_bytes_read += (option_length - 2);
      
      /* read the field */
      switch(kind)
	{
	case TCP_OPTION_EOL:
	case TCP_OPTION_NOP:
	  /* these should be taken care of above... */
	  break;

	case TCP_OPTION_MSS:
	  memcpy((void *) &mss, holder, 2);
	  mss = ntohs(mss);
	  if (!my_args->m)
	    display("  MSS", (u_int8_t *) &mss, 2, DISP_DEC);
	  else
	    {
	      display_minimal_string(" ");
	      display_minimal((u_int8_t *) &mss, 2, DISP_DEC);
	      display_minimal_string(">");
	    }
	  break;
	  
	case TCP_OPTION_WS:
	  if (!my_args->m)
	    display("  Shift count", (u_int8_t *) holder, 1, DISP_DEC);
	  else
	    {
	      display_minimal_string(" ");
	      display_minimal((u_int8_t *) holder, 1, DISP_DEC);
	      display_minimal_string(">");
	    }
	  break;

	case TCP_OPTION_TS:
	  memcpy((void *) &ts1, holder, 4);
	  ts1 = ntohl(ts1);
	  memcpy((void *) &ts2, holder+4, 4);
	  ts2 = ntohl(ts2);
	  if (!my_args->m)
	    {
	      display("  Timestamp value", (u_int8_t *) &ts1, 4, 
		      DISP_DEC);
	      display("  Timestamp reply", (u_int8_t *) &ts2, 4, 
		      DISP_DEC);
	    }
	  else
	    {
	      display_minimal_string(" ");
	      display_minimal((u_int8_t *) &ts1, 4, DISP_DEC);
	      display_minimal_string(" ");
	      display_minimal((u_int8_t *) &ts2, 4, DISP_DEC);
	      display_minimal_string(">");
	    }
	  break;

	case 64:
	  display_minimal_string(" ");
	  memcpy((void *) &ts1, holder, 4);
	  ts1 = ntohl(ts1);
	  display_minimal((u_int8_t *) &ts1, 4, DISP_DEC);
	  display_minimal_string(">");
	  break;

	default:
	  if (my_args->m)
	    display_minimal_string(">");
	  break;

	}
    } /* while */
  
  /*
   * Trailing space for minimal mode
   */

  if (my_args->m)
    display_minimal_string(" ");

#ifdef DEBUG
  printf("\nLeaving TCP options\n");
#endif

}

/*----------------------------------------------------------------------------
**
** dump_tcp()
**
** Parse TCP header and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_tcp(packet_t *pkt)
{
  tcp_header_t tcp;
  u_int8_t hlen, unused;
  char flag_str[8];
  char holder[64];
  app_func_t app;

#ifdef DEBUG
  printf("\nEntering TCP\n");
#endif

  /*
   * Grab the static TCP header
   */

  if (get_packet_bytes((u_int8_t *) &tcp, pkt, 20) == 0)
    return;

  /*
   * Conversions
   */

  tcp.src = ntohs(tcp.src);
  tcp.dst = ntohs(tcp.dst);
  tcp.seq_number = ntohl(tcp.seq_number);
  tcp.ack_number = ntohl(tcp.ack_number);
  hlen = tcp.header_length;
  unused = tcp.unused;
  tcp.window = ntohs(tcp.window);
  tcp.checksum = ntohs(tcp.checksum);
  tcp.urgent = ntohs(tcp.urgent);

  /*
   * Prepare flag string
   */

  flag_str[0] = '\0';
  if (tcp.flags & TCP_FLAG_SYN) 
    strcat(flag_str, "S");
  if (tcp.flags & TCP_FLAG_FIN) 
    strcat(flag_str, "F");
  if (tcp.flags & TCP_FLAG_RST) 
    strcat(flag_str, "R");
  if (tcp.flags & TCP_FLAG_PSH) 
    strcat(flag_str, "P");
  if (tcp.flags & TCP_FLAG_ACK) 
    strcat(flag_str, "A");
  if (tcp.flags & TCP_FLAG_URG) 
    strcat(flag_str, "U");

  /*
   * Minimal mode
   */

  if (my_args->m)
    {
      display_minimal_string("TCP ");
      display_minimal((u_int8_t *) &tcp.src, 2, DISP_DEC);
      display_minimal_string("->");
      display_minimal((u_int8_t *) &tcp.dst, 2, DISP_DEC);
      display_minimal_string(" (");
      display_minimal(flag_str, strlen(flag_str), DISP_STRING);
      display_minimal_string(",");
      display_minimal((u_int8_t *) &tcp.seq_number, 4, DISP_DEC);      
      display_minimal_string(",");
      display_minimal((u_int8_t *) &tcp.ack_number, 4, DISP_DEC);      
      display_minimal_string(",");
      display_minimal((u_int8_t *) &tcp.window, 2, DISP_DEC);      
      display_minimal_string(") ");
    }
  else
    if (!my_args->t)
      {
	/* Dump TCP header announcement */
	display_header_banner("TCP Header");
	
	/* port fields */
	sprintf(holder, "%d (%s)", tcp.src, map2str(port_map, tcp.src));
	display("Source port", (u_int8_t *) holder, strlen(holder), DISP_STRING);
	sprintf(holder, "%d (%s)", tcp.dst, map2str(port_map, tcp.dst));
	display("Destination port", (u_int8_t *) holder, strlen(holder), 
		DISP_STRING);
	
	/* sequence and acknowledgement */
	display("Sequence number", (u_int8_t *) &tcp.seq_number, 4, 
		DISP_DEC);
	display("Acknowledgement number", (u_int8_t *) &tcp.ack_number, 4, 
		DISP_DEC);
	
	/* header length and unused bits */
	display("Header length", (u_int8_t *) &hlen, 1, DISP_DEC);
	display("Unused", (u_int8_t *) &unused, 1, DISP_DEC);
	
	/* flags */
	display("Flags", (u_int8_t *) &flag_str, strlen(flag_str), 
		DISP_STRING);
	
	/* window size, checksum and urgent ptr */
	display("Window size", (u_int8_t *) &tcp.window, 2, DISP_DEC);
	display("Checksum", (u_int8_t *) &tcp.checksum, 2, DISP_DEC);
	display("Urgent", (u_int8_t *) &tcp.urgent, 2, DISP_DEC);
      }
  
  
  /*
   * Handle any options. There should be option(s) if the header length
   * is > 20 bytes (5 words).  Pass in the total length of the options.
   */

  if (hlen > 5)
      dump_tcp_options(pkt, hlen*4 - 20);

  /*
   * If there is something left ot the packet, process the application
   * data
   */

  if (get_packet_bytesleft(pkt))
    {
      /*
       * Let's see if we can decode the application from the source port...
       */
      
      app = port2func(tcp.src);
      if (app)
	{
	  app(pkt);
	  return;
	}
      
      /*
       * Let's see if we can decode the application from the destination 
       * port...
       */
      
      app = port2func(tcp.dst);
      if (app)
	{
	  app(pkt);
	  return;
	}
      
      /*
       * Ok, let's just print the payload if it is turned on
       */
      
      if (my_args->p) 
	dump_payload(pkt);
    }

#ifdef DEBUG
  printf("\nLeaving TCP\n");
#endif

  return;
  
}

