/**************************************************************************** 
** File: arp.c
**
** Author: Mike Borella
**
** Comments: Dump ICMP header information
**
** $Id: arp.c,v 1.7 2000/08/16 21:14:05 mborella Exp $
**
*****************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include "global.h"
#include "protocols.h"

/*
 * ARP opcode map
 */

strmap_t arp_opcode_map[] =
{
  { ARP_OPCODE_REQUEST,  "ARP request" },
  { ARP_OPCODE_REPLY,    "ARP reply" },
  { RARP_OPCODE_REQUEST, "RARP request" },
  { ARP_OPCODE_REPLY,    "RARP reply" },
  { 0, "" }
};

/*
 * ARP hardware map
 */

strmap_t arp_hardware_map[] =
{
  { ARP_HWTYPE_ETHERNET,    "Ethernet" },
  { ARP_HWTYPE_IEEE802,     "IEEE 802" },
  { ARP_HWTYPE_IEEE1394,    "IEEE 1394" },
  { 0, "" }
};

extern strmap_t ethernet_type_map[];
extern struct arg_t *my_args;

/*----------------------------------------------------------------------------
**
** dump_etherarp()
**
** Parse Ethernet-specifc parts of ARP messages
**
**----------------------------------------------------------------------------
*/

void dump_etherarp(packet_t *pkt)
{
  ether_arp_t etharp;

  /*
   * Get the Ethernet section
   */

  if (get_packet_bytes((u_int8_t *) &etharp, pkt, 20) == 0)
    return;

  /*
   * Do minimal or not-so-minimal mode
   */

  if (my_args->m && !my_args->n)
    {
      display_minimal_ipv4((u_int8_t *) &etharp.target_protoaddr);
    }
  else
    if (!my_args->n)
      {
	display("Sender Ether address", (u_int8_t *) &etharp.sender_hwaddr, 6, 
		DISP_HEXCOLONS);
	display_ipv4("Sender IP address",
		     (u_int8_t *) &etharp.sender_protoaddr); 
	display("Target Ether address", (u_int8_t *) &etharp.target_hwaddr, 6, 
		DISP_HEXCOLONS);
	display_ipv4("Target IP address",
		     (u_int8_t *) &etharp.target_protoaddr); 
      }
}

/*----------------------------------------------------------------------------
**
** dump_arp()
**
** Parse ARP header and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_arp(packet_t *pkt)
{
  arp_header_t arp;
  char holder[64];

  /*
   * Get the ARP fixed header
   */

  if (get_packet_bytes((u_int8_t *) &arp, pkt, 8) == 0)
    return;

  /*
   * Conversions
   */

  arp.hwaddr_format = ntohs(arp.hwaddr_format);
  arp.protoaddr_format = ntohs(arp.protoaddr_format);
  arp.opcode = ntohs(arp.opcode);

  /* 
   * Check for minimal mode first
   */

  if (my_args->m && !my_args->n)
    {
      switch(arp.opcode)
	{
	case ARP_OPCODE_REQUEST:
	case ARP_OPCODE_REPLY:
	  display_minimal_string(map2str(arp_opcode_map, arp.opcode));
	  display_minimal_string(" ");
	  break;
	}
    }
  else
    if (!my_args->n)
      {
	/* 
	 * Banner announcement 
	 */
	
	display_header_banner("ARP Header");
	
	/*
	 * Dump hardware type
	 */
	
	sprintf(holder, "0x%x (%s)", arp.hwaddr_format, 
		map2str(arp_hardware_map, arp.hwaddr_format));
	display("Hardware type", (u_int8_t *) holder, strlen(holder), 
		DISP_STRING);
	
	/*
	 * Dump protocol type
	 */
	
	sprintf(holder, "0x%x (%s)", arp.protoaddr_format, 
		map2str(ethernet_type_map, arp.protoaddr_format));
	display("Protocol type", (u_int8_t *) holder, strlen(holder), 
		DISP_STRING);
	
	/* 
	 * Dump hardware and protocol lengths
	 */
	
	display("Hardware length", (u_int8_t *) &arp.hwaddr_length, 1, 
		DISP_1BYTEDEC);
	display("Protocol length", (u_int8_t *) &arp.protoaddr_length, 1, 
		DISP_1BYTEDEC);
	
	/*
	 * Now do the opcode
	 */
	
	sprintf(holder, "%d (%s)", arp.opcode, 
		map2str(arp_opcode_map, arp.opcode));
	display("Opcode", holder, strlen(holder), DISP_STRING);
      }
  
  /*
   * Check hardware type and layer 3 address format
   */
  
  switch(arp.hwaddr_format)
    {
    case ARP_HWTYPE_ETHERNET: 
      
      switch(arp.protoaddr_format)
	{
	case ETHERNET_TYPE_IP:
	  dump_etherarp(pkt);
	  break;
	  
	default:
	  break;
	} /* inner switch */

    default:

    } /* outer switch */


  /* dump the hex buffer */
  hexbuffer_flush();

}



