/**************************************************************************** 
** File: ethernet.c
**
** Author: Mike Borella
**
** Comments: Dump ethernet packets
**
** $Id: ethernet.c,v 1.12 2000/08/18 17:00:10 mborella Exp $
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU Library General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "global.h"
#include "protocols.h"

extern struct arg_t *my_args;

/*
 * Ethernet type map
 */

strmap_t ethernet_type_map[] =
{
  { ETHERNET_TYPE_IP,   "IP" },
  { ETHERNET_TYPE_ARP,  "ARP" },
  { ETHERNET_TYPE_RARP, "RARP" },
  { ETHERNET_TYPE_IPX,  "IPX" },
  { ETHERNET_TYPE_IPV6, "IPV6" },
  { 0, "" }
};

/*----------------------------------------------------------------------------
**
** dump_ethernet()
**
** Process packets from the DLT_EN10MB interface type
**
**----------------------------------------------------------------------------
*/

void dump_ethernet(packet_t *pkt)
{
  ether_header_t eth;
  char holder[64];

  /*
   * Read the header
   */

  if (get_packet_bytes((u_int8_t *) &eth, pkt, 14) == 0)
    return;

  /*
   * Dump header
   */
  if (my_args->m && !my_args->l)
    {
      display_minimal((u_int8_t *) &eth.src, 6, DISP_HEXCOLONS);
      display_minimal_string("->");
      display_minimal((u_int8_t *) &eth.dst, 6, DISP_HEXCOLONS);
      display_minimal_string(" ");
    }
  else
    if (!my_args->l)
      {
	/* announcement */
	sprintf(holder, "%s (%s)", "Ethernet header", pkt->timestamp);
	display_header_banner(holder);
	
	/* fields */
	display("Hardware source", (u_int8_t *) &eth.src, 6, 
		DISP_HEXCOLONS);
	display("Hardware destination", (u_int8_t *) &eth.dst, 6, 
		DISP_HEXCOLONS);
	sprintf(holder, "0x%x (%s)", ntohs(eth.type), 
		map2str(ethernet_type_map, ntohs(eth.type)));
	display("Protocol", (u_int8_t *) holder, strlen(holder), 
		DISP_STRING);
	display("Length", (u_int8_t *) &pkt->media_length, 2, 
		DISP_2BYTEDEC);

      }
  
  /* dump the hex buffer */
  if (!my_args->l)  
    hexbuffer_flush();
  
  /*
   * Check for IEEE 802 (LLC) encapsulation.  If not, assume regular ethernet
   */

  if (ntohs(eth.type) <= ETHERMTU) 
    {
      /* Do something intelligent with LLC */
      if (my_args->m  && !my_args->l)
	{
	  display_minimal_string("(");
	  display_minimal((u_int8_t *) &eth.type, 2, DISP_HEX);
	  display_minimal_string(") ");
	}
      dump_llc(pkt);
    }
  else 
    {
      switch (ntohs(eth.type))
	{
        case ETHERNET_TYPE_IP:
	  dump_ip(pkt);
	  return;

        case ETHERNET_TYPE_IPV6:
	  dump_ipv6(pkt);
	  return;

	case ETHERNET_TYPE_ARP:
        case ETHERNET_TYPE_RARP:
	  dump_arp(pkt);
	  return;

	case ETHERNET_TYPE_IPX:
	  dump_ipx(pkt);
	  return;

        default:
	  /* If minimal mode, dump protocol type */
	  if (my_args->m  && !my_args->l)
	    {
	      display_minimal_string("(");
	      display_minimal((u_int8_t *) &eth.type, 2, DISP_HEX);
	      display_minimal_string(") ");
	    }
	  break;
	}

    } /* else */

}

