/**************************************************************************** 
** File: ip_protocols.c
**
** Author: Mike Borella
**
** Comments: Functions to handle certain IP protocol numbers
**
*****************************************************************************/

#include <stdio.h>
#include "global.h"
#include "protocols.h"
#include "ip_protocols.h"

/*
 * Make an array that will hold all of the pointers to protocol functions 
 */

void (*ip_proto_func[256])(packet_t *);

/*
 * Fill the array up with pointers.  I'd like to do this in global space
 * so that this function does not have to be explicitly called...but I can't
 * get that to work right (won't compile).  Help?
 */

void init_ip_protocols(void)
{
  int i;

  for (i = 0; i < 256; i++)
    ip_proto_func[i] = NULL;

  ip_proto_func[PROTO_ICMP]     = dump_icmp;
  ip_proto_func[PROTO_IGMP]     = dump_igmp;
  ip_proto_func[PROTO_IPENCAP]  = dump_ip;
  ip_proto_func[PROTO_TCP]      = dump_tcp;
  ip_proto_func[PROTO_UDP]      = dump_udp;
  ip_proto_func[PROTO_ESP]      = dump_esp;
  ip_proto_func[PROTO_AH]       = dump_ah;
  ip_proto_func[PROTO_OSPF]     = dump_ospf;
  ip_proto_func[PROTO_IPV6ICMP] = dump_icmpv6;
  ip_proto_func[PROTO_IPV6]     = dump_ipv6;
}

/*
 * IP protocol map.  There appears to be 3 different types of IP in IP 
 * encapsulation.  Weird.  I'm not sure which is(are) really used...
 */

strmap_t ipproto_map[] =
{
  { PROTO_IPV6HOP,         "IPv6 hop by hop" },
  { PROTO_ICMP,            "ICMP" },
  { PROTO_IGMP,            "IGMP" },
  { PROTO_GGP,             "GGP" },
  { PROTO_IPENCAP,         "IP encapsulation" },
  { PROTO_ST,              "ST" },
  { PROTO_TCP,             "TCP" },
  { PROTO_CBT,             "CBT" },
  { PROTO_EGP,             "EGP" },
  { PROTO_IGP,             "IGP" },
  { PROTO_PUP,             "PUP" },
  { PROTO_UDP,             "UDP" },
  { PROTO_HMP,             "HMP" },
  { PROTO_XNSIDP,          "XNS IDP" },
  { PROTO_RDP,             "RDP" },
  { PROTO_IPV6,            "IPv6" },
  { PROTO_IPV6ROUTE,       "IPv6 route" },
  { PROTO_IPV6FRAG,        "IPv6 fragmentation" },
  { PROTO_IDRP,            "IDRP" },
  { PROTO_RSVP,            "RSVP" },
  { PROTO_ESP,             "ESP" },
  { PROTO_AH,              "AH" },
  { PROTO_NARP,            "NARP" },
  { PROTO_IPV6ICMP,        "ICMPv6" },
  { PROTO_IPV6NONEXT,      "IPv6 no next header" },
  { PROTO_IPV6OPTS,        "IPv6 options" },
  { PROTO_RSPF,            "Radio shortest path first" },
  { PROTO_VMTP,            "Versatile message transport" },
  { PROTO_OSPF,            "OSPF" },
  { PROTO_IPIP,            "IPinIP" },
  { PROTO_ENCAP,           "IPinIP encapsulation" },
  { 0, "" }
};

/*
 * We'll put the port map here even though it is used by TCP and UDP.
 * BTW, we can call these anything we'd like.  Suggestions welcome :-)
 */

strmap_t port_map[] = 
{
  { PORT_TCPMUX,           "TCP multiplexing" },
  { PORT_ECHO,             "echo" },
  { PORT_DISCARD,          "discard" },
  { PORT_SYSTAT,           "systat" },
  { PORT_DAYTIME,          "daytime" },
  { PORT_NETSTAT,          "netstat" },
  { PORT_QOTD,             "quote of the day" },
  { PORT_MSP,              "MSP" },
  { PORT_CHARGEN,          "character generator" },
  { PORT_FTPDATA,          "FTP data" },
  { PORT_FTPCTRL,          "FTP control" },
  { PORT_SSH,              "secure shell" },
  { PORT_TELNET,           "telnet" },
  { PORT_SMTP,             "SMTP" },
  { PORT_TIME,             "time" },
  { PORT_RLP,              "RLP" },
  { PORT_NAMESERVER,       "name server" },
  { PORT_WHOIS,            "whois" },
  { PORT_REMOTEMAILCHECK,  "remote mail checking" },
  { PORT_DNS,              "DNS" },
  { PORT_MTP,              "MTP" },
  { PORT_DHCPSERVER,       "DHCP/BOOTP server" },
  { PORT_DHCPCLIENT,       "DHCP/BOOTP client" },
  { PORT_TFTP,             "TFTP" },
  { PORT_GOPHER,           "gopher" },
  { PORT_RJE,              "RJE" },
  { PORT_FINGER,           "finger" },
  { PORT_HTTP,             "http" },
  { PORT_TTYLINK,          "ttylink" },
  { PORT_KERBEROS,         "Kerberos" },
  { PORT_SUPDUP,           "SUPDUP" },
  { PORT_HOSTNAMES,        "hostnames" },
  { PORT_ISOTSAP,          "ISO TSAP" },
  { PORT_CSNETNAMESERVICE, "CSNET name service" },
  { PORT_EUDORA,           "Eudora" },
  { PORT_RTELNET,          "rtelnet" },
  { PORT_POP2,             "POP2" },
  { PORT_POP3,             "POP3" },
  { PORT_SUNRPC,           "sunrpc" },
  { PORT_AUTH,             "auth" },
  { PORT_SNMP,             "SNMP" },

  { PORT_NETBIOSNS,        "NETBIOS name service" },
  { PORT_NETBIOSDGM,       "NETBIOS datagram service" },
  { PORT_NETBIOSSSN,       "NETBIOS session service" },

  { PORT_SLP,              "SLP" },

  { PORT_RIP,              "RIP" },
  { PORT_RIPNG,            "RIPng" },

  { PORT_SIP,              "SIP" },
  { 0, "" }
};


/*----------------------------------------------------------------------------
**
** port2func(u_int16_t)
**
** Map a port number to a function to call
**
**----------------------------------------------------------------------------
*/

app_func_t port2func(u_int16_t port)
{
  app_func_t f;

  switch(port)
    {
    case PORT_SIP:
      f = dump_sip;
      break;
      
    case PORT_DNS:
      f = dump_dns;
      break;

    case PORT_DHCPCLIENT:
    case PORT_DHCPSERVER:
      f = dump_dhcp;
      break;

    case PORT_HTTP:
      f = dump_http;
      break;

    case PORT_NETBIOSNS:
      f = dump_netbios_ns;
      break;

    case PORT_SNMP:
      f = dump_snmp;
      break;

    case PORT_SLP:
      f = dump_slp;
      break;

    case PORT_RIP:
      f = dump_rip;
      break;

    case PORT_RIPNG:
      f = dump_ripng;
      break;

    default:
      f = NULL;
      break;
    }

  return f;
}
