/**************************************************************************** 
**
** File: ipgrab.c
**
** Author: Mike Borella
**
** Comments: Sniffs all packets on the link and dumps the fields of
**	the data link, IP, TCP, and UDP headers.  
**
** $Log: ipgrab.c,v $
** Revision 1.6  2000/08/16 20:26:05  mborella
** Added basic hexbuffer support.  Tested with Ethernet, will add to other
** protocols.
**
** Revision 1.5  2000/05/21 15:26:41  mborella
** Added OSPF shim.  Doesn't actually work yet.  Also added autoconf hack
** that will look in several places for pcap.h.  Finally, added a few new
** IP protocols.
**
** Revision 1.4  2000/05/12 21:56:49  mborella
** Got rid of more build and dependency problems...
**
*****************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <pcap.h>
#include "global.h"
#include "datalink.h"
#include "ipgrab.h"
#include "open_pcap.h"
#include "protocols.h"
#include "error.h"

char    *pcap_cmd;         /* command string for pcap */
pcap_t  *pd;               /* pcap device descriptor */
int     cnt;               /* number of packets to read */
pcap_handler dev_prcsr;    /* ptr to func that processes packet for a device */
struct arg_t *my_args;     /* Command line arguments */

RETSIGTYPE do_stats(void);

/*----------------------------------------------------------------------------
 *
 * main()
 *
 *----------------------------------------------------------------------------
 */

int main(int argc, char *argv[])
{
  u_int8_t * userdata;
  int link;

  /*
   * Initiatilize the IP family module
   */

  init_ip_protocols();

  /*
   * Clear packet count
   */

  cnt = -1;

  /*
   * Parse command line for options
   */

  my_args = Cmdline(argc, argv);
  if (my_args->c) 
    cnt = my_args->c;

  /*
   * Make stdout buffered, if necessary
   */
  
  if (my_args->b)
#ifdef HAVE_SETLINEBUF
    setlinebuf(stdout);
#else
  setvbuf(stdout, NULL, _IOLBF, 0);
#endif

  
  /* 
   * Copy filter command into a string only if there is a command
   */
  
  if (my_args->optind)
    pcap_cmd = copy_argv(&argv[my_args->optind]);

  /*
   * Open the pcap device for sniffing
   */

  link = open_pcap();

  /*
   * Get rid of root privs
   */

  setuid(getuid());

  /*
   * Print intro stuff to stderr so output files have consistent 
   * format
   */

  fprintf(stderr, "%s %s\n", PACKAGE, VERSION);
  fprintf(stderr, "Listening on device %s ", my_args->i);
  switch(link)
    {
    case DLT_NULL:
      fprintf(stderr,"(loopback)\n");
      break;
    case DLT_EN10MB:
      fprintf(stderr, "(ethernet)\n");
      break;
    case DLT_SLIP:
      fprintf(stderr, "(slip)\n");
      break;
#ifdef DLT_RAW /* Not supported in some arch or older pcap versions */
    case DLT_RAW:
      fprintf(stderr, "(raw)\n");
      break;
#endif
    default:
      GWF_error_fatal("\n%s cannot handle data link type %d", argv[0], link);
    }

  /* 
   * Put the link type into a string
   */
  
  userdata = malloc(4);
  userdata = (u_char *) &link;

  /*
   * Set the signals so that we can clean up when ctrl-C is pressed
   */
        
  signal(SIGTERM, (sighandler_t) do_stats);
  signal(SIGINT, (sighandler_t) do_stats);

  /*
   * Read all packets on the device.  Continue until cnt packets read 
   */
  
  if (pcap_loop(pd, cnt, (pcap_func_t) datalink, userdata) < 0)
    GWF_error_fatal("pcap_loop: %s", pcap_geterr(pd));
  
  /*
   * Close the pcap device
   */

  pcap_close(pd);

  exit(0);
}


/*----------------------------------------------------------------------------
 *
 * do_stats()
 *
 * Collect PCAP statistics
 *
 *----------------------------------------------------------------------------
 */

RETSIGTYPE do_stats(void)
{
  struct pcap_stat stats;

  if (pcap_stats(pd, &stats) < 0)
    fprintf(stderr, "pcap_stats: %s\n", pcap_geterr(pd));
  else 
    {
      fprintf(stderr, "%d packets received\n", stats.ps_recv);
      fprintf(stderr, "%d packets dropped by kernel\n", stats.ps_drop);
    }
  exit(1);
}

/*----------------------------------------------------------------------------
 *
 * copy_argv()
 *
 * Copy arg vector into a new buffer, concatenating arguments with spaces.
 * Lifted from tcpdump.
 *
 *----------------------------------------------------------------------------
 */

char *copy_argv(char **argv)
{
  char **p;
  u_int len = 0;
  char *buf;
  char *src, *dst;
  
  p = argv;
  if (*p == 0) return 0;

  while (*p)
    len += strlen(*p++) + 1;

  buf = (char *) malloc (len);
  if (buf == NULL) 
    GWF_error_system("copy_argv: malloc() failed");
      
  p = argv;
  dst = buf;
  while ((src = *p++) != NULL) 
    {
      while ((*dst++ = *src++) != '\0');
      dst[-1] = ' ';
    }
  dst[-1] = '\0';
  
  return buf;
}
