/**************************************************************************** 
** File: chap.c
**
** Author: Mike Borella
**
** Comments: CHAP decoding.
**
** $Id: chap.c,v 1.2 2000/09/01 22:23:00 mborella Exp $
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU Library General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "chap.h"

/*
 * CHAP header
 */

typedef struct chap
{
  u_int8_t  code;
  u_int8_t  identifier;
  u_int16_t length;
} chap_t;

/*
 * CHAP codes
 */

#define CHAP_CODE_CHALLENGE   1
#define CHAP_CODE_RESPONSE    2
#define CHAP_CODE_SUCCESS     3
#define CHAP_CODE_FAILURE     4

/*
 * CHAP code map
 */

strmap_t chap_code_map[] =
{
  { CHAP_CODE_CHALLENGE,      "challenge" },
  { CHAP_CODE_RESPONSE,       "response" },
  { CHAP_CODE_SUCCESS,        "success" },
  { CHAP_CODE_FAILURE,        "failure" },
  { 0, "" }
};

extern struct arg_t * my_args;

/*----------------------------------------------------------------------------
**
** dump_chap()
**
** Dump CHAP headers and payload.
**
**----------------------------------------------------------------------------
*/

void dump_chap(packet_t *pkt)
{
  chap_t chap;
  
  /*
   * Get the header
   */
  
  if (get_packet_bytes((u_int8_t *) &chap, pkt, sizeof(chap_t)) == 0)
    return;
      
  /*
   * Conversion
   */
  
  chap.length = ntohs(chap.length);
      
  /*
   * Dump the header
   */
  
  if (!my_args->l)
    {
      if (my_args->m)
	{
	  display_minimal_string("CHAP ");
	}
      else
	{
	  display_header_banner("CHAP Header");
	  display_strmap("Code", chap.code, chap_code_map);
	  display("Identifier", &chap.identifier, 1, DISP_DEC);
	  display("Length", (u_int8_t *) &chap.length, 2, DISP_DEC);
	}
    }

  /*
   * Get the CHAP payload 
   */

  switch(chap.code)
    {
    case CHAP_CODE_CHALLENGE:
    case CHAP_CODE_RESPONSE:
      {
	u_int8_t * value;
	u_int8_t   value_size;
	u_int8_t * name;
	u_int16_t  name_size;

	/* Get the value size */
	if (get_packet_bytes(&value_size, pkt, 1) == 0)
	  return;
	
	/* Get the value */
	value = (u_int8_t *) malloc(value_size);
	if (get_packet_bytes(value, pkt, value_size) == 0)
	  return;

	/* Figure out the size of the name and get it */
	name_size = chap.length - sizeof(chap_t) - value_size - 1;
	name = (u_int8_t *) malloc(name_size+1);
	if (get_packet_bytes(name, pkt, name_size) == 0)
	  return;
	name[name_size] = '\0';

	/* display everything */
	if (!my_args->l)
	  {
	    if (my_args->m)
	      {
		display_minimal_string(name);
	      }
	    else
	      {
		display("Value size", &value_size, 1, DISP_DEC);
		display("Value", value, value_size, DISP_HEX);
		display_string("Name", name);
	      }
	  } /* if */
      }
      break;

    case CHAP_CODE_SUCCESS:
    case CHAP_CODE_FAILURE:
      {
	u_int8_t * message;
	u_int8_t   message_len;

	/* figure out length of the message and then grab it */
	message_len = chap.length - sizeof(chap_t);
	message = (u_int8_t *) malloc (message_len + 1);
	if (get_packet_bytes(message, pkt, message_len) == 0)
	  return;
	message[message_len] = '\0';

	/* display it */
	if (!my_args->l)
	  {
	    if (my_args->m)
	      {
		display_minimal_string(message);
	      }
	    else
	      {
		display_string("Message", message);
	      }
	  } /* if */	
      }
      break;

    }

}
