/**************************************************************************** 
** File: datalink.c
**
** Author: Mike Borella
**
** Comments: Generic datalink module
**
** $Id: datalink.c,v 1.10 2000/09/27 19:27:29 mborella Exp $
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU Library General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "global.h"
#include "error.h"
#include "datalink.h"
#include "ethernet.h"
#include "loopback.h"
#include "slip.h"
#include "raw.h"
#include "ppp.h"

extern struct arg_t *my_args;
extern int packet_displayed;

/*----------------------------------------------------------------------------
**
** datalink()
**
**----------------------------------------------------------------------------
*/

void datalink(u_char *user, const struct pcap_pkthdr *h, u_char *pkt)
{
  packet_t packet;
  u_int16_t len;
  u_int32_t *link;

  /*
   * For minimal mode.  We haven't displayed part of a packet yet...
   */

  packet_displayed = 0;

  /*
   * Determine real length of the data that we received
   */

  len = h->caplen;

  /*
   * Set up packet into data structure
   */

  packet.contents = malloc(len+1);
  memcpy(packet.contents, pkt, len);
  packet.current = &packet.contents[0];
  packet.end = &packet.contents[0] + len;
  packet.media_length = h->len;
  sprintf(packet.timestamp, "%u.%06u", (u_int32_t) h->ts.tv_sec, 
	  (u_int32_t) h->ts.tv_usec);

  /*
   * If we're in minimal mode, start off with the packet timestamp
   */

  if (my_args->m)
    {
      display_minimal_string(packet.timestamp);
      display_minimal_string(" ");
    }

  /* 
   * Delete any remaining hex buffer contents
   */

  hexbuffer_kill();

  /*
   * Determine what to do next
   */

  link = (u_int32_t *) user;
  switch(*link)
    {
    case DLT_NULL:
      dump_loopback(&packet);
      break;

    case DLT_EN10MB:
      dump_ethernet(&packet);
      break;

    case DLT_SLIP:
      dump_slip(&packet);
      break;

    case DLT_PPP:
      dump_ppp(&packet);
      break;

#ifdef DLT_RAW /* Not supported in some arch or older pcap versions */
    case DLT_RAW:
      dump_raw(&packet);
      break;
#endif
    default:
      error_fatal("\ncannot handle data link type %d", link);
    }

  /*
   * Deallocate memory
   */

  free(packet.contents);

  /*
   * For minimal mode, finish with a carriage return
   */

  if (my_args->m && packet_displayed)
    {
      display_minimal_cr();
    }

}
