/**************************************************************************** 
** File: ppp.c
**
** Author: Mike Borella
**
** Comments: PPP module.
**
** $Id: ppp.c,v 1.6 2000/09/01 22:28:19 mborella Exp $
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU Library General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "ppp.h"

/*
 * PPP protocol types
 */

#define PPP_PROTOCOL_IP      0x0021
#define PPP_PROTOCOL_IPCP    0x8021
#define PPP_PROTOCOL_IPXCP   0x802b
#define PPP_PROTOCOL_LCP     0xc021
#define PPP_PROTOCOL_PAP     0xc023
#define PPP_PROTOCOL_CHAP    0xc223
#define PPP_PROTOCOL_CCP     0x80fd

/*
 * PPP protocol map
 */

strmap_t ppp_protocol_map[] =
{
  { PPP_PROTOCOL_IP,      "IP" },
  { PPP_PROTOCOL_IPCP,    "IPCP" },
  { PPP_PROTOCOL_IPXCP,   "IPXCP" },
  { PPP_PROTOCOL_LCP,     "LCP" },
  { PPP_PROTOCOL_PAP,     "PAP" },
  { PPP_PROTOCOL_CHAP,    "CHAP" },
  { PPP_PROTOCOL_CCP,     "CCP" },
  { 0, "" }
};

/*
 * PPP frame format
 */

typedef struct ppp
{
  u_int8_t  address;
  u_int8_t  control;
  u_int16_t protocol;
} ppp_t;

extern struct arg_t * my_args;

/*----------------------------------------------------------------------------
**
** dump_ppp()
**
** Displays PPP packets.
**
**----------------------------------------------------------------------------
*/

void dump_ppp(packet_t *pkt)
{
  ppp_t ppp;
  u_int16_t protocol;
  u_int16_t two_bytes;

  /*
   * Look at the first 2 bytes to see if address/control compression is 
   * set or not.
   */

  if (look_packet_bytes((u_int8_t *) &two_bytes, pkt, sizeof(u_int16_t)) == 0)
    return;

  /*
   * If the result is 0xff03, then process the packet normally.  Anything else
   * and we'll assume that compression is being used and that these two
   * bytes are a protocol type.
   */

  if (ntohs(two_bytes) == 0xff03)
    {
      
      /*
       * Get the header
       */
      
      if (get_packet_bytes((u_int8_t *) &ppp, pkt, sizeof(ppp_t)) == 0)
	return;
      
      /*
       * Conversion
       */
      
      protocol = ntohs(ppp.protocol);
      
      /*
       * Dump the header
       */
      
      if (!my_args->l)
	{
	  if (my_args->m)
	    {
	      display_minimal_string("PPP ");
	    }
	  else
	    {
	      display_header_banner("PPP Header");
	      display("Address", &ppp.address, 1, DISP_HEX);
	      display("Control", &ppp.control, 1, DISP_HEX);
	      display_strmap_hex("Protocol type", protocol, 
				 ppp_protocol_map);
	    }
	  
	}
    }
  else
    {
      /*
       * Get the protocol
       */
      
      if (get_packet_bytes((u_int8_t *) &protocol, pkt, 2) == 0)
	return;
      
      /*
       * Conversion
       */
      
      protocol = ntohs(protocol);
      
      /*
       * Dump the header
       */
      
      if (!my_args->l)
	{
	  if (my_args->m)
	    {
	      display_minimal_string("PPP ");
	    }
	  else
	    {
	      display_header_banner("PPP Header");
	      display_strmap_hex("Protocol type", protocol, ppp_protocol_map);
	    }
	}
    } /* else */

  /* dump the hex buffer */
  if (!my_args->l)
    hexbuffer_flush();
  
  /*
   * Determine the next layer protocol
   */

  switch(protocol)
    {
    case PPP_PROTOCOL_IP:
      dump_ip(pkt);
      break;

    case PPP_PROTOCOL_LCP:
      dump_lcp(pkt);
      break;

    case PPP_PROTOCOL_CHAP:
      dump_chap(pkt);
      break;

    case PPP_PROTOCOL_IPCP:
      break;

    default:
      /* dump data here! */
      break;
    }

  /* Should be able to display the PPP trailer here.... */
  
}
