/**************************************************************************** 
** File: utilities.c
**
** Author: Mike Borella
**
** Comments: General utility functions that are not tied to IPgrab in any
** particular way.
**
** $Id: utilities.c,v 1.4 2000/10/17 16:35:37 mborella Exp $
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU Library General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "utilities.h"
#include "error.h"

/*----------------------------------------------------------------------------
**
** reverse_byte_order()
**
** Reverses the byte order of an arbitrarily long byte string
**
**----------------------------------------------------------------------------
*/

inline void reverse_byte_order(u_int8_t *s, int len)
{
  u_int8_t * temp;
  int i;

  temp = (u_int8_t *) my_malloc(len);
  for (i = 0; i < len; i++)
      temp[len-i-1] = s[i];
  memcpy(s, temp, len);
  my_free(temp);
}

/*----------------------------------------------------------------------------
**
** isspace_str(char *s, int len)
**
** Returns 1 if the first len bytes of s are whitespace, 0 otherwise.
**
**----------------------------------------------------------------------------
*/

inline int isspace_str(char *s, int len)
{
  int i;

  for (i = 0; i < len; i++)
    {
      if (!isspace(s[i]))
	return 0;
    }
  return 1;
}

/*----------------------------------------------------------------------------
**
** isprint_str(char *s, int len)
**
** Returns 1 if the first len bytes of s are printable, 0 otherwise.
**
**----------------------------------------------------------------------------
*/

inline int isprint_str(char *s, int len)
{
  int i;

  for (i = 0; i < len; i++)
    {
      if (!(isprint(s[i]) || isspace(s[i])))
	return 0;
    }
  return 1;
}

/*----------------------------------------------------------------------------
**
** argv2str(char **)
**
** Converts an argv[][] style array of strings into a single string, with
** the original string separated by spaces
**
**----------------------------------------------------------------------------
*/

inline char * argv2str(char **argv)
{
  char **p;
  u_int len = 0;
  char *buf;
  char *src, *dst;
  
  p = argv;
  if (*p == 0) return 0;

  while (*p)
    len += strlen(*p++) + 1;

  buf = (char *) my_malloc (len);
      
  p = argv;
  dst = buf;
  while ((src = *p++) != NULL) 
    {
      while ((*dst++ = *src++) != '\0');
      dst[-1] = ' ';
    }
  dst[-1] = '\0';
  
  return buf;  
}

/*----------------------------------------------------------------------------
**
** my_malloc()
**
** Performs a malloc but tests for error conditions.
**
**----------------------------------------------------------------------------
*/

inline void * my_malloc(size_t n)
{
  void * p;

  /* make sure that we always allocate something in the zero case */
  if (n == 0)
    n = 1;

  /* do it */
  p = malloc(n);
  if (p == NULL)
    error_fatal("can't allocate memory");

  return p;
}

/*----------------------------------------------------------------------------
**
** my_free()
**
** Frees memory safely by checking for the viability of the pointer
**
**----------------------------------------------------------------------------
*/

inline void my_free(void * p)
{
  if (p)
    free(p);
}
