/**************************************************************************** 
**
** File: dynports.c
**
** Author: Mike Borella
**
** Comments: Support for dynamic port mapping.  This allows a protocol to
** to listened for on a non-standard port.
**
** $Id: dynports.c,v 1.3 2001/10/07 19:17:40 mborella Exp $
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU Library General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "global.h"
#include "error.h"
#include "dynports.h"

#define DYNPORTS_NUM 16 /* max number of dynamic ports on the system */

typedef struct portmap
{
  u_int16_t      port;
  service_func_t f;
} portmap_t;

portmap_t dynport_map[DYNPORTS_NUM];
int       current_ptr = 0;

/*----------------------------------------------------------------------------
**
** dynports_add()
**
** Adds a dynamic port mapping
** 
**----------------------------------------------------------------------------
*/

void dynports_add(u_int16_t port, service_func_t f)
{
  if (current_ptr >= DYNPORTS_NUM)
    error_fatal("too many dynamic port mappings");
  
  dynport_map[current_ptr].port = port;
  dynport_map[current_ptr].f = f;

  current_ptr++;
}

/*----------------------------------------------------------------------------
**
** dynports_parse()
**
** Parses a string of the form 'protocol=port', such as rtp=12341.
** 
**----------------------------------------------------------------------------
*/

void dynports_parse(char *s)
{
  char           * ptr;
  u_int16_t        port;
  service_func_t   f = NULL;

  /* find = */
  ptr = strchr(s,'=');
  if (ptr == NULL)
    error_system("unable to parse dynamic port assignment %s", s);

  /* determine port # */
  port = (u_int16_t) atoi(ptr+1);
  
  /* determine protocol */
  *ptr = '\0';
  f = NULL;
  if (!strcmp(s, "dhcp"))
    f = dump_dhcp;
  if (!strcmp(s, "dns"))
    f = dump_dns;
  if (!strcmp(s, "http"))
    f = dump_http;
  if (!strcmp(s, "isakmp"))
    f = dump_isakmp;
  if (!strcmp(s, "l2tp"))
    f = dump_l2tp;
  if (!strcmp(s, "mgcp"))
    f = dump_mgcp;
  if (!strcmp(s, "mobileip"))
    f = dump_mobileip;
  if (!strcmp(s, "netbios_ns"))
    f = dump_netbios_ns;
  if (!strcmp(s, "pptp"))
    f = dump_pptp;
  if (!strcmp(s, "rip"))
    f = dump_rip;
  if (!strcmp(s, "ripng"))
    f = dump_ripng;
  if (!strcmp(s, "rtp"))
    f = dump_rtp;
  if (!strcmp(s, "sip"))
    f = dump_sip;
  if (!strcmp(s, "snmp"))
    f = dump_snmp;
  if (!strcmp(s, "slp"))
    f = dump_slp;
  if (!strcmp(s, "ftpctrl"))
    f = dump_ftpctrl;
  if (f==NULL)
    error_fatal("dynamic port assignment to unknown protocol %s", s);

  dynports_add(port, f);
}

/*----------------------------------------------------------------------------
**
** dynports_find()
**
** Find a protocol function pointer given the port number. Returns the 
** function pointer or null if not found.
** 
**----------------------------------------------------------------------------
*/

service_func_t dynports_find(u_int16_t port)
{
  int i;

  for (i=0; i<current_ptr; i++)
    if (dynport_map[i].port == port)
      return dynport_map[i].f;

  return NULL;
}
