/**************************************************************************** 
** File: mobileip.c
**
** Author: Mike Borella
**
** Comments: 
**
** $Id: mobileip.c,v 1.2 2001/01/20 00:49:36 mborella Exp $
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU Library General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "global.h"
#include "mobileip.h"

/*
 * Types of Mobile IP messages
 */

#define MOBILEIP_TYPE_REGREQ      1
#define MOBILEIP_TYPE_REGREP      3

/*
 * Mobile IP type map
 */

strmap_t mobileip_type_map[] =
{
  { MOBILEIP_TYPE_REGREQ,       "registration request"},
  { MOBILEIP_TYPE_REGREP,       "registration reply"},
  { 0, ""}
};

/*
 * Mobile IP response codes
 */

#define MOBILEIP_RESPCODE_REGACCEPTED              0
#define MOBILEIP_RESPCODE_REGACCEPTEDNOSMB         1
#define MOBILEIP_RESPCODE_FA_UNSPECIFIED           64
#define MOBILEIP_RESPCODE_FA_ADMINPROHIBITED       65
#define MOBILEIP_RESPCODE_FA_INSUFFICIENTRESOURCES 66
#define MOBILEIP_RESPCODE_FA_MNAUTHFAILED          67
#define MOBILEIP_RESPCODE_FA_HAAUTHFAILED          68
#define MOBILEIP_RESPCODE_FA_LIFETIMETOOLONG       69
#define MOBILEIP_RESPCODE_FA_POORLYFORMEDREQ       70
#define MOBILEIP_RESPCODE_FA_POORLYFORMEDREP       71
#define MOBILEIP_RESPCODE_FA_ENCAPSUNAVAILABLE     72
#define MOBILEIP_RESPCODE_FA_VJCOMPUNAVAILABLE     73
#define MOBILEIP_RESPCODE_FA_HNUNREACHABLE         80
#define MOBILEIP_RESPCODE_FA_HAHOSTUNREACHABLE     81
#define MOBILEIP_RESPCODE_FA_HAPORTUNREACHABLE     82
#define MOBILEIP_RESPCODE_FA_HAUNREACHABLE         88
#define MOBILEIP_RESPCODE_HA_UNSPECIFIED           128
#define MOBILEIP_RESPCODE_HA_ADMINPROHIBITED       129
#define MOBILEIP_RESPCODE_HA_INSUFFICIENTRESOURCES 130
#define MOBILEIP_RESPCODE_HA_MNAUTHFAILED          131
#define MOBILEIP_RESPCODE_HA_FAAUTHFAILED          132
#define MOBILEIP_RESPCODE_HA_IDMISMATCH            133
#define MOBILEIP_RESPCODE_HA_POORLYFORMEDREQ       134
#define MOBILEIP_RESPCODE_HA_TOOMANYSMB            135
#define MOBILEIP_RESPCODE_HA_UNKNOWNHAADDR         136

/*
 * Format of Mobile IP registration request
 *
 * 
 *  0                   1                   2                   3
 *  0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |     Type      |S|B|D|M|G|V|rsv|          Lifetime             |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                          Home Address                         |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                           Home Agent                          |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                        Care-of Address                        |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                                                               |
 * +                         Identification                        +
 * |                                                               |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | Extensions ...
 * +-+-+-+-+-+-+-+-
 */

typedef struct mobileip_regreq
{
  u_int8_t  type;
  u_int8_t  flags;
  u_int16_t lifetime;
  u_int32_t home_addr;
  u_int32_t home_agent;
  u_int32_t care_of_addr;
  u_int8_t  identification[8];
} mobileip_regreq_t;

/*
 * Format of Mobile IP registration reply
 *
 *  0                   1                   2                   3
 *  0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |     Type      |     Code      |           Lifetime            |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                          Home Address                         |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                           Home Agent                          |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                                                               |
 * +                         Identification                        +
 * |                                                               |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | Extensions ...
 * +-+-+-+-+-+-+-+-
 */

typedef struct mobileip_regrep
{
  u_int8_t  type;
  u_int8_t  code;
  u_int16_t lifetime;
  u_int32_t home_addr;
  u_int32_t home_agent;
  u_int8_t  identification[8];
} mobileip_regrep_t;

extern struct arg_t * my_args;

/*----------------------------------------------------------------------------
**
** dump_mobileip()
**
** Dumps mobile IP messages
**
**----------------------------------------------------------------------------
*/

void dump_mobileip(packet_t *pkt)
{
  u_int8_t          first_byte;

  /*
   * Look at the first byte to determine the type
   */

  if (look_packet_bytes(&first_byte, pkt, 1) == 0)
    return;

  switch(first_byte)
    {
    case MOBILEIP_TYPE_REGREQ:
      {
	mobileip_regreq_t regreq;
	
	/* Get the message */
	if (get_packet_bytes((u_int8_t *) &regreq, pkt, sizeof(regreq)) == 0)
	  return;

	/* Conversions */
	regreq.lifetime = ntohs(regreq.lifetime);

	/* Display */
	if (my_args->m)
	  {

	  }
	else
	  {
	    display_strmap("Type", regreq.type, mobileip_type_map);
	    display("Flags", &regreq.flags, 1, DISP_HEX);
	    display("Lifetime", (u_int8_t *) &regreq.lifetime, 2, DISP_DEC);
	    display_ipv4("Home address", (u_int8_t *) &regreq.home_addr);
	    display_ipv4("Home agent", (u_int8_t *) &regreq.home_agent);
	    display_ipv4("Care of address", (u_int8_t *) &regreq.care_of_addr);
	    display("Identification", (u_int8_t *) &regreq.identification, 8, 
		    DISP_HEX);
	  }
      }
      break;

    case MOBILEIP_TYPE_REGREP:
      {
	mobileip_regrep_t regrep;

	/* Get the message */
	if (get_packet_bytes((u_int8_t *) &regrep, pkt, sizeof(regrep)) == 0)
	  return;

	/* Conversions */
	regrep.lifetime = ntohs(regrep.lifetime);

	/* Display */
	if (my_args->m)
	  {

	  }
	else
	  {
	    display_strmap("Type", regrep.type, mobileip_type_map);
	    display("Code", &regrep.code, 1, DISP_HEX);
	    display("Lifetime", (u_int8_t *) &regrep.lifetime, 2, DISP_DEC);
	    display_ipv4("Home address", (u_int8_t *) &regrep.home_addr);
	    display_ipv4("Home agent", (u_int8_t *) &regrep.home_agent);
	    display("Identification", (u_int8_t *) &regrep.identification, 8, 
		    DISP_HEX);
	  }
      }
      break;
      
    default:
      break;
    }
}
