/*
 * $Id: ngrep.c,v 1.38 2000/06/21 18:42:37 jpr5 Exp $
 *
 * Copyright (c) 2000  Jordan Ritter <jpr5@darkridge.com>
 *
 * Please refer to the COPYRIGHT file for more information. 
 * 
 */

#if defined(BSD) || defined(SOLARIS)
#include <unistd.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <net/if.h>
#endif

#if defined(OSF1)
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <net/route.h>
#include <sys/mbuf.h>
#endif

#if defined(LINUX)
#include <getopt.h>
#include <arpa/inet.h>
#include <ctype.h>
#endif

#if defined(WIN32)
#include <io.h>
#include <getopt.h>
#include <winsock.h>
#include <nettypes.h>
#else 
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <netinet/ip_icmp.h>
#endif

#include <pcap.h>
#include <net/bpf.h>

#include <stdlib.h>
#include <string.h>
#include <signal.h>

#include "regex.h"
#include "ngrep.h"


static char rcsver[] = "$Revision: 1.38 $ (win32)";

int snaplen = 65535, promisc = 1, to = 1000;
int show_empty = 0, show_hex = 0, quiet = 0;
int match_after = 0, keep_matching = 0;
int invert_match = 0, bin_match = 0;
int matches = 0, max_matches = 0;
int live_read = 1, want_delay = 0;;

char pc_err[PCAP_ERRBUF_SIZE], *re_err;

int re_match_word = 0, re_ignore_case = 0;
struct re_pattern_buffer pattern;

char *match_data = NULL, *bin_data = NULL, *filter = NULL;
int (*match_func)() = &blank_match_func;
int match_len = 0;

struct bpf_program pcapfilter;
struct in_addr net, mask;
pcap_t *pd = NULL;
char *dev = NULL;
int link_offset;

char *read_file = NULL, *dump_file = NULL;
pcap_dumper_t *pd_dump = NULL;

struct timeval prev_ts = {0, 0}, prev_delay_ts = {0,0};
void (*print_time)() = NULL;

#ifdef WIN32
struct timeval delay_tv;
FD_SET delay_fds;
int delay_socket = 0;
#endif


int main(int argc, char **argv) {
  int c;

#ifdef WIN32
  if (!init_winsock()) 
    clean_exit(-1);
#else
  signal(SIGQUIT, clean_exit);
  signal(SIGPIPE, clean_exit);
#endif

  signal(SIGINT, clean_exit);
  signal(SIGABRT, clean_exit);

#ifdef WIN32
  while ((c = getopt(argc, argv, "LhXViwqevxlDtTn:d:A:I:O:")) != EOF) {
#else
  while ((c = getopt(argc, argv, "hXViwqevxlDtTn:d:A:I:O:")) != EOF) {
#endif
    switch (c) {
    case 'O':
      dump_file = optarg;
      break;
    case 'I':  
      read_file = optarg;
      break;
    case 'A': 
      match_after = atoi(optarg) + 1;
      break;
#ifdef WIN32	
    case 'L':
      PrintDeviceList();

      clean_exit(0);

    case 'd': {
      char *devicelist;
      int i;
		
      if ((i = atoi(optarg)) < 1) {
	fprintf(stderr, "Invalid interface.\n");
	clean_exit(-1);
      }
		
      if ((devicelist = pcap_lookupdev(pc_err)) == NULL) {
	perror(pc_err);
	clean_exit(-1);
      }
				
      if ((dev = GetAdapterFromList(devicelist, i)) == NULL) {
	fprintf(stderr, "Invalid interface.\n");
	clean_exit(-1);
      }
    }
#else
    case 'd': 
      dev = optarg;
#endif
      break;
    case 'n':
      max_matches = atoi(optarg);
      break;
    case 'T':
      print_time = &print_time_diff;
      break;
    case 't':
      print_time = &print_time_absolute;
      break;
    case 'D':
      want_delay = 1;
      break;
    case 'l':
      setvbuf(stdout, NULL, _IOLBF, 0);
      break;
    case 'x':
      show_hex++;
      break;
    case 'v':
      invert_match++;
      break;
    case 'e':
      show_empty++;
      break;
    case 'q':
      quiet++;
      break;
    case 'w':
      re_match_word++;
      break;
    case 'i':
      re_ignore_case++;
      break;
    case 'V':
      version();
    case 'X':
      bin_match++;
      break;
    case 'h':
      usage(0);
    default:
      usage(-1);
    }
  }


  if (argv[optind]) 
    match_data = argv[optind++];

  if (read_file) {
    if (!(pd = pcap_open_offline(read_file, pc_err))) {
      perror(pc_err);
      clean_exit(-1);
    }

    live_read = 0;
    printf("input: %s\n", read_file);

  } else {
    if (!dev)
      if (!(dev = pcap_lookupdev(pc_err))) {
	perror(pc_err);
	clean_exit(-1);
      }
    
    if ((pd = pcap_open_live(dev, snaplen, promisc, to, pc_err)) == NULL) {
      perror(pc_err);
      clean_exit(-1);
    }

    if (pcap_lookupnet(dev, &net.s_addr, &mask.s_addr, pc_err) == -1) {
      perror(pc_err);
      memset(&net, 0, sizeof(net));
      memset(&mask, 0, sizeof(mask));
    } 

    if (!quiet) {
#ifdef WIN32
		// the win32 port of libpcap has borked code for pcap_lookupnet.
		// look for this to be fixed in the next version.
      printf("interface: %S", dev);
#else
      printf("interface: %s", dev);
      if (net.s_addr && mask.s_addr) {
	printf(" (%s/", inet_ntoa(net));
	printf("%s)", inet_ntoa(mask)); 
      }	
#endif
      printf("\n");
    }
  }


  if (argv[optind]) {
    filter = get_filter(&argv[optind]); 

    if (pcap_compile(pd, &pcapfilter, filter, 0, mask.s_addr)) {
      free(filter); 
      filter = get_filter(&argv[optind-1]); 

      PCAP_RESTART();
      if (pcap_compile(pd, &pcapfilter, filter, 0, mask.s_addr)) {
	pcap_perror(pd, "pcap compile");
	clean_exit(-1);
      } else match_data = NULL;
    }

    if (!quiet) printf("filter: %s\n", filter); 
    
    if (pcap_setfilter(pd, &pcapfilter)) {
      pcap_perror(pd, "pcap set");
      clean_exit(-1);
    }
  }

  if (match_data) {
    if (bin_match) {
      int i = 0, n;
      char *s, *d;
      int len;

      if (re_match_word || re_ignore_case) {
	fprintf(stderr, "fatal: regex switches are incompatible with binary matching\n");
	clean_exit(-1);
      }

      len = strlen(match_data);
      if (len % 2 != 0 || !strishex(match_data)) {
	fprintf(stderr, "fatal: invalid hex string specified\n");
	clean_exit(-1);
      }

      bin_data = malloc(len / 2);
      memset(bin_data, 0, len / 2);
      d = bin_data;

      if ((s = strchr(match_data, 'x'))) 
	len -= ++s - match_data - 1;
      else s = match_data;

      while (i <= len) {
	sscanf(s+i, "%2x", &n);
	*d++ = n;
	i += 2;
      }

      match_len = len / 2;
      match_func = &bin_match_func;

    } else {
      re_syntax_options = RE_SYNTAX_EGREP;
      
      if (re_ignore_case) {
	char *s;
	int i;
	
	pattern.translate = (char*)malloc(256);
	s = pattern.translate;
	for (i = 0; i < 256; i++) 
	  s[i] = i;
	for (i = 'A'; i <= 'Z'; i++) 
	  s[i] = i + 32;

	s = match_data;
	while (*s) 
	  *s++ = tolower(*s);
      } else pattern.translate = NULL;
      
      if (re_match_word) {
	char *word_regex = malloc(strlen(match_data) * 3 + strlen(WORD_REGEX));
	sprintf(word_regex, WORD_REGEX, match_data, match_data, match_data);
	match_data = word_regex;
      }
      
      (const char *)re_err = re_compile_pattern(match_data, strlen(match_data), &pattern);
      if (re_err) {
	fprintf(stderr, "regex compile: %s\n", re_err);
	clean_exit(-1);
      }
     
      pattern.fastmap = (char*)malloc(256);
      if (re_compile_fastmap(&pattern)) {
	perror("fastmap compile failed");
	clean_exit(-1);
      }

      match_func = &re_match_func;
    }

    if (!quiet && match_data && strlen(match_data)) 
      printf("%smatch: %s%s\n", invert_match?"don't ":"", 
	     (bin_data && !strchr(match_data, 'x'))?"0x":"", match_data);
  }


  if (filter) free(filter);
  if (re_match_word) free(match_data);


  switch(pcap_datalink(pd)) {
  case DLT_EN10MB:
  case DLT_IEEE802:
    link_offset = ETHHDR_SIZE;
    break;

  case DLT_FDDI:
    link_offset = FDDIHDR_SIZE;
    break;

  case DLT_SLIP: 
    link_offset = SLIPHDR_SIZE;
    break;

  case DLT_PPP:
    link_offset = PPPHDR_SIZE;
    break;

  case DLT_RAW: 
    link_offset = RAWHDR_SIZE;
    break;

  case DLT_NULL:
    link_offset = LOOPHDR_SIZE;
    break;

  default:
    fprintf(stderr, "fatal: unsupported interface type\n");
    clean_exit(-1);
  }

  if (dump_file) {
    if (!(pd_dump = pcap_dump_open(pd, dump_file))) {
      fprintf(stderr, "fatal: %s\n", pcap_geterr(pd));
      clean_exit(-1);
    } else printf("output: %s\n", dump_file);
  }

  while (pcap_loop(pd, 0, (pcap_handler)process, 0));

  clean_exit(0);
}	


void process(u_char *data1, struct pcap_pkthdr* h, u_char *p) {
  struct ip* ip_packet = (struct ip *)(p + link_offset);

  unsigned ip_hl = ip_packet->ip_hl*4;
  unsigned ip_off = ntohs(ip_packet->ip_off);
  unsigned fragmented = ip_off & (IP_MF | IP_OFFMASK);
  unsigned frag_offset = fragmented?(ip_off & IP_OFFMASK) * 8:0;

  char *data;
  int len;

  switch (ip_packet->ip_p) {
  case IPPROTO_TCP: {
    struct tcphdr* tcp = (struct tcphdr *)(((char *)ip_packet) + ip_hl);
    unsigned tcphdr_offset = fragmented?0:(tcp->th_off * 4);

    if (!quiet) {
      printf("#");
      fflush(stdout);
    }

    data = ((char*)tcp) + tcphdr_offset;
    len = ntohs(ip_packet->ip_len) - ip_hl - tcphdr_offset;

    if (((len || show_empty) && (((int)(*match_func)(data, len)) != invert_match))
	|| keep_matching) { 

      if (!live_read && want_delay)
	dump_delay(h);

      printf("\nT ");

      if (print_time) 
	print_time(h);

      if (tcphdr_offset || !frag_offset) {
	printf("%s:%d -", inet_ntoa(ip_packet->ip_src), ntohs(tcp->th_sport));
	printf("> %s:%d", inet_ntoa(ip_packet->ip_dst), ntohs(tcp->th_dport));
	printf(" [%s%s%s%s%s%s]",
	       (tcp->th_flags & TH_ACK)?"A":"",
	       (tcp->th_flags & TH_SYN)?"S":"",
	       (tcp->th_flags & TH_RST)?"R":"",
	       (tcp->th_flags & TH_FIN)?"F":"",
	       (tcp->th_flags & TH_URG)?"U":"",
	       (tcp->th_flags & TH_PUSH)?"P":"");
      } else {
	printf("%s -", inet_ntoa(ip_packet->ip_src));
	printf("> %s", inet_ntoa(ip_packet->ip_dst));
      }

      if (fragmented) 
	printf(" %s%d@%d:%d\n", frag_offset?"+":"", ntohs(ip_packet->ip_id),
	       frag_offset, len); 
      else printf("\n");

      if (pd_dump) {
	pcap_dump((u_char*)pd_dump, h, p);
	if (!quiet) dump(data, len);
      } else dump(data, len);
    }
  }
  break;

  case IPPROTO_UDP: {
    struct udphdr* udp = (struct udphdr *)(((char *)ip_packet) + ip_hl);
    unsigned udphdr_offset = (fragmented)?0:sizeof(struct udphdr); 

    if (!quiet) {
      printf("#"); 
      fflush(stdout);
    }

    data = ((char*)udp) + udphdr_offset;
    len = ntohs(ip_packet->ip_len) - ip_hl - udphdr_offset;

    if (((len || show_empty) && (((int)(*match_func)(data, len)) != invert_match))
	|| keep_matching) { 

      if (!live_read && want_delay)
	dump_delay(h);

      printf("\nU ");

      if (print_time) 
	print_time(h);

      if (udphdr_offset || !frag_offset) {
#ifdef HAVE_DUMB_UDPHDR
	printf("%s:%d -", inet_ntoa(ip_packet->ip_src), ntohs(udp->source));
	printf("> %s:%d", inet_ntoa(ip_packet->ip_dst), ntohs(udp->dest));
#else
	printf("%s:%d -", inet_ntoa(ip_packet->ip_src), ntohs(udp->uh_sport));
	printf("> %s:%d", inet_ntoa(ip_packet->ip_dst), ntohs(udp->uh_dport));
#endif
      } else {
	printf("%s -", inet_ntoa(ip_packet->ip_src));
	printf("> %s", inet_ntoa(ip_packet->ip_dst));
      }

      if (fragmented) 
	printf(" %s%d@%d:%d\n", frag_offset?"+":"", ntohs(ip_packet->ip_id),
	       frag_offset, len); 
      else printf("\n");

      if (pd_dump) {
	pcap_dump((u_char*)pd_dump, h, p);
	if (!quiet) dump(data, len);
      } else dump(data, len);
    }
  }
  break;

  case IPPROTO_ICMP: {
    struct icmp* ic = (struct icmp *)(((char *)ip_packet) + ip_hl);
    unsigned icmphdr_offset = fragmented?0:4;

    if (!quiet) {
      printf("#"); 
      fflush(stdout);
    }

    data = ((char*)ic) + icmphdr_offset;
    len = ntohs(ip_packet->ip_len) - ip_hl - icmphdr_offset;

    if (((len || show_empty) && (((int)(*match_func)(data, len)) != invert_match))
	|| keep_matching) { 

      if (!live_read && want_delay)
	dump_delay(h);

      printf("\nI ");

      if (print_time) 
	print_time(h);

      printf("%s -", inet_ntoa(ip_packet->ip_src));
      printf("> %s", inet_ntoa(ip_packet->ip_dst));

      if (icmphdr_offset || !frag_offset) 
	printf(" %d:%d", ic->icmp_type, ic->icmp_code);

      if (fragmented) 
	printf(" %s%d@%d:%d\n", frag_offset?"+":"", ntohs(ip_packet->ip_id),
	       frag_offset, len); 
      else printf("\n");

      if (pd_dump) {
	pcap_dump((u_char*)pd_dump, h, p);
	if (!quiet) dump(data, len);
      } else dump(data, len);
    }
  }
  break;
  
  }

  if (match_after && keep_matching)
    keep_matching--;
}


int re_match_func(char *data, int len) {
  switch (re_search(&pattern, data, len, 0, len, 0)) {
  case -2: 
    perror("she's dead, jim\n");
    clean_exit(-2);
  case -1:
    return 0;
  }
  
  if (max_matches && ++matches > max_matches)
    clean_exit(0);

  if (match_after && keep_matching != match_after)
    keep_matching = match_after;

  return 1;
}


int bin_match_func(char *data, int len) {
  int stop = len - match_len;
  int i = 0;

  if (stop < 0)
    return 0;

  while (i <= stop) 
    if (!memcmp(data+(i++), bin_data, match_len)) {
      if (max_matches && ++matches > max_matches)
	clean_exit(0);

      if (match_after && keep_matching != match_after)
	keep_matching = match_after;

      return 1;
    }

  return 0;
}


int blank_match_func(char *data, int len) {
  if (max_matches && ++matches > max_matches)
    clean_exit(0);

  return 1;
}


void dump(char *data, int len) {  
  if (len > 0) {
    int width = show_hex?16:70;
    char *str = data;
    int j, i = 0;

    while (i < len) {
      printf("  ");

      if (show_hex) 
	for (j = 0; j < width; j++) {
	  if (i+j < len) 
	    printf("%02x ", (unsigned char)str[j]);
	  else printf("   ");

	  if ((j+1) % (width/2) == 0)
	    printf("   ");
	}

      for (j = 0; j < width; j++) 
	if (i+j < len) 
	  printf("%c", isprint(str[j])?str[j]:'.');
	else printf(" ");
      
      str += width;
      i += j;

      printf("\n");
    }
  }
}


char *get_filter(char **argv) {
  char **arg = argv, *theirs, *mine;
  char *from, *to;
  int len = 0;

  if (!*arg)
    return NULL;

  while (*arg) 
    len += strlen(*arg++) + 1;
 
  if (!(theirs = (char*)malloc(len + 1)) || 
      !(mine = (char*)malloc(len + sizeof(IP_ONLY))))
    return NULL;

  memset(theirs, 0, len + 1);
  memset(mine, 0, len + sizeof(IP_ONLY));

  arg = argv;
  to = theirs;

  while ((from = *arg++)) {
    while ((*to++ = *from++));
    *(to-1) = ' ';
  }

  sprintf(mine, IP_ONLY, theirs);

  free(theirs);
  return mine;
}


void clean_exit(int sig) {
  struct pcap_stat s;
  if (!quiet && sig >= 0) printf("exit\n");

  if (pattern.translate) free(pattern.translate);
  if (pattern.fastmap) free(pattern.fastmap);

  if (bin_data) free(bin_data);
  
  if (!quiet && sig >= 0 && !read_file && pd && !pcap_stats(pd, &s)) 
    printf("%d received, %d dropped\n", s.ps_recv, s.ps_drop);

  if (pd) pcap_close(pd);
  if (pd_dump) pcap_dump_close(pd_dump);

#ifdef WIN32
  if (delay_socket) close(delay_socket);
  WSACleanup();
#endif

  exit(sig);
}


int strishex(char *str) {
  char *s;
  if ((s = strchr(str, 'x'))) 
    s++;
  else s = str;

  while (*s) 
    if (!isxdigit(*s++))
      return 0;

  return 1;
}


void print_time_absolute(struct pcap_pkthdr *h) {
  struct tm *t = localtime(&h->ts.tv_sec);

  printf("%02d/%02d/%02d %02d:%02d:%02d.%06d ",
	 t->tm_year+1900, t->tm_mon, t->tm_mday, t->tm_hour,
	 t->tm_min, t->tm_sec, h->ts.tv_usec);
}


void print_time_diff(struct pcap_pkthdr *h) { 
  unsigned secs, usecs;

  if (!prev_ts.tv_sec && !prev_ts.tv_usec) {
    prev_ts.tv_sec = h->ts.tv_sec;
    prev_ts.tv_usec = h->ts.tv_usec;
  }

  secs = h->ts.tv_sec - prev_ts.tv_sec;
  if (h->ts.tv_usec >= prev_ts.tv_usec)
    usecs = h->ts.tv_usec - prev_ts.tv_usec;
  else {
    secs--; 
    usecs = 1000000 - (prev_ts.tv_usec - h->ts.tv_usec);
  }

  printf("+%d.%06d ", secs, usecs);

  prev_ts.tv_sec = h->ts.tv_sec;
  prev_ts.tv_usec = h->ts.tv_usec;
}


void dump_delay(struct pcap_pkthdr *h) {
  unsigned long secs, usecs;

  if (!prev_delay_ts.tv_sec && !prev_delay_ts.tv_usec) {
    prev_delay_ts.tv_sec = h->ts.tv_sec;
    prev_delay_ts.tv_usec = h->ts.tv_usec;
  }

  secs = h->ts.tv_sec - prev_delay_ts.tv_sec;
  if (h->ts.tv_usec >= prev_delay_ts.tv_usec)
    usecs = h->ts.tv_usec - prev_delay_ts.tv_usec;
  else {
    secs--; 
    usecs = 1000000 - (prev_delay_ts.tv_usec - h->ts.tv_usec);
  }

#ifdef WIN32
  {	
    // grevious hack, yes, but windows sucks.  sorry. :(   --jordan
    if ((delay_socket = socket(AF_INET, SOCK_STREAM, 6)) == -1) {
      fprintf(stderr, "delay socket creation failed, disabling -D\n");
      Sleep(3000); // give them time to read the message
      want_delay = 0;
      return;
    }

    FD_ZERO(&delay_fds);
    FD_SET(delay_socket, &delay_fds);

    delay_tv.tv_sec = secs;
    delay_tv.tv_usec = usecs;

    if (select(0, &delay_fds, 0, 0, &delay_tv) == -1) 
      fprintf(stdout, "WSAGetLastError = %d\n", WSAGetLastError());

    close(delay_socket);
    delay_socket = 0; // in case someone ^C's out of me
  }
#else
  sleep(secs);
  usleep(usecs);
#endif

  prev_delay_ts.tv_sec = h->ts.tv_sec;
  prev_delay_ts.tv_usec = h->ts.tv_usec;
}


void usage(int e) {
#ifdef WIN32
  printf("usage: ngrep <-LhXViwqevxlDtT> <-IO pcap_dump> <-n num> <-d dev> <-A num>\n"
	 "                               <match expression> <bpf filter>\n");
#else
  printf("usage: ngrep <-hXViwqevxlDtT> <-IO pcap_dump> <-n num> <-d dev> <-A num>\n"
	 "                              <match expression> <bpf filter>\n");
#endif
  exit(e);
}


void version(void) {
  printf("ngrep: %s\n", rcsver);
  exit(0);
}


#ifdef WIN32
void *GetAdapterFromList(void *device, int index) {
  int n = 1;
  DWORD dwVersion = GetVersion();
  DWORD dwWindowsMajorVersion = (DWORD)(LOBYTE(LOWORD(dwVersion)));

  if (dwVersion >= 0x80000000 && dwWindowsMajorVersion >= 4) {
    char *t = (char *)device;
    while (*t != '\0') {
      if (n++ == index)
	return t;

      while (*t != '\0')
	t++;
      t++; 
    }
    return NULL;

  } else {
    wchar_t *t = (wchar_t *)device;
    while (*t != '\0') {
      if (n++ == index)
	return t;

      while (*t != '\0')
	t++;
      t++; 
    }
    return NULL;
  }

}

void PrintDeviceList(void) {
  // apparently pcap_lookupdev returns a string that is:
  //   1. delimited by nulls
  //   2. terminated by a double null
  // windows port of libpcap indicates there is might
  // be a description somewhere, but I couldn't figure out how 
  // reproduce them.  --jordan
  int n = 1;
  char *device;
  DWORD dwVersion = GetVersion();
  DWORD dwWindowsMajorVersion = (DWORD)(LOBYTE(LOWORD(dwVersion)));

  if ((device = pcap_lookupdev(pc_err)) == NULL) {
    perror(pc_err);
    return;
  }

  if (dwVersion >= 0x80000000 && dwWindowsMajorVersion >= 4) {
    const char *t = (char *)device;

    printf("interface\tdevice\n");
    printf("---------\t------\n");

    while (*t != '\0') {
      printf("%9d\t", n++);
      printf("%s\n", t);

      while (*t != '\0')
	t++;
      t++; 
    }

  } else {           
    const wchar_t *t = (wchar_t *)device;

    printf("interface\tdevice\n");
    printf("---------\t------\n");

    while (*t != '\0') {
      printf("%9d\t", n++);
      printf("%S\n", t);

      while (*t != '\0')
	t++;
      t++; 
    }
  }
}

int init_winsock(void) {
  WORD wVersionRequested = MAKEWORD(2, 0);
  WSADATA wsaData;

  if (WSAStartup(wVersionRequested, &wsaData)) {
    fprintf(stderr, "fatal: unable to find a usable winsock\n");
    return 0;
  }

  // we want at least major version 2
  if (LOBYTE(wsaData.wVersion) < 2) {
    fprintf(stderr, "fatal: unable to find winsock 2.0 or greater (found %d.%d)\n",
	    LOBYTE(wsaData.wVersion), HIBYTE(wsaData.wVersion));
    WSACleanup();
    return 0;
  }
	
  return 1;
}	

unsigned short swap_int16(short s) {
  return (((unsigned short)s & 0xFF) << 8) | ((unsigned short)s >> 8);
}

#endif
