/* ASS
 * Autonomous System Scanner
 * - IGRP
 *
 * FX <fx@phenoelit.de>
 * Phenoelit (http://www.phenoelit.de)
 * (c) 2k
 *
 * $Id: ass.c,v 1.7 2000/10/05 14:54:53 fx Exp fx $
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <netinet/in.h>
#include <rpc/types.h>
#include <netdb.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <errno.h>
#include <math.h>
#include <signal.h>

#include "protocols.h"
#include "packets.h"

/* we need a sniffer enigine here */
#include <pcap.h>
#include <net/bpf.h>

/* definitions */
#define IP_IGRP_TTL	0x80
#define IP_IRDP_TTL	0x80

#define IP_BCAST	"255.255.255.255"
#define CAPLENGTH	1542

#define IGRP_OPCODE_REQUEST	2
#define IGRP_OPCODE_UPDATE	1

#define DEST_LENGTH	15

#define BANNER		"ASS [Autonomous System Scanner] $Revision: 1.7 $\n"\
			"\t(c) 2k FX <fx@phenoelit.de>\n"\
			"\tPhenoelit (http://www.phenoelit.de)\n"

#define FCHAR_IGRP	"I"
#define FCHAR_IRDP	"R"
#define FCHAR_CDP	"C"

/* new result management 
 * ... types */
typedef struct {
    char	*dest;
    u_int16_t	as;
    void	*next;			// next record
} RES_route_igrp_t;

typedef struct {
    char	*dest;
    void	*next;
} RES_route_irdp_t;

#define CAP_IGRP	0x01
#define CAP_IRDP	0x02
#define CAP_EIGRP	0x04
#define CAP_OSPF	0x08
#define CAP_SOMETHING	0x0F
#define CAP_CDP		0x10

typedef struct {
    char		*addr;
    int			capa;
    RES_route_igrp_t	*igrp;
    RES_route_irdp_t	*irdp;
    void		*next;		// next record
} result_t;


/* config */
struct {
    int			verbose;
    char		*device;

    u_int16_t		as_start,as_stop;

    int			spoof_src;
    struct in_addr	src;
    int			set_dest;
    struct in_addr	dest;

    int			prom;
    int			cont;
} cfg;

/************************************
 * globals */
u_char			*rawpacket;
int			atsock;
pcap_t			*cap;

int			stop_flag=0;

result_t		*anchor;


/************************************
 * prototypes */
void	usage(char *n);

/* IGRP construction */
u_char	*construct_igrp_request(u_int16_t autosys, int *psize);
/* IRDP construction */
u_char	*construct_irdp_request(int *psize);

/* PCAP */
int     initialize_pcap(void);
void	signaler(int sig);
void	evaluate_packet(u_char *frame,int frame_length);

/* result management */
int	add_route(int rtype,char *addr,int as,char *dest);
void	print_results();
void	clean_results();


/* the main function */
int	main(int argc, char **argv) {
    char		option;
    extern char		*optarg;

    u_char		*pcap_data, *ppacket;
    struct pcap_pkthdr	*pcap_head,phead;
    int			i;

    /* scanner */
    int			plength;
    struct timespec	slept = {0,0};



    memset(&cfg,0,sizeof(cfg));
    while ((option=getopt(argc,argv,"vcpi:a:b:S:D:T:"))!=EOF) {
	switch (option) {
	    case 'v':	/* verbose */
			cfg.verbose++;
			break;
	    case 'i':	/* local network device */
			cfg.device=smalloc(strlen(optarg)+1);
			strcpy(cfg.device,optarg);
			break;
	    case 'a':	/* autonomous system start*/
			cfg.as_start=atoi(optarg);
			break;
	    case 'b':	/* autonomous system stop */
			cfg.as_stop=atoi(optarg);
			break;
	    case 'S':	/* spoof source */
			if (inet_aton(optarg,&(cfg.src))==0) {
			    fprintf(stderr,
				    "source IP address seems to be wrong\n");
			    return (1);
			}
			cfg.spoof_src++;
			break;
	    case 'D':	/* set destination */
			if (inet_aton(optarg,&(cfg.dest))==0) {
			    fprintf(stderr,
				    "dest. IP address seems to be wrong\n");
			    return (1);
			}
			cfg.set_dest++;
			break;
	    case 'p':	/* promiscous mode */
			cfg.prom=1;
			break;
	    case 'c':	/* don't continue capture */
			cfg.cont++;
			break;
	    case 'T':	/* delay between each packet */
			slept.tv_nsec=atol(optarg);
			break;
	    default:	usage(argv[0]);
	}
    }

    if (!cfg.device) usage(argv[0]);
    if (!cfg.set_dest)
	inet_aton(IP_BCAST,&(cfg.dest));

    if (cfg.as_stop==0) 
	cfg.as_stop=65535;
    if (cfg.as_start>cfg.as_stop) {
	fprintf(stderr,"Start has to be smaller then stop ...\n");
	return (1);
    }


    /* set up socket ... */
    if ((atsock=init_socket_IP4(cfg.device,1))==(-1)) return(1);
    /* if spoofing is enabled, copy it */
    if (!cfg.spoof_src) {
	memcpy(&(cfg.src.s_addr), &(packet_ifconfig.ip.s_addr), IP_ADDR_LEN);
    }
    /* set up sniffer */
    if (initialize_pcap()==(-1)) return (1);


    /* get mem for pcap's header structure */
    pcap_head=(struct pcap_pkthdr *)smalloc(sizeof(struct pcap_pkthdr));

    /* signal handling */
    signal(SIGTERM,&signaler);
    signal(SIGABRT,&signaler);
    signal(SIGINT,&signaler);

    /* my shit */
    printf(BANNER);

    if (cfg.verbose==1) printf("Scanning ");


    /* scan for IRDP */
    if (cfg.verbose>1) printf("..sending IRDP (ICMP Router Discovery) ...\n");
    rawpacket=construct_irdp_request(&plength);
    sendpack_IP4(atsock,rawpacket,plength);
    free(rawpacket);

    /* scan IGRP */
    for (i=cfg.as_start;i<=cfg.as_stop;i++) {

	if (cfg.verbose==1) {
	    printf("."); fflush(stdout);
	} else if (cfg.verbose>1) {
	    printf("Scanning %s - AS# %d\n",
		    inet_ntoa(cfg.dest),i);
	}
	
	/* send packet */
	rawpacket=construct_igrp_request(i,&plength);
	sendpack_IP4(atsock,rawpacket,plength);
	free(rawpacket);
	
	if ((pcap_data=(u_char *)pcap_next(cap,pcap_head))!=NULL) {
	    /* make a local copy of the data, 
	     * pcap will overwrite the buffer if needed */
	    memcpy(&phead,pcap_head,sizeof(struct pcap_pkthdr));
	    ppacket=(u_char *)smalloc(phead.caplen);
	    memcpy(ppacket,pcap_data,phead.caplen);

	    evaluate_packet(ppacket,phead.caplen);

	    free(ppacket);
	}

	nanosleep(&slept,NULL);
	if (stop_flag) break;
    }

    if (!cfg.cont) {
	/* ... continue capture, may be there are still responses */
	printf("\nContinuing capture ... (hit Ctrl-C to finish)\n");
	while (!stop_flag) {
	    if ((pcap_data=(u_char *)pcap_next(cap,pcap_head))!=NULL) {
		/* make a local copy of the data, 
		 * pcap will overwrite the buffer if needed */
		memcpy(&phead,pcap_head,sizeof(struct pcap_pkthdr));
		ppacket=(u_char *)smalloc(phead.caplen);
		memcpy(ppacket,pcap_data,phead.caplen);

		evaluate_packet(ppacket,phead.caplen);

		free(ppacket);
	    }
	}
    } else {
	printf("\n");
	pcap_close(cap);
    }

    /* at the end of the day, close our socket */
    close(atsock);
    free(pcap_head);

    print_results();
    clean_results();

    return (0);
}




/********************** FUNCTIONS **********************/

void	evaluate_packet(u_char *frame,int frame_length) {
    struct ether_header		*eth;
    iphdr_t			*ip;
    igrp_t			*igrp;
    igrp_system_entry_t		*isys;
    icmphdr_t			*icmph;
    irdp_t			*irdph;
    irdp_rec_t			*irdpr;

    /* for CDP */
    struct eth_LLC		*eth_llc;
    u_char			*cdpt;
    struct cdp_address_entry	*cdpa;
    
    int				i;
    char			buffer[215];

    

    if (cfg.verbose>2) printf("... Packet ...\n");

    eth=(struct ether_header *)frame;
    if (ntohs(eth->ether_type)==ETHERTYPE_IP) {

	if (cfg.verbose>2) printf("\tIP\n");
	ip=(iphdr_t *)(frame+sizeof(struct ether_header));

	/* if it is from myself, igore it */
	if (!memcmp(&(ip->saddr),
		    &(packet_ifconfig.ip.s_addr),IP_ADDR_LEN)) return;
	if (cfg.verbose>2) printf("\tnot me\n");
	/* if it is from the spoofed self, igore it */
	if (!memcmp(&(ip->saddr),&(cfg.src),IP_ADDR_LEN)) return;

	/* IGRP  ??? */
	if (ip->protocol==IPPROTO_IGRP) {

	    if (cfg.verbose>2) printf("\tigrp\n");
	    igrp=(igrp_t *)(frame+sizeof(struct ether_header)+
		    sizeof(iphdr_t));

	    if (cfg.verbose==1) {
		printf(FCHAR_IGRP); fflush(stdout);
	    }

	    /* if this wasn't an update, just add the router */
	    if (igrp->opcode!=IGRP_OPCODE_UPDATE) {
		add_route(CAP_IGRP,inet_ntoa(ip->saddr),
			ntohs(igrp->autosys),NULL); 

	    } else {

		/* now, check for routes in the update */
		if (cfg.verbose>1) 
		    printf("\t%d exterior routes in update"
			    " (unfortunately not supported yet)\n"
			    "\t%d interior routes in update"
			    " (unfortunately not supported yet)\n"
			    "\t%d system routes in update (supported!)\n",
			    ntohs(igrp->exterior),
			    ntohs(igrp->interior),
			    ntohs(igrp->system));


		for (i=0;i<ntohs(igrp->system);i++) {

		    isys=(igrp_system_entry_t *)(frame+
			    sizeof(struct ether_header)+
			    sizeof(iphdr_t)+
			    sizeof(igrp_t)+
			    (i*sizeof(igrp_system_entry_t)));

		    memset(&buffer,0,sizeof(buffer));
		    snprintf(buffer,sizeof(buffer),"%d.%d.%d.0",
			    isys->destination[0],
			    isys->destination[1],
			    isys->destination[2]);

		    add_route(CAP_IGRP,inet_ntoa(ip->saddr),
			    ntohs(igrp->autosys),buffer); 
		} 
	    } /* end of UPDATE */

	    /* end of IGRP */
	} else if (ip->protocol==IPPROTO_ICMP) {
	    /* it's an ICMP */
	    if (cfg.verbose>2) printf("\ticmp\n");
	    icmph=(icmphdr_t *)(frame+sizeof(struct ether_header)+
		    sizeof(iphdr_t));
	    if (icmph->type==ICMP_ROUTER_ADVERT) {
		/* it's actually a router advertisement */
		if (cfg.verbose>2) printf("\tIRDP\n");
		if (cfg.verbose==1) { printf(FCHAR_IRDP); fflush(stdout); }

		irdph=(irdp_t *)(frame+sizeof(struct ether_header)+
			sizeof(iphdr_t)+sizeof(icmphdr_t));

		for (i=0;i<(irdph->num_addr);i++) {
		    irdpr=(irdp_rec_t *)(frame
			    +sizeof(struct ether_header)
			    +sizeof(iphdr_t)
			    +sizeof(icmphdr_t)
			    +sizeof(irdp_t)
			    +(i*sizeof(irdp_rec_t)));

		    memset(&buffer,0,sizeof(buffer));
		    snprintf(buffer,sizeof(buffer),"%d.%d.%d.%d",
			    irdpr->addr[0],
			    irdpr->addr[1],
			    irdpr->addr[2],
			    irdpr->addr[3]);

		    add_route(CAP_IRDP,inet_ntoa(ip->saddr),
			    0,buffer); 
		} /* end of for */
	    } 
	    /* end of ICMP IRDP */
	}
    } /* not IP */ 

    /* maybe it's CDP, this could help */

    eth_llc=(struct eth_LLC *)(frame
	    +sizeof(struct eth_ieee802_3));
    if (
	    (ntohs(eth_llc->proto)==0x2000)&&
	    (eth_llc->orgcode[0]==0x00)&&
	    (eth_llc->orgcode[1]==0x00)&&
	    (eth_llc->orgcode[2]==0x0c)) {
	/* could be CDP */

	if (cfg.verbose==1) { printf(FCHAR_CDP); fflush(stdout); } 
	else if (cfg.verbose>1) printf("CDP packet ...\n");

	cdpt=(u_char *)(frame
		+sizeof(struct eth_ieee802_3)
		+sizeof(struct eth_LLC)
		+sizeof(struct cdphdr));

	do {
	    //printf("Type %d\n",ntohs(*((u_int16_t *)cdpt)));
	    if (ntohs(*((u_int16_t *)cdpt))==TYPE_ADDRESS) {
		if (cfg.verbose>1) 
		    printf("Number of addr's in it: %d\n",
			ntohl(*((u_int32_t *)(cdpt+2*sizeof(u_int16_t)))));

		cdpa=(struct cdp_address_entry *)(cdpt+
			2*sizeof(u_int16_t)+sizeof(u_int32_t));
		if (cdpa->proto!=0xCC) {
		    printf("CDP not speeking IP !\n");
		} else {
		    memset(&buffer,0,sizeof(buffer));
		    snprintf(buffer,sizeof(buffer),"%d.%d.%d.%d",
			    (cdpa->addr),
			    *(&(cdpa->addr)+1),
			    *(&(cdpa->addr)+2),
			    *(&(cdpa->addr)+3));
		    add_route(CAP_CDP,buffer,0,NULL); 
		}
	    }
	    /*printf("Size: %d\n",
		    htons(
			*((u_int16_t *)((u_char*)cdpt+sizeof(u_int16_t)))
			));*/
	    cdpt+=htons(
		    *((u_int16_t *)((u_char*)cdpt+sizeof(u_int16_t)))
		    );

	} while (
		((u_char*)cdpt-(u_char*)frame)
		+htons(
		    *((u_int16_t *)((u_char*)cdpt+sizeof(u_int16_t)))
		    )
		<frame_length);
	/* means: loop as long through the CDP sections until the pointer
	 * would reach outside the packet */
    }
    return;
}



void	signaler(int sig) {
    stop_flag++;
    if (cfg.verbose>2)
	fprintf(stderr,"\nSignal received.\n");
    pcap_close(cap);
}


int	initialize_pcap(void) {
#define PATTERNSTRING	"not src host "
#define IPSTRLEN	16
    char                pcap_err[PCAP_ERRBUF_SIZE]; /* buffer for pcap errors */
    struct bpf_program  cfilter;                   /* the compiled filter */
    bpf_u_int32		network,netmask;

    char		tipstr[IPSTRLEN+1];
    char		*notfilter;

    /* prepare filter */
    memset(&tipstr,0,IPSTRLEN+1);
    strcpy(tipstr,inet_ntoa(cfg.src));
    notfilter=(char *)smalloc(strlen(PATTERNSTRING)+IPSTRLEN+1);
    strcpy(notfilter,PATTERNSTRING);
    strcat(notfilter,tipstr);

    /* get my network and netmask */
    if (pcap_lookupnet(cfg.device,&network,&netmask,pcap_err)!=0) {
	fprintf(stderr,"pcap_lookupnet(): %s\n",pcap_err);
	return (-1);
    }

    /* open the sniffer */
    if ((cap=pcap_open_live(cfg.device,CAPLENGTH,
		    cfg.prom, /* in promi mode */
		    0, /* not timeouts */
		    pcap_err))==NULL) {
	fprintf(stderr,"pcap_open_live(): %s\n",pcap_err);
	return (-1);
    }

    if (pcap_datalink(cap)!=DLT_EN10MB) {
	fprintf(stderr,"works on Ethernet only, sorry.\n");
	return (-1);
    }

    if (pcap_compile(cap,&cfilter,notfilter,0,netmask)!=0) {
	pcap_perror(cap,"pcap_compile()");
	return (-1);
    }

    if (pcap_setfilter(cap,&cfilter)!=0) {
	pcap_perror(cap,"pcap_setfilter()");
	return (-1);
    }
    
    free(notfilter);
    return 0;
}


/* constructs the IGRP request packet
 * * Returns a pointer to the packet or NULL if failed
 * * returns also the size in *psize */
u_char	*construct_igrp_request(u_int16_t autosys, int *psize) {
#define PADDING_SIZE	14
    u_char			*tpacket;
    iphdr_t			*iph;
    igrp_t			*igrph;
    u_int16_t			cs;		/* checksum */


    *psize=PADDING_SIZE+sizeof(igrp_t)+sizeof(iphdr_t);
    tpacket=(u_char *)smalloc(PADDING_SIZE+sizeof(igrp_t)+sizeof(iphdr_t)
	    +3 /* for my checksum function, which sometimes 
		  steps over the mark */
	    );

    /* make up IP packet */
    iph=(iphdr_t *)tpacket;

    iph->version=4;
    iph->ihl=sizeof(iphdr_t)/4;

    iph->tot_len=htons(*psize);
    iph->ttl=IP_IGRP_TTL;
    iph->protocol=IPPROTO_IGRP;

    memcpy(&(iph->saddr.s_addr),&(cfg.src.s_addr),IP_ADDR_LEN);
    memcpy(&(iph->daddr.s_addr),&(cfg.dest.s_addr),IP_ADDR_LEN);

    /* make up the IGRP header */
    igrph=(igrp_t *)(tpacket+sizeof(iphdr_t));
    igrph->version=1;
    igrph->opcode=2;		/* Update */
    igrph->edition=0;
    igrph->autosys=htons(autosys);
    igrph->interior=0;
    igrph->system=0;
    igrph->exterior=0;

    /* make up checksum */
    cs=chksum((u_char *)igrph,(*psize-sizeof(iphdr_t)));
    igrph->checksum=cs;

    return tpacket;
}

/* constructs the IRDP request packet
 * * Returns a pointer to the packet or NULL if failed
 * * returns also the size in *psize */
u_char	*construct_irdp_request(int *psize) {
    u_char			*tpacket;
    iphdr_t			*iph;
    irdp_solicitation_t		*irdph;
    u_int16_t			cs;		/* checksum */


    *psize=sizeof(irdp_solicitation_t)+sizeof(iphdr_t);
    tpacket=(u_char *)smalloc(sizeof(irdp_solicitation_t)+sizeof(iphdr_t)
	    +3 /* for my checksum function, which sometimes 
		  steps over the mark */
	    );

    /* make up IP packet */
    iph=(iphdr_t *)tpacket;

    iph->version=4;
    iph->ihl=sizeof(iphdr_t)/4;

    iph->tot_len=htons(*psize);
    iph->ttl=IP_IRDP_TTL;
    iph->protocol=IPPROTO_ICMP;

    memcpy(&(iph->saddr.s_addr),&(cfg.src.s_addr),IP_ADDR_LEN);
    memcpy(&(iph->daddr.s_addr),&(cfg.dest.s_addr),IP_ADDR_LEN);

    /* make up the irdp_solicitation_t header */
    irdph=(irdp_solicitation_t *)(tpacket+sizeof(iphdr_t));
    irdph->type=ICMP_SOLICITATION;
    irdph->code=0;
    irdph->checksum=0;
    irdph->reserved=0;

    /* make up checksum */
    cs=chksum((u_char *)irdph,(*psize-sizeof(iphdr_t)));
    irdph->checksum=cs;

    return tpacket;
}


void	usage(char *n) {
    printf(
	    "Usage: \n"
	    "%s [-v[v[v]]] -i <interface> [-p] [-c]\n\t"
	    "-a <autonomous system start> -b <autonomous system stop>\n\t"
	    "[-S <spoofed source IP>] [-D <destination ip>]\n"
	    "[-T <millisec delay>] ",
	    n);
    exit (1);
}


int	add_route(int rtype,char *addr,int as,char *dest) {
    result_t		*current,*c2;
    RES_route_igrp_t	*c_igrp;
    RES_route_irdp_t	*c_irdp;

    current=anchor;

    c2=NULL;
    /* look for already exisiting one */
    while (current!=NULL) {
	if (!strcmp(current->addr,addr)) break;
	c2=current;
	current=current->next;
    }

    /* if not existing, add one */
    if (current==NULL) {
	current=smalloc(sizeof(result_t));
	current->addr=(char *)smalloc(strlen(addr)+1);
	strcpy(current->addr,addr);
	current->next=NULL;
	current->igrp=NULL;

	if (c2) c2->next=current;
	/* if it is the first, make it anchor */
	if (anchor==NULL) anchor=current;
    }

    /* may be there are no rooutes in it ... */
    if (rtype==0) return 0;

    switch (rtype) { 
	case CAP_IGRP: /* add an IGRP route */
	    current->capa=current->capa|CAP_IGRP;
	    
	    if (current->igrp==NULL) {
		current->igrp=(RES_route_igrp_t *)
		    smalloc(sizeof(RES_route_igrp_t));
		current->igrp->dest=(char *)smalloc(strlen(dest)+1);
		strcpy(current->igrp->dest,dest);
		current->igrp->as=as;
	    } else {
		c_igrp=(RES_route_igrp_t *)(current->igrp);
		if (!strcmp(c_igrp->dest,dest)) return 1;
		while (c_igrp->next!=NULL) {
		    if (!strcmp(c_igrp->dest,dest)) return 1;
		    c_igrp=c_igrp->next;
		}
		c_igrp->next=(RES_route_igrp_t *)
		    smalloc(sizeof(RES_route_igrp_t));
		c_igrp=c_igrp->next;
		c_igrp->dest=(char *)smalloc(strlen(dest)+1);
		strcpy(c_igrp->dest,dest);
		c_igrp->as=as;
	    }
	    break;
		
	case CAP_IRDP:
	    current->capa=current->capa|CAP_IRDP;
	    if (current->irdp==NULL) {
		current->irdp=(RES_route_irdp_t *)
		    smalloc(sizeof(RES_route_irdp_t));
		current->irdp->dest=(char *)smalloc(strlen(dest)+1);
		strcpy(current->irdp->dest,dest);
	    } else {
		c_irdp=(RES_route_irdp_t *)(current->irdp);
		if (!strcmp(c_irdp->dest,dest)) return 1;
		while (c_irdp->next!=NULL) {
		    if (!strcmp(c_irdp->dest,dest)) return 1;
		    c_irdp=c_irdp->next;
		}
		c_irdp->next=(RES_route_irdp_t *)
		    smalloc(sizeof(RES_route_irdp_t));
		c_irdp=c_irdp->next;
		c_irdp->dest=(char *)smalloc(strlen(dest)+1);
		strcpy(c_irdp->dest,dest);
	    }
	    break;

	case CAP_CDP:
	    current->capa=current->capa|CAP_CDP;
	    break;

	default:
	    fprintf(stderr,"Well ... internal function called with "
		    "bullshit argument. bad sign!\n");
	    exit(-15);
    }

    return 0;
}


void	print_results() {
    result_t		*current;
    RES_route_igrp_t	*c_igrp;
    RES_route_irdp_t	*c_irdp;

    printf("\n\n>>>Results>>>\n");

    current=anchor;
    while (current!=NULL) {
	printf("Router %15s\t(",current->addr);

	/* capabilities */
	if (current->capa&CAP_IGRP) printf("IGRP ");
	if (current->capa&CAP_IRDP) printf("IRDP ");
	if (current->capa&CAP_CDP) printf("CDP ");
	printf(")\n");

	c_igrp=current->igrp;
	while (c_igrp!=NULL) {
	    printf("\tIGRP\t[%5d]\tnetwork %17s\n",c_igrp->as,c_igrp->dest);
	    c_igrp=c_igrp->next;
	}
	c_irdp=current->irdp;
	while (c_irdp!=NULL) {
	    printf("\tIRDP\t[xxxxx]\tinterface %15s\n",c_irdp->dest);
	    c_irdp=c_irdp->next;
	}

	current=current->next;
    }
}


void	clean_results() {
    result_t		*k,*current;
    RES_route_igrp_t	*k_igrp,*c_igrp;

    current=anchor;
    while (current!=NULL) {

	c_igrp=current->igrp;
	while (c_igrp!=NULL) {
	    k_igrp=c_igrp;
	    c_igrp=c_igrp->next;
	    free(k_igrp);
	}

	k=current;
	current=current->next;
	free(k);
    }
}



