/*
 *  $Id: libnet_build_tcp.c,v 1.16 2002/02/25 17:24:05 route Exp $
 *
 *  libnet
 *  libnet_build_tcp.c - TCP packet assembler
 *
 *  Copyright (c) 1998 - 2002 Mike D. Schiffman <mike@infonexus.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#if (HAVE_CONFIG_H)
#include "../include/config.h"
#endif
#include "../include/libnet.h"

libnet_ptag_t
libnet_build_tcp(u_short sp, u_short dp, u_long seq, u_long ack,
            u_char control, u_short win, u_short sum, u_short urg, u_short len,
            u_char *payload, u_long payload_s, libnet_t *l, libnet_ptag_t ptag)
{
    u_long n;
    u_short h;
    libnet_pblock_t *p;
    struct libnet_tcp_hdr tcp_hdr;

    if (l == NULL)
    { 
        return (-1);
    } 

    n = LIBNET_TCP_H + payload_s;
    h = len;                        /* header stuff for csum */

    /*
     *  Find the existing protocol block if a ptag is specified, or create
     *  a new one.
     */
    p = libnet_pblock_probe(l, ptag, n, LIBNET_PBLOCK_TCP_H);
    if (p == NULL)
    {
        return (-1);
    }

    tcp_hdr.th_sport   = htons(sp);    /* source port */
    tcp_hdr.th_dport   = htons(dp);    /* destination port */
    tcp_hdr.th_seq     = htonl(seq);   /* sequence number */
    tcp_hdr.th_ack     = htonl(ack);   /* acknowledgement number */
    tcp_hdr.th_flags   = control;      /* control flags */
    tcp_hdr.th_x2      = 0;            /* UNUSED */
    tcp_hdr.th_off     = 5;            /* 20 byte header */
    tcp_hdr.th_win     = htons(win);   /* window size */
    tcp_hdr.th_sum     = sum;          /* checksum */ 
    tcp_hdr.th_urp     = urg;          /* urgent pointer */

    n = libnet_pblock_append(l, p, (u_char *)&tcp_hdr, LIBNET_TCP_H);
    if (n == -1)
    {
        goto bad;
    }

    if (payload && payload_s)
    {
        n = libnet_pblock_append(l, p, payload, payload_s);
        if (n == -1)
        {
            goto bad;
        }
    }

    if (sum == 0)
    {
        /*
         *  If checksum is zero, by default libnet will compute a checksum
         *  for the user.  The programmer can override this by calling
         *  libnet_toggle_checksum(l, ptag, 1);
         */
        libnet_pblock_setflags(p, LIBNET_PBLOCK_DO_CHECKSUM);
    }
    return (ptag ? ptag : libnet_pblock_update(l, p, h, LIBNET_PBLOCK_TCP_H));
bad:
    libnet_pblock_free(p);
    return (-1);
}


libnet_ptag_t
libnet_build_tcp_options(u_char *options, u_long options_s, libnet_t *l, 
        libnet_ptag_t ptag)
{
    u_char i, j;
    u_long n;
    u_short h;
    libnet_pblock_t *p;
    struct libnet_ipv4_hdr *ip_hdr;
    struct libnet_tcp_hdr *tcp_hdr;

    if (l == NULL)
    { 
        return (-1);
    }

    /* check options list size */
    if (options_s > LIBNET_MAXOPTION_SIZE)
    {
        snprintf(l->err_buf, LIBNET_ERRBUF_SIZE,
            "libnet_build_tcp_options: options list is too large %ld\n",
            options_s);
        return (-1);
    }

    n = options_s;       /* size of memory block */

    /*
     *  Find the existing protocol block if a ptag is specified, or create
     *  a new one.
     */
    p = libnet_pblock_probe(l, ptag, n, LIBNET_PBLOCK_TCPO_H);
    if (p == NULL)
    {
        return (-1);
    }

    /* check to make sure we've got a TCP and IP header behind us */
    if (p->prev->type != LIBNET_PBLOCK_TCP_H)
    {
        snprintf(l->err_buf, LIBNET_ERRBUF_SIZE,
            "libnet_build_tcp_options(): TCP header doesn't precede\n");
        goto bad;
    }
    if (p->prev->prev->type != LIBNET_PBLOCK_IPV4_H)
    {
        snprintf(l->err_buf, LIBNET_ERRBUF_SIZE,
        "libnet_build_tcp_options(): IP header doesn't precede TCP header\n");
        goto bad;
    }

    /* this is ugly, we should abstract all of this down to the pblock layer */
    tcp_hdr = (struct libnet_tcp_hdr *)p->prev->buf;
    ip_hdr = (struct libnet_ipv4_hdr *)p->prev->prev->buf;

    h = ntohs(ip_hdr->ip_len);
    if (options_s + h > IP_MAXPACKET)
    {
        snprintf(l->err_buf, LIBNET_ERRBUF_SIZE,
            "libnet_build_tcp_options(): packet would be too large\n");
        goto bad;
    }

    n = libnet_pblock_append(l, p, options, options_s);
    if (n == -1)
    {
        goto bad;
    }

    /*
     *  Count up number of 32-bit words in options list, padding if
     *  neccessary.
     */
    for (i = 0, j = 0; i < options_s; i++)
    {
        (i % 4) ? j : j++;
    }
         
    tcp_hdr->th_off   += j;
    /* ip_hdr->ip_len  = htons(options_s + h); */

    return (ptag ? ptag : libnet_pblock_update(l, p, options_s,
            LIBNET_PBLOCK_TCPO_H));
bad:
    libnet_pblock_free(p);
    return (-1);
}

/* EOF */
