/*
 *  libnet
 *  libnet_build_fddi.c - Fiber Distributed Data Interface packet assembler
 *
 *  Copyright (c) 1998 - 2003 Mike D. Schiffman <mike@infonexus.com>
 *                            Jason Damron      <jsdamron@hushmail.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#if (HAVE_CONFIG_H)
#include "../include/config.h"
#endif
#include "../include/libnet.h"

libnet_ptag_t
libnet_build_fddi(u_char fc, u_char *dst, u_char *src, u_char dsap,
                  u_char ssap, u_char cf, u_char *org, u_short type,
                  u_char *payload, u_long payload_s, libnet_t *l, 
                  libnet_ptag_t ptag)
{
    u_long n, h;
    u_short protocol_type;
    libnet_pblock_t *p;
    struct libnet_fddi_hdr fddi_hdr;

    if (l == NULL)
    { 
        return (-1);
    } 

    /* sanity check injection type if we're not in advanced mode */
    if (l->injection_type != LIBNET_LINK &&
            !(((l->injection_type) & LIBNET_ADV_MASK)))
    {
        sprintf(l->err_buf,
            "%s(): called with non-link layer wire injection primitive",
                    __FUNCTION__);
        p = NULL;
        goto bad;
    }

    n = LIBNET_FDDI_H + payload_s;
    h = 0;
 
    /*
     *  Find the existing protocol block if a ptag is specified, or create
     *  a new one.
     */
    p = libnet_pblock_probe(l, ptag, n, LIBNET_PBLOCK_FDDI_H);
    if (p == NULL)
    {
        return (-1);
    }

    fddi_hdr.fddi_frame_control     = fc;             /* Class/Format/Priority */
    memcpy(fddi_hdr.fddi_dhost, dst, FDDI_ADDR_LEN);  /* destination fddi address */
    memcpy(fddi_hdr.fddi_shost, src, FDDI_ADDR_LEN);  /* source fddi address */
    fddi_hdr.fddi_llc_dsap          = dsap;           /* */
    fddi_hdr.fddi_llc_ssap          = ssap;           /* */
    fddi_hdr.fddi_llc_control_field = cf;             /* Class/Format/Priority */
    memcpy(&fddi_hdr.fddi_llc_org_code, org, LIBNET_ORG_CODE_SIZE); 

    /* Deal with unaligned short for type */
    protocol_type = htons(type);
    memcpy(&fddi_hdr.fddi_type, &protocol_type, sizeof(short));   /* Protocol Type */

    n = libnet_pblock_append(l, p, (u_char *)&fddi_hdr, LIBNET_FDDI_H);
    if (n == -1)
    {
        goto bad;
    }
 
    if ((payload && !payload_s) || (!payload && payload_s))
    {
        sprintf(l->err_buf, "%s(): payload inconsistency\n", __FUNCTION__);
        goto bad;
    }

    if (payload && payload_s)
    {
        n = libnet_pblock_append(l, p, payload, payload_s);
        if (n == -1)
        {
            goto bad;
        }
    }
 
    return (ptag ? ptag : libnet_pblock_update(l, p, h, LIBNET_PBLOCK_FDDI_H));
bad:
    libnet_pblock_delete(l, p);
    return (-1);
}


libnet_ptag_t
libnet_autobuild_fddi(u_char fc, u_char *dst, u_char dsap, u_char ssap,
                      u_char cf, u_char *org, u_short type, libnet_t *l)
{
    u_long n, h;
    u_short protocol_type;
    struct libnet_fddi_addr *src;
    libnet_pblock_t *p;
    libnet_ptag_t ptag;
    struct libnet_fddi_hdr fddi_hdr;

    if (l == NULL)
    { 
        return (-1);
    } 

    /* sanity check injection type if we're not in advanced mode */
    if (l->injection_type != LIBNET_LINK &&
            !(((l->injection_type) & LIBNET_ADV_MASK)))
    {
        sprintf(l->err_buf,
            "%s(): called with non-link layer wire injection primitive",
                    __FUNCTION__);
        p = NULL;
        goto bad;
    }

    n = LIBNET_FDDI_H;
    h = 0;
    ptag = LIBNET_PTAG_INITIALIZER;

    /* FDDI and Ethernet have the same address size - so just typecast */
    src = (struct libnet_fddi_addr *) libnet_get_hwaddr(l);
    if (src == NULL)
    {
        /* err msg set in libnet_get_hwaddr() */
        return (-1);
    }

    p = libnet_pblock_probe(l, ptag, n, LIBNET_PBLOCK_FDDI_H);
    if (p == NULL)
    {
        return (-1);
    }

    fddi_hdr.fddi_frame_control     = fc;             /* Class/Format/Priority */
    memcpy(fddi_hdr.fddi_dhost, dst, FDDI_ADDR_LEN);  /* destination fddi address */
    memcpy(fddi_hdr.fddi_shost, src, FDDI_ADDR_LEN);  /* source fddi address */
    fddi_hdr.fddi_llc_dsap          = dsap;           /* */
    fddi_hdr.fddi_llc_ssap          = ssap;           /* */
    fddi_hdr.fddi_llc_control_field = cf;             /* Class/Format/Priority */
    memcpy(&fddi_hdr.fddi_llc_org_code, org, LIBNET_ORG_CODE_SIZE); 

    /* Deal with unaligned short for type */
    protocol_type = htons(type);
    memcpy(&fddi_hdr.fddi_type, &protocol_type, sizeof(short));   /* Protocol Type */

    n = libnet_pblock_append(l, p, (u_char *)&fddi_hdr, LIBNET_FDDI_H);
    if (n == -1)
    {
        goto bad;
    }

    return (libnet_pblock_update(l, p, h, LIBNET_PBLOCK_FDDI_H));
bad:
    libnet_pblock_delete(l, p);
    return (-1); 
}
/* EOF */
