/*
 * THE NEMESIS PROJECT
 * Copyright (C) 2002, 2003 Jeff Nathan <jeff@snort.org>
 *
 * nemesis-ethernet.c (Ethernet Packet Injector)
 *
 */

#include "nemesis-ethernet.h"
#include "nemesis.h"

static ETHERhdr etherhdr;
static PayloadData pd;
static int got_payload;
static int got_link;
static char *file = NULL;              /* payload file name */
static char *device = NULL;            /* Ethernet device */

int nemesis_ethernet(int argc, char **argv)
{
    const char *module = "Ethernet Packet Injection";

    maketitle(title, module, version);
  
#if 0
    if (argc < 2)
        ethernet_usage(argv[0]);
#endif

    if (argc > 1 && !strncmp(argv[1], "help", 4))
        ethernet_usage(argv[0]);

    ethernet_initdata();

    ethernet_cmdline(argc, argv);    

    ethernet_validatedata();
    
    ethernet_verbose();

    if (got_payload)
    {
        if ((pd.payload = calloc(ETHERBUFFSIZE, sizeof(char))) == NULL)
        {
            perror("ERROR: Unable to allocate packet payload memory");
            exit(1);
        }

        if ((pd.payload_s = readfile(pd.payload, file, ETHERBUFFSIZE, 
                PAYLOADMODE)) < 0)
        {
            fprintf(stderr, "ERROR: readfile() failure.\n");
            exit(1);
        }
    }

    if (buildether(&etherhdr, &pd, device) < 0)
    {
        printf("\nEthernet Injection Failure\n");

        if (got_payload)
            free(pd.payload);

        exit(1);
    }
    else
    {
        printf("\nEthernet Packet Injected\n");

        if (got_payload)
            free(pd.payload);

        exit(0);
    }
}

void ethernet_initdata(void)
{
    /* defaults */
    etherhdr.ether_type = ETHERTYPE_IP;     /* Ethernet type IP */
    memset(etherhdr.ether_shost, 0, 6);     /* Ethernet source address */
    memset(etherhdr.ether_dhost, 0xff, 6);  /* Ethernet destination address */
    pd.payload = NULL;
    pd.payload_s = 0;
}

void ethernet_validatedata(void)
{
    struct sockaddr_in sin;

    /* validation tests */
    if (!device)
    { 
        if (libnet_select_device(&sin, &device, (char *)&errbuf) < 0)
        {
            fprintf(stderr, "ERROR: Device not specified and unable to "
                    "automatically select\n a device.\n");
            exit(1);
        }
        else
        {
            got_link = 1;
        }
    }

    /* Determine if there's a source hardware address set */
    if ((nemesis_check_link(&etherhdr, device)) < 0)
    {
        fprintf(stderr, "ERROR: Cannot retrieve hardware address of %s.\n", 
                device);
        exit(1);
    } 
}

void ethernet_usage(char *arg)
{
    putchar('\n');
    puts(title);
    putchar('\n');
  
    printf("Ethernet Usage:\n  %s [-v (verbose)] [options]\n\n", arg);
    printf("Ethernet Options: \n"
           "  -d <Ethernet device>\n"
           "  -H <Source MAC address>\n"
           "  -M <Destination MAC address>\n"
           "  -P <Payload file>\n"
           "  -T <Ethernet frame type (defaults to IP)>\n\n");
    exit(1);
}

void ethernet_cmdline(int argc, char **argv)
{
    int opt, i;
    u_int32_t addr_tmp[6];
    char *ethernet_options;
    extern char *optarg;
    extern int optind;

#if defined(ENABLE_PCAPOUTPUT)
    ethernet_options = "d:H:M:P:T:v?";
#else
    ethernet_options = "d:H:M:P:T:v?";
#endif

    while ((opt = getopt(argc, argv, ethernet_options)) != -1)
    {
        switch (opt)
        {
            case 'd':   /* Ethernet device */
                if (strlen(optarg) < 256)
                {
                    device = strdup(optarg);
                    got_link = 1;
                }
                else
                {
                    fprintf(stderr, "ERROR: device %s > 256 characters.\n",
                            optarg);
                    exit(1);
                }
                break;
            case 'H':    /* Ethernet source address */
                memset(addr_tmp, 0, sizeof(addr_tmp));
                sscanf(optarg, "%02X:%02X:%02X:%02X:%02X:%02X", &addr_tmp[0],
                        &addr_tmp[1], &addr_tmp[2], &addr_tmp[3], &addr_tmp[4],
                        &addr_tmp[5]);
                for (i = 0; i < 6; i++)
                    etherhdr.ether_shost[i] = (u_int8_t)addr_tmp[i];
                break;
            case 'M':    /* Ethernet destination address */
                memset(addr_tmp, 0, sizeof(addr_tmp));
                sscanf(optarg, "%02X:%02X:%02X:%02X:%02X:%02X", &addr_tmp[0],
                        &addr_tmp[1], &addr_tmp[2], &addr_tmp[3], &addr_tmp[4],
                        &addr_tmp[5]);
                for (i = 0; i < 6; i++)
                    etherhdr.ether_dhost[i] = (u_int8_t)addr_tmp[i];
                break;
            case 'P':   /* payload file */
                if (strlen(optarg) < 256)
                {
                    file = strdup(optarg);
                    got_payload = 1;
                }
                else
                {
                    fprintf(stderr, "ERROR: payload file %s > 256 "
                            "characters.\n", optarg);
                    exit(1);
                }
                break;
            case 'T':
                etherhdr.ether_type = xgetint16(optarg);
            case 'v':
                verbose++;
                if (verbose == 1)
                {
                    putchar('\n');
                    puts(title);
                    putchar('\n');
                }
                break;
            case '?':    /* FALLTHROUGH */
            default:
                ethernet_usage(argv[0]);
                break;
        }    
    }
    argc -= optind;
    argv += optind;
}

void ethernet_verbose(void)
{
    if (verbose)
        printmac(etherhdr);
}
