/* THE NEMESIS PROJECT
 * Copyright (C) 2001 - 2003 Jeff Nathan <jeff@snort.org>
 *
 * nemesis-functions.c (nemesis utility functions)
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <unistd.h>
#include <limits.h>
#include <netdb.h>
#include <sys/time.h>
#include <math.h>
#include <errno.h>
#include <netinet/in.h>
#if defined(HAVE_NETINET_IP_VAR_H)
    #include <netinet/ip_var.h>
#endif
#include <libnet.h>
#include "nemesis.h"

const char *version = " -=- The NEMESIS Project Version 1.4beta2";

char zero[6];
char one[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
char title[81];
char errbuf[256];       /* all-purpose error buffer */
char *pcap_outfile;     /* pcap output file name */

/****************************************************************************
 * Function: xgetint32(const char *)
 *
 * Purpose: Convert user supplied string to a u_int32_t or exit on invalid data.
 *          
 * Arguments: str => string to be converted
 *
 * Returns: u_int32_t conversion of input string
 *
 ****************************************************************************/
u_int32_t xgetint32(const char *str)
{
    char *endp;
    u_long val;

    val = strtoul(str, &endp, 0);
    if(str == endp || *endp)
    {
        fprintf(stderr, 
                "ERROR: Argument %s must be a positive integer.\n", str);
        exit(1);     
    }          

    if (val > UINT_MAX)
    {
        fprintf(stderr, 
                "ERROR: Argument %s must be a positive integer between "
                "0 and 4294967295.\n", str);
        exit(1);
    }
    return (u_int32_t)val;
}

/****************************************************************************
 * Function: xgetint16(const char *)
 *
 * Purpose: Convert user supplied string to a u_int16_6 or exit on invalid data.
 *          
 * Arguments: str => string to be converted
 *
 * Returns: u_int16_t conversion of input string
 *
 ****************************************************************************/
u_int16_t xgetint16(const char *str)
{
    char *endp;
    u_long val;

    val = strtoul(str, &endp, 0);
    if(str == endp || *endp)
    {
        fprintf(stderr, 
                "ERROR: Argument %s must be a positive integer.\n", str);
        exit(1);     
    }          

    if (val > USHRT_MAX)
    {
        fprintf(stderr, 
                "ERROR: Argument %s must be a positive integer between "
                "0 and 65535.\n", str);
        exit(1);
    }
    return (u_int16_t)val;
}

/****************************************************************************
 * Function: xgetint8(const char *)
 *
 * Purpose: Convert user supplied string to a u_int8_t or exit on invalid data.
 *          
 * Arguments: str => string to be converted
 *
 * Returns: u_int8_t conversion of input string
 *
 ****************************************************************************/
u_int8_t xgetint8(const char *str)
{
    char *endp;
    u_long val;

    val = strtoul(str, &endp, 0);
    if(str == endp || *endp)
    {
        fprintf(stderr, 
                "ERROR: Argument %s must be a positive integer.\n", str);
        exit(1);     
    }          

    if (val > UCHAR_MAX)
    {
        fprintf(stderr, 
                "ERROR: Argument %s must be a positive integer between "
                "0 and 255.\n", str);
        exit(1);
    }
    return (u_int8_t)val;
}

/****************************************************************************
 * Function: hexdump(char *, u_int32_t)
 *
 * Purpose: Dumps a packet payload in hex format
 *
 * Arguments: buf => pointer to allocated payload memory
 *            len => length of buffer to print in hex format
 *
 * Notes: Contributed by Dragos Ruiu <dr@kyx.net>.  A very nice piece of code.
 *
 * Returns: void function
 *
 ****************************************************************************/
void hexdump(char *buf, u_int32_t len)
{
    int c;
    char *p, *l, *dump, dumpbuf[40];

    l = &(buf[len-1]);
    dump = dumpbuf;
    putchar('\n');

    printf("[Hexdump]\n");
    for (p = buf; p <= l; p++)
    {
        *(dump++) = (isprint(*p) && (*p < 0x7f) ? *p : '.');
        putchar((c = (*p & 0xF0) >> 4) < 10 ? c + '0' : c + '7');
        putchar((c = *p & 0x0F) < 10 ? c + '0' : c + '7');
        if (!(((p-buf)+1) % 16) || (p == l))
        {
            if(p == l) // pad last line
            {
                while(((((p++)-buf)+1) % 16))
                {
                    putchar(' ');
                    putchar(' ');
                    putchar(' ');
                }
            }
            *dump = 0;
            putchar(' ');
            putchar(' ');
            puts(dumpbuf);
            dump = dumpbuf;
        }
        else
            putchar(' ');
    }
    putchar('\n');
}

/****************************************************************************
 * Function: readfile(u_int8_t *, char *, u_int32_t, u_int32_t)
 *
 * Purpose: Reads a user supplied file or stdin into memory for use in 
 *          building packet payloads or adding options to IP or TCP headers.
 *
 * Arguments: memory => pointer to allocated payload/options memory
 *            file => pointer to filename to open
 *            maxsize => maximum number of bytes to read from file or stdin
 *            mode => switch controlling mode of operation (for error reporting
 *                    only)
 *
 * Returns: number of bytes read on success, -1 on failure
 *
 ****************************************************************************/
int readfile(u_int8_t *memory, char *file, u_int32_t maxsize, u_int32_t mode)
{
    int fd = -1, bytesread = 0;
    FILE *fp = NULL;

    if (memory == NULL)
    {
        fprintf(stderr, "ERROR: %s memory unitialized.\n", 
                (mode == PAYLOADMODE ? "Payload" : "Options"));
        return -1;
    }

    if (!strncmp(file, "-", 1))
    {
        fp = stdin;
        fd = fileno(fp);
    }
    else if ((fd = open(file, O_RDONLY)) < 0)
    {
        fprintf(stderr, "ERROR: Unable to open %s file: %s. %s\n",
                (mode == PAYLOADMODE) ? "Payload" : "Options", file, 
                strerror(errno));
        return -1;
    }

    if ((bytesread = read(fd, (void *)memory, maxsize)) < 0)
    {
        fprintf(stderr, "ERROR: Unable to read %s file: %s. %s\n",
                (mode == PAYLOADMODE) ? "Payload" : "Options", file, 
                strerror(errno));
        return -1;
    }
    return bytesread;
}

/****************************************************************************
 * Function: printmac(ETHERhdr eth)
 *
 * Purpose: Print the source and destination within the supplied ETHERhdr 
 *          struct in ascii form.
 *
 * Arguments: eth => ETHERhdr struct
 *
 * Returns: void function
 *
 ****************************************************************************/
void printmac(ETHERhdr eth)
{
    printf("               [MAC] %02X:%02X:%02X:%02X:%02X:%02X > ", 
            eth.ether_shost[0], eth.ether_shost[1], eth.ether_shost[2], 
            eth.ether_shost[3], eth.ether_shost[4], eth.ether_shost[5]);
    printf("%02X:%02X:%02X:%02X:%02X:%02X\n", eth.ether_dhost[0],
            eth.ether_dhost[1], eth.ether_dhost[2], eth.ether_dhost[3],
            eth.ether_dhost[4], eth.ether_dhost[5]);

    return;
}

/****************************************************************************
 * Function: printip(IPhdr ip);
 *
 * Purpose: Verbosely print portions of the IP header in ascii form.
 *
 * Arguments: ip => IPhdr struct
 *
 * Returns: void function
 *
 ****************************************************************************/
void printip(IPhdr ip)
{
    char *protoname = "Unknown";

    printf("                [IP] %s > %s\n", 
            strdup(inet_ntoa(ip.ip_src)), strdup(inet_ntoa(ip.ip_dst)));
    printf("             [IP ID] %hu\n", ip.ip_id);

    switch(ip.ip_p)
    {
        case 0:
            protoname = "IP";
            break;
        case 1:
            protoname = "ICMP";
            break;
        case 2:
            protoname = "IGMP";
            break;
        case 3:
            protoname = "GGP";
            break;
        case 4:
            protoname = "IP-ENCAP";
            break;
        case 5:
            protoname = "ST";
            break;
        case 6:
            protoname = "TCP";
            break;
        case 7:
            protoname = "UCL";
            break;
        case 8:
            protoname = "EGP";
            break;
        case 9:
            protoname = "IGP";
            break;
        case 10:
            protoname = "BBN-RCC-MON";
            break;
        case 11:
            protoname = "NVP-II";
            break;
        case 12:
            protoname = "PUP";
            break;
        case 13:
            protoname = "ARGUS";
            break;
        case 14:
            protoname = "EMCON";
            break;
        case 15:
            protoname = "XNET";
            break;
        case 16:
            protoname = "CHAOS";
            break;
        case 17:
            protoname = "UDP";
            break;
        case 18:
            protoname = "MUX";
            break;
        case 19:
            protoname = "DCN-MEAS";
            break;
        case 20:
            protoname = "HMP";
            break;
        case 21:
            protoname = "PRM";
            break;
        case 22:
            protoname = "XNS-IDP";
            break;
        case 23:
            protoname = "TRUNK-1";
            break;
        case 24:
            protoname = "TRUNK-2";
            break;
        case 25:
            protoname = "LEAF-1";
            break;
        case 26:
            protoname = "LEAF-2";
            break;
        case 27:
            protoname = "RDP";
            break;
        case 28:
            protoname = "IRTP";
            break;
        case 29:
            protoname = "ISO-TP4";
            break;
        case 30:
            protoname = "NETBLT";
            break;
        case 31:
            protoname = "MFE-NSP";
            break;
        case 32:
            protoname = "MERIT-INP";
            break;
        case 33:
            protoname = "SEP";
            break;
        case 34:
            protoname = "3PC";
            break;
        case 35:
            protoname = "IDPR";
            break;
        case 36:
            protoname = "XTP";
            break;
        case 37:
            protoname = "DDP";
            break;
        case 38:
            protoname = "IDPR-CMTP";
            break;
        case 39:
            protoname = "IDPR-CMTP";
            break;
        case 40:
            protoname = "IL";
            break;
        case 41:
            protoname = "IPv6";
            break;
        case 42:
            protoname = "SDRP";
            break;
        case 43:
            protoname = "SIP-SR";
            break;
        case 44:
            protoname = "SIP-FRAG";
            break;
        case 45:
            protoname = "IDRP";
            break;
        case 46:
            protoname = "RSVP";
            break;
        case 47:
            protoname = "GRE";
            break;
        case 48:
            protoname = "MHRP";
            break;
        case 49:
            protoname = "BNA";
            break;
        case 50:
            protoname = "IPSEC-ESP";
            break;
        case 51:
            protoname = "IPSEC-AH";
            break;
        case 52:
            protoname = "I-NLSP";
            break;
        case 53:
            protoname = "SWIPE";
            break;
        case 54:
            protoname = "NHRP";
            break;
        case 55:
            protoname = "MOBILEIP";
            break;
        case 57:
            protoname = "SKIP";
            break;
        case 58:
            protoname = "IPv6-ICMP";
            break;
        case 59:
            protoname = "IPv6-NoNxt";
            break;
        case 60:
            protoname = "IPv6-Opts";
            break;
        case 61:
            protoname = "any";
            break;
        case 62:
            protoname = "CFTP";
            break;
        case 63:
            protoname = "any";
            break;
        case 64:
            protoname = "SAT-EXPAK";
            break;
        case 65:
            protoname = "KRYPTOLAN";
            break;
        case 66:
            protoname = "RVD";
            break;
        case 67:
            protoname = "IPPC";
            break;
        case 68:
            protoname = "any";
            break;
        case 69:
            protoname = "SAT-MON";
            break;
        case 70:
            protoname = "VISA";
            break;
        case 71:
            protoname = "IPCV";
            break;
        case 72:
            protoname = "CPNX";
            break;
        case 73:
            protoname = "CPHB";
            break;
        case 74:
            protoname = "WSN";
            break;
        case 75:
            protoname = "PVP";
            break;
        case 76:
            protoname = "BR-SAT-MON";
            break;
        case 77:
            protoname = "SUN-ND";
            break;
        case 78:
            protoname = "WB-MON";
            break;
        case 79:
            protoname = "WB-EXPAK";
            break;
        case 80:
            protoname = "ISO-IP";
            break;
        case 81:
            protoname = "VMTP";
            break;
        case 82:
            protoname = "SECURE-VMTP";
            break;
        case 83:
            protoname = "VINES";
            break;
        case 84:
            protoname = "TTP";
            break;
        case 85:
            protoname = "NSFNET-IGP";
            break;
        case 86:
            protoname = "DGP";
            break;
        case 87:
            protoname = "TCF";
            break;
        case 88:
            protoname = "IGRP";
            break;
        case 89:
            protoname = "OSPFIGP";
            break;
        case 90:
            protoname = "Sprite-RPC";
            break;
        case 91:
            protoname = "LARP";
            break;
        case 92:
            protoname = "MTP";
            break;
        case 93:
            protoname = "AX.25";
            break;
        case 94:
            protoname = "IPIP";
            break;
        case 95:
            protoname = "MICP";
            break;
        case 96:
            protoname = "SCC-SP";
            break;
        case 97:
            protoname = "ETHERIP";
            break;
        case 98:
            protoname = "ENCAP";
            break;
        case 99:
            protoname = "any";
            break;
        case 100:
            protoname = "GMTP";
            break;
        case 103:
            protoname = "PIM";
            break;
        case 108:
            protoname = "IPComp";
            break;
        case 112:
            protoname = "VRRP";
            break;
        case 255:
            protoname = "Reserved";
            break;
    }
    printf("          [IP Proto] %s (%hu)\n", protoname, ip.ip_p);
    printf("            [IP TTL] %u\n", ip.ip_ttl);
    printf("            [IP TOS] 0x%02x\n", ip.ip_tos);
    printf("    [IP Frag offset] 0x%04x\n\n", ip.ip_off);

    return;
}

/****************************************************************************
 * Function: printtcp(TCPhdr tcp);
 *
 * Purpose: Verbosely print portions of the TCP header in ascii form.
 *
 * Arguments: tcp => TCPhdr struct
 *
 * Returns: void function
 *
 ****************************************************************************/
void printtcp(TCPhdr tcp)
{
        printf("         [TCP Ports] %hu > %hu\n", tcp.th_sport, tcp.th_dport);
        printf("         [TCP Flags] ");
        if (tcp.th_flags & TH_SYN)
            printf("SYN ");
        if (tcp.th_flags & TH_ACK)
            printf("ACK ");
        if (tcp.th_flags & TH_RST)
            printf("RST ");
        if (tcp.th_flags & TH_PUSH)
            printf("PSH ");
        if (tcp.th_flags & TH_URG)
            printf("URG ");
        if (tcp.th_flags & TH_FIN)
            printf("FIN ");
        printf("\n");
        printf("[TCP Urgent Pointer] %u\n", tcp.th_urp);
        printf("   [TCP Window Size] %u\n", tcp.th_win);
        if (tcp.th_flags & TH_ACK)
            printf("    [TCP Ack number] %lu\n", tcp.th_ack);
        if (tcp.th_flags & TH_SYN)
            printf("    [TCP Seq number] %lu\n", tcp.th_seq);

        return;
}

/****************************************************************************
 * Function: printicmp(ICMPhdr icmp);
 *
 * Purpose: Verbosely print portions of the ICMP header in ascii form.
 *
 * Arguments: icmp => ICMPhdr struct
 *
 * Returns: void function
 *
 ****************************************************************************/
void printicmp(ICMPhdr icmp, int mode)
{
    char *icmptype;
    char *icmpcode = "Unknown";

    switch (icmp.icmp_type)
    {
        case 0:
            icmptype = "Echo Reply";
            break;
        case 3:
            icmptype = "Destination Unreachable";
            switch (icmp.icmp_code)
            {
                case 0:
                    icmpcode = "Network Unreachable";
                    break;
                case 1:
                    icmpcode =  "Host Unreachable";
                    break;
                case 2:
                    icmpcode = "Protocol Unreachable";
                    break;
                case 3:
                    icmpcode = "Port Unreachable";
                    break;
                case 4:
                    icmpcode = "Fragmentation Needed";
                    break;
                case 5:
                    icmpcode = "Source Route Failed";
                    break;
                case 6:
                    icmpcode = "Destination Network Unknown";
                    break;
                case 7:
                    icmpcode = "Destination Host Unknown";
                    break;
                case 8:
                    icmpcode = "Source Host Isolated (obsolete)";
                    break;
                case 9:
                    icmpcode = "Destination Network Administratively "
                            "Prohibited";
                    break;
                case 10:
                    icmpcode = "Destination Host Administratively "
                            "Prohibited";
                    break;
                case 11:
                    icmpcode = "Network Unreachable For TOS";
                    break;
                case 12:
                    icmpcode = "Host Unreachable For TOS";
                    break;
                case 13: 
                    icmpcode = "Communication Administratively Prohibited "
                            "By Filtering";
                    break;
                case 14:
                    icmpcode = "Host Precedence Violation";
                    break;
                case 15:
                    icmpcode = "Precedence Cutoff In Effect";
                    break;
                }
            break;
        case 4:
            icmptype = "Source Quench";
            break;
        case 5:
            icmptype = "Redirect";
            switch (icmp.icmp_code)
            {
                case 0:
                    icmpcode = "Redirect For Network";
                    break;
                case 1:
                    icmpcode = "Redirect For Host";
                    break;
                case 2:
                    icmpcode = "Redirect For TOS and Network";
                    break;
                case 3:
                    icmpcode = "Redirect For TOS and Host";
                    break;
            }
            break;
        case 8:
            icmptype = "Echo Request";
            break;
        case 9:
            icmptype = "Router Advertisement";
            break;
        case 10:
            icmptype = "Router Solicitation";
            break;
        case 11:
            icmptype = "Time Exceeded";
            switch (icmp.icmp_code)
            {
                case 0:
                    icmpcode = "TTL = 0 During Transmit";
                    break;
                case 1:
                    icmpcode = "TTL = 0 During Reassembly";
                    break;
            }
            break;
        case 12:
            icmptype = "Parameter Problem";
            switch (icmp.icmp_code)
            {
                case 0:
                    icmpcode = "IP Header Bad (catchall error)";
                    break;
                case 1:
                    icmpcode = "Required Option Missing";
                    break;
            }
        case 13:
            icmptype = "Timestamp Request";
            break;
        case 14:
            icmptype = "Timestamp Reply";
            break;
        case 15:
            icmptype = "Information Request";
            break;
        case 16:
            icmptype = "Information Reply";
            break;
        case 17:
            icmptype = "Address Mask Request";
            break;
        case 18:
            icmptype = "Address Mask Reply";
            break;
        default:
            icmptype = "Unknown";
            break;
    }
    printf("         [ICMP Type] %s\n", icmptype);
    printf("         [ICMP Code] %s\n", icmpcode);

    if (mode == ECHO || mode == MASK || mode == TIMESTAMP)
    {
        printf("           [ICMP ID] %hu\n", icmp.hun.echo.id);
        printf("   [ICMP Seq number] %hu\n", icmp.hun.echo.seq);
    }
    if (mode == MASK)
        printf(" [ICMP Address Mask] %s\n", 
                inet_ntoa(*(struct in_addr *)&icmp.dun.mask));
    if (mode == REDIRECT)
    {
        printf(" [ICMP Pref Gateway] %s\n", 
                inet_ntoa(*(struct in_addr *)&icmp.hun.gateway));
    }

    return;
}

/****************************************************************************
 * Function: nemesis_name_resolve(u_char *hostname, u_int32_t *)
 *
 * Purpose: Convert a hostname or IP address, supplied in ascii format, to
 *          an u_int32_t in network byte order.
 *
 * Arguments: hostname => host name or IP address in ascii
 *            address => u_int32_t pointer to hold converted IP
 *
 * Returns: 0 on sucess, -1 on failure
 *
 ****************************************************************************/
int nemesis_name_resolve(char *hostname, u_int32_t *address)
{
    struct in_addr saddr;
    struct hostent *hp = NULL;
    extern int h_errno;

    if (address == NULL || hostname == NULL)
        return -1;

    if ((inet_aton(hostname, &saddr)) < 1)
    {
        if ((hp = gethostbyname(hostname)) == NULL)
        {
            fprintf(stderr, "ERROR: Unable to resolve supplied hostname: "
                    "%s. %s\n", hostname, hstrerror(h_errno));
            return -1;
        }
        /* Do not blindly disregard the size of the address returned */
        if (hp->h_length != 4)
        {
            fprintf(stderr, 
                    "ERROR: nemesis_name_resolve() received a non IPv4 "
                    "address.\n");
            return -1;
        }
        memcpy((u_int32_t *)address, hp->h_addr, 4);
        return 0;
    }
    else
    {
        if (!memcmp(&saddr.s_addr, zero, 4))
            return -1;

        memcpy((u_int32_t *)address, &saddr.s_addr, 4);
        return 0;
    }
}

/****************************************************************************
 * Function: nemesis_check_link(ETHERhdr *)
 *
 * Purpose: Determine if a source Ethernet address has been specified and
 *          fill in the ETHERhdr structure if necessary.
 *
 * Arguments: eth => ETHERhdr pointer containing the source Ethernet address
 *            device => char pointer containing the Ethernet device name
 *
 * Returns: 0 on sucess, -1 on failure
 *
 ****************************************************************************/
int nemesis_check_link(ETHERhdr *eth, char *device)
{
    int i;
    struct ether_addr *e;
    struct libnet_link_int *l2 = NULL;

    if (!memcmp(eth->ether_shost, zero, 6))
    {
        if ((e = libnet_get_hwaddr(l2, device, errbuf)) == NULL)
            return -1;

        for (i = 0; i < 6; i++)
            eth->ether_shost[i] = e->ether_addr_octet[i];
        if (!e)
        {
            return -1;
        }
        else
            return 0;
    } 
    else
        return 0;
}

/****************************************************************************
 * Function: nemesis_lookup_linktype(int linktype)
 *
 * Purpose: Lookup and return the string associated with each link type.
 *
 * Arguments: linktype => integer represntation of linktype
 *
 * Returns: char * containing the appropriate linktype or Unknown on a failed
 *          match.
 *
 ****************************************************************************/
char *nemesis_lookup_linktype(int linktype)
{
    char *dlt;

    switch(linktype)
    {
        case 0:
            dlt = "DLT_NULL";
            break;
        case 1:
            dlt = "DLT_EN10MB";
            break;
        case 2:
            dlt = "DLT_EN3MB";
            break;
        case 3:
            dlt = "DLT_AX25";
            break;
        case 4:
            dlt = "DLT_PRONET";
            break;
        case 5:
            dlt = "DLT_CHAOS";
            break;
        case 6:
            dlt = "DLT_IEEE802";
            break;
        case 7:
            dlt = "DLT_ARCNET";
            break;
        case 8:
            dlt = "DLT_SLIP";
            break;
        case 9:
            dlt = "DLT_PPP";
            break;
        case 10:
            dlt = "DLT_FDDI";
            break;
        case 11:
            dlt = "DLT_ATM_RFC1483";
            break;
        case 12:
            dlt = "DLT_LOOP";
            break;
        case 13:
            dlt = "DLT_ENC";
            break;
        case 14:
            dlt = "DLT_RAW";
            break;
        case 15:
            dlt = "DLT_SLIP_BSDOS";
            break;
        case 16:
            dlt = "DLT_PPP_BSDOS";
            break;
        default:
            dlt = "UNKNOWN";
    }
    return dlt;
}


/****************************************************************************
 * Function: nemesis_seedrand(void))
 *
 * Purpose: Generate a random IP address in the form of an u_int32_t.
 *
 * Arguments: none
 *
 * Returns: 0 on success, -1 on failure
 *
 ****************************************************************************/
int nemesis_seedrand(void)
{
    extern int errno;
    struct timeval tv;

    if (gettimeofday(&tv, NULL) == -1)
    {
        fprintf(stderr, "ERROR: nemesis_seedrand() failed in gettimeofday(): "
                "%s.\n", strerror(errno));
        return -1;
    }

    /* srandom((unsigned int)tv.tv_usec ^ (unsigned int)fabs(fmod(time(NULL), 
            UINT_MAX)) ^ (u_int32_t)getpid()); */

    //srandom((unsigned int)tv.tv_usec ^ getpid());

    srandom((unsigned int)tv.tv_usec ^ (unsigned int)fabs(fmod(time(NULL), 
            UINT_MAX)));

    return 0;
}

/****************************************************************************
 * Function: maketitle(char *title, char *module, char *version)
 *
 * Purpose: Build the title string for each nemesis protocol builder.
 *
 * Arguments: title => the buffer containing the concatenated title
 *            module => the name of the protocol builder module
 *            version => release version
 *
 * Returns: void function
 *
 ****************************************************************************/
void maketitle(char *title, const char *module, const char *version)
{
    char tmptitle[81], buildnum[13];


    strncpy(tmptitle, module, sizeof(tmptitle));
    tmptitle[sizeof(tmptitle) - 1] = '\0';
    strncat(tmptitle, version, sizeof(tmptitle) - 1 - strlen(tmptitle));
    snprintf(buildnum, sizeof(buildnum), " (Build %d)", BUILD);
    strncat(tmptitle, buildnum, sizeof(tmptitle) - 1 - strlen(tmptitle));

    memcpy(title, tmptitle, sizeof(tmptitle));
}

/****************************************************************************
 *
 * Function: gmt2local(time_t)
 *
 * Purpose: Figures out how to adjust the current clock reading based on the
 *          timezone you're in.  Ripped off from TCPdump.
 *
 * Arguments: time_t => offset from GMT
 *
 * Returns: offset seconds from GMT
 *
 ****************************************************************************/
int gmt2local(time_t t)
{
    register int dt, dir;
    register struct tm *gmt, *loc;
    struct tm sgmt;

    if(t == 0)
        t = time(NULL);

    gmt = &sgmt;
    *gmt = *gmtime(&t);
    loc = localtime(&t);

    dt = (loc->tm_hour - gmt->tm_hour) * 60 * 60 +
        (loc->tm_min - gmt->tm_min) * 60;

    dir = loc->tm_year - gmt->tm_year;

    if(dir == 0)
        dir = loc->tm_yday - gmt->tm_yday;

    dt += dir * 24 * 60 * 60;

    return(dt);
}
