/*
 * THE NEMESIS PROJECT
 * Copyright (C) 1999, 2000 Mark Grimes <obecian@packetninja.net>
 * Copyright (C) 2001 - 2003 Jeff Nathan <jeff@snort.org>
 *
 * nemesis-ospf.c (OSPF Packet Injector)
 *
 */

#include "nemesis-ospf.h"
#include "nemesis.h"

static ETHERhdr etherhdr;
static IPhdr iphdr;
static OSPFhdr ospfhdr;
static OSPFHELLOhdr ospfhellohdr;
static LSAhdr lsahdr;
static LSRhdr lsrhdr;
static LSUhdr lsuhdr;
static ASLSAhdr aslsahdr;
static RTRLSAhdr rtrlsahdr;
static DBDhdr dbdhdr;
static NETLSAhdr netlsahdr;
static SUMLSAhdr sumlsahdr;
static PayloadData pd;
static OptionsData ipod;

int nemesis_ospf(int argc, char **argv)
{
    int opt, i;
    u_int addr_tmp[6];
    char *payloadfile = NULL;       /* payload file name */
    char *ipoptionsfile = NULL;     /* TCP options file name */
    char *device = NULL;
    const char *module = "OSPF Packet Injection";
    extern char *optarg;
    extern int optind;
    //struct sockaddr_in sin;

    got_link = 0;
    got_options = 0;
    got_payload = 0;

    maketitle(title, module, version);

    if (argc < 2)
        ospf_usage(argv[0]);

    if ((i = libnet_seed_prand()) < 0)
        fprintf(stderr, "ERROR: Unable to seed random number generator.\n");

    /* defaults */
    etherhdr.ether_type = ETHERTYPE_IP;     /* Ethernet type IP */
    memset(etherhdr.ether_shost, 0, 6);     /* Ethernet source address */
    memset(etherhdr.ether_dhost, 0xff, 6);  /* Ethernet destination address */
    memset(&iphdr.ip_src.s_addr, 0, 4);     /* IP source address */
    memset(&iphdr.ip_dst.s_addr, 0, 4);     /* IP destination address */
    iphdr.ip_tos = IPTOS_LOWDELAY;          /* IP type of service */
    iphdr.ip_id = (u_int16_t)libnet_get_prand(PRu16);
                                            /* IP ID */
    iphdr.ip_off = 0;                       /* IP fragmentation offset */
    iphdr.ip_ttl = 255;                     /* IP TTL - set to 1 purposely */
    pd.payload = NULL;
    pd.payload_s = 0;
    ipod.options = NULL;
    ipod.options_s = 0;


    while ((opt = getopt(argc, argv, "A:B:d:D:f:F:g:G:H:i:k:I:l:L:m:M:n:N:o:O:p:P:r:R:s:S:t:T:u:x:y:z:v?")) != -1)
    {
        switch (opt)
        {
            case 'A':   /* OSPF area ID */
                if ((nemesis_name_resolve(optarg, 
                        &ospfhdr.ospf_area_id.s_addr)) < 0)
                addaid = strtoul(optarg, NULL, 0);
                {
                    fprintf(stderr, "ERROR: Invalid OSPF area ID IP address: "
                            "\"%s\".\n", optarg);
                    exit(1);
                }
                break;
            case 'B':   /* Number of broadcasted LSAs */
                lsuhdr.lsu_num = (u_int)htonl(xgetint32(optarg));
                break;
            case 'd':   /* Ethernet device */
                if (strlen(optarg) < 256)
                {
                    device = strdup(optarg);
                    got_link = 1;
                }
                else
                {
                    fprintf(stderr, "ERROR: device %s > 256 characters.\n",
                            device);
                    exit(1);
                }
                break;
            case 'D':   /* destination IP address */
                if ((nemesis_name_resolve(optarg, &iphdr.ip_dst.s_addr)) < 0)
                {
                    fprintf(stderr, "ERROR: Invalid destination IP address: "
                            "\"%s\".\n", optarg);
                    exit(1);
                }
                break;
            case 'f':   /* external AS LSA forwarding IP address */
                if ((nemesis_name_resolve(optarg, 
                        &aslsahdr.as_fwd_addr.s_addr)) < 0)
                {
                    fprintf(stderr, "ERROR: Invalid external LSA forwarding IP "
                            "address: \"%s\".\n", optarg);
                    exit(1);
                }
                break;
            case 'F':   /* fragmentation offset */
                iphdr.ip_off = xgetint16(optarg) & IP_OFFMASK;
                break;
            case 'g':   /* OSPF external route tag */
                aslsahdr.as_rte_tag = (u_int32_t)htonl(xgetint32(optarg));
                break;
            case 'G':   /* OSPF link state acknowledgment age in seconds */
                lsahdr.lsa_age = (u_int16_t)htons(xgetint16(optarg));
                break;
            case 'H':   /* Ethernet source address */
                memset(addr_tmp, 0, sizeof(addr_tmp));
                sscanf(optarg, "%02X:%02X:%02X:%02X:%02X:%02X", &addr_tmp[0],
                        &addr_tmp[1], &addr_tmp[2], &addr_tmp[3], &addr_tmp[4],
                        &addr_tmp[5]);
                for (i = 0; i < 6; i++)
                    enet_src[i] = (u_char)addr_tmp[i];
                break;
            case 'i':   /* OSPF HELLO link state countdown timer in seconds */
                ospfhellohdr.hello_dead_intvl = (u_int16_t)htonl(xgetint32(optarg));
                break;
            case 'I':   /* IP ID */
                iphdr.ip_id = xgetint16(optarg);
                break;
            case 'k':   /* OSPF link state acknowledgment link data */
                rtrlsahdr.rtr_link_data = (u_int16_t)htonl(xgetint32(optarg));
                break;
            case 'l':   /* OSPF HELLO las packet interval in seconds */
                ospfhellohdr.hello_intrvl = (u_int16_t)htons(xgetint16(optarg));
                break;
            case 'L':   /* OSPF link state request ID */
                lsrhdr.lsr_lsid = (u_int32_t)htonl(xgetint32(optarg));
                break;
            case 'm':   /* OSPF link state acknowledgment link metric */
                rtrlsahdr.rtr_metric = (u_int16_t)htons(xgetint16(optarg));
                break;
            case 'M':   /* Ethernet destination address */
                memset(addr_tmp, 0, sizeof(addr_tmp));
                sscanf(optarg, "%02X:%02X:%02X:%02X:%02X:%02X", &addr_tmp[0],
                        &addr_tmp[1], &addr_tmp[2], &addr_tmp[3], &addr_tmp[4],
                        &addr_tmp[5]);
                for (i = 0; i < 6; i++)
                    enet_dst[i] = (u_char)addr_tmp[i]; 
                break;
            case 'n':   /* OSPF multi-purpose netmask placement */
                if ((nemesis_name_resolve(optarg, 
                        &ospfhellohdr.hello_nmask.s_addr)) < 0)
                {
                    fprintf(stderr, "ERROR: Invalid netmask IP address: \"%s\""
                            ".\n", optarg);
                    exit(1);
                }
                else
                {
                    memcpy(&ospfhellohdr.hello_nmask.s_addr, 
                            &netlsahdr.net_nmask.s_addr, sizeof(u_int));
                    memcpy(&ospfhellohdr.hello_nmask.s_addr, 
                            &sumlsahdr.sum_nmask.s_addr, sizeof(u_int));
                    memcpy(&ospfhellohdr.hello_nmask.s_addr, 
                            &aslsahdr.as_nmask.s_addr, sizeof(u_int));
                }
                break;
            case 'N':   /* OSPF HELLO neighbor router */
                if ((nemesis_name_resolve(optarg, 
                        &ospfhellohdr.hello_nbr.s_addr)) < 0)
                {
                    fprintf(stderr, "ERROR: Invalid OSPF HELLO neighbor "
                            "IP address: \"%s\".\n", optarg);
                    exit(1);
                }
                break;
            case 'o':   /* OSPF multi purpose options */
                ospfhellohdr.hello_opts = xgetint8(optarg);
                lsahdr.lsa_opts = xgetint8(optarg);
                dbdhdr.dbd_opts = xgetint8(optarg);
                break;
            case 'O':   /* IP options file */
                if (strlen(optarg) < 256)
                {
                    ipoptionsfile = strdup(optarg);
                    got_ipoptions = 1;
                }
                else
                {
                    fprintf(stderr, "ERROR: IP options file %s > 256 "
                            "characters.\n", optarg);
                    exit(1);
                }
                break;
            case 'p':
                switch (*optarg)
                {
                    case 'H':
                        ospftype = 0;
                        break;
                    case 'D':
                        ospftype = 1;
                        break;
                    case 'L':
                        ospftype = 2;
                        break;
                    case 'U':
                        ospftype = 3;
                        break;
                    case 'R':
                        ospftype = 4;
                        break;
                    case 'N':
                        ospftype = 5;
                        break;
                    case 'M':
                        ospftype = 6;
                        break;
                    case 'A':
                        ospftype = 7;
                        break;
                }
                break;
            case 'P':   /* payload file */
                if (strlen(optarg) < 256)
                {
                    payloadfile = strdup(optarg);
                    got_payload = 1;
                }
                else
                {
                    fprintf(stderr, "ERROR: payload file %s > 256 "
                            "characters.\n", payloadfile);
                    exit(1);
                }
                break;
            case 'r':
                if (!(router = libnet_name_resolve(optarg, 0)))
                {
                    fprintf(stderr, "Invalid advertising router address: \"%s\""
                            ".\n", optarg);
                    exit(1);
                }
                if (verbose)
                    printf("[Advertising Router IP] %s\n", optarg);
                break;
            case 'R':
                addrid = strtoul(optarg, NULL, 0);
                break;
            case 's':
                seqnum = atoi(optarg);
                break;
            case 'S':   /* source IP address */
                if ((nemesis_name_resolve(optarg, &iphdr.ip_src.s_addr)) < 0)
                {
                    fprintf(stderr, "ERROR: Invalid source IP address: \"%s\"."
                            "\n", optarg);
                    exit(1);
                }
                break;
            case 't':   /* IP type of service */
                iphdr.ip_tos = xgetint8(optarg);
                break;
            case 'T':   /* IP time to live */
                iphdr.ip_ttl = xgetint8(optarg);
                break;
            case 'u':
                num = atoi(optarg);
                break;
            case 'v':
                verbose++;
                if (verbose == 1)
                {
                    putchar('\n');
                    puts(title);
                    putchar('\n');
                }
                break;
            case 'x':
                exchange = strtoul(optarg, NULL, 0);
                break;
            case 'y':
                rtrtype = strtoul(optarg, NULL, 0);
                break;
            case 'z':
                mtusize = atoi(optarg);
                break;
            case '?':   /* FALLTHROUGH */
            default:
                ospf_usage(argv[0]);
                break;
        }
    }
    argc -= optind;
    argv += optind;

    /* sanity checks */
    if (source == 0 || dest == 0)
    {
        printf("Source and/or Destination Address Missing.\n");
        exit(1);
    }

    if (got_link && !device)
    {
        printf("Unspecified Device.\n");
        exit(1);
    }

    if (ospftype == -1)
    {
        printf("OSPF Packet type not supplied.\n");
        exit(1);
    }

    if (got_link)
    {
        if (enet_src[0] == 0 && enet_src[1] == 0 && enet_src[2] == 0 &&
                enet_src[3] == 0 && enet_src[4] == 0 && enet_src[5] == 0)
        {
            e = libnet_get_hwaddr(l2, device, errbuf);
            for (i = 0; i < 6; i++)
                enet_src[i] = e->ether_addr_octet[i];
            if (!e)
            {
                fprintf(stderr, "cannot retrieve hardware address of %s: %s\n",
                        device, errbuf);
                exit(1);
            }
        }
    }


    if (verbose)
    {
        if (got_link)
        {
            printf("[MAC]  ");
            printf("%02X:%02X:%02X:%02X:%02X:%02X > ", enet_src[0], enet_src[1],
                    enet_src[2], enet_src[3], enet_src[4], enet_src[5]);
            printf("%02X:%02X:%02X:%02X:%02X:%02X\n", enet_dst[0], enet_dst[1], 
                    enet_dst[2], enet_dst[3], enet_dst[4], enet_dst[5]);
        }
        printf("[OSPF Options] 0x%x\n", ooptions);
        printf("[Priority] %d\n", priority);
        printf("[Advertising Router ID] 0x%ld\n", addrid);
        printf("[Advertising Area ID] 0x%ld\n", addaid);

        if (ospftype == 1)
        {
            printf("[Dead router interval] %d\n", dead_int);
        } else if (ospftype == 2)
        {
            printf("[Netmask] %ld\n", mask);
            printf("[Sequence Number] %d\n", seqnum);
            printf("[Router Advertisement Age] %d\n", ospf_age);
            printf("[Link State ID] %d\n", rtrid);
        } else if (ospftype == 3)
        {
            printf("[Link State ID]	%d\n", rtrid);
        }
        printf("[IP ID] %d\n", id);
        printf("[IP TTL] %d\n", ttl);
        printf("[IP TOS] 0x%x\n", tos);
        printf("[IP Frag] 0x%x\n", frag);
        printf("[IP Options] %s\n", options);
    }
#if 0
    if (got_payload)
    {
        payload = buff;
        while ((nbytes = read(fd, buff, sizeof(buff))) != 0)
        {
            payload_s += nbytes;
            if (verbose)
            {
                hexdump(payload, payload_s);
                if (buildospf() != -1)
                {
                    if (verbose)
                        printf("\nOSPF Packet Injected\n");
                } else
                {
                    if (verbose)
                        printf("\nOSPF Injection Failure\n");
                }
            }
        }
        close(fd);
        exit(0);
    }
#endif
    if (buildospf() != -1)
    {
        if (verbose)
            printf("\nUDP Packet Injected\n");
    } else
    {
        if (verbose)
            printf("\nUDP Injection Failure\n");
    }
    //close(fd);
    exit(0);
}

void ospf_usage(char *arg)
{
    putchar('\n');
    puts(title);
    putchar('\n');

    printf("OSPF usage:\n  %s [-v] [options]\n\n", arg);
    printf("OSPF Packet Types: \n"
           "  -p <OSPF Protocol>\n"
           "     -pH HELLO, -pD DBD, -pL LSR, -pU LSU, -pR LSA (router),\n"
           "     -pN LSA (network), -pM LSA (summary), -pA LSA (AS)\n");
    printf("OSPF HELLO options: \n"
           "  -N <Neighbor Router Address>\n"
           "  -i <Dead Router Interval>\n"
           "  -l <OSPF Interval>\n");
    printf("OSPF DBD options: \n"
           "  -z <MAX DGRAM Length>\n"
           "  -x <Exchange Type>\n");
    printf("OSPF LSU options: \n"
           "  -B <num of LSAs to bcast>\n");
    printf("OSPF LSA related options: \n"
           "  -L <router id>\n"
           "  -G <LSA age>\n");
    printf("OSPF LSA_RTR options: \n"
           "  -u <LSA_RTR num>\n"
           "  -y <LSA_RTR router type>\n"
           "  -k <LSA_RTR router data>\n");
    printf("OSPF LSA_AS_EXT options: \n"
           "  -f <LSA_AS_EXT forward address>\n"
           "  -g <LSA_AS_EXT tag>\n");
    printf("OSPF options: \n"
           "  -m <OSPF Metric>\n"
           "  -s <Sequence Number>\n"
           "  -r <Advertising Router Address>\n"
           "  -n <OSPF Netmask>\n"
           "  -o <OSPF Options>\n"
           "  -R <OSPF Router id>\n"
           "  -A <OSPF Area id>\n"
           "  -P <Payload File (Binary or ASCII)>\n"
           "  (-v VERBOSE - packet struct to stdout)\n\n");
    printf("IP Options\n"
           "  -S <Source Address>\n"
           "  -D <Destination Address>\n"
           "  -I <IP ID>\n"
           "  -T <IP TTL>\n"
           "  -t <IP/OSPF tos>\n"
           "  -F <IP frag>\n"
           "  -O <IP options file>\n\n");
    printf("Data Link Options: \n"
           "  -d <Ethernet Device>\n"
           "  -H <Source MAC Address>\n"
           "  -M <Destination MAC Address>\n\n");
    printf("You must define a Source, Destination, Protocol & its dependent"
           " options.\n");
    exit(1);
}

void defaults()
{
    memset(enet_src, 0, 6);
    memset(enet_dst, 0xff, 6);


    id = 0;
    tos = IPTOS_LOWDELAY | IPTOS_THROUGHPUT;
    ttl = 254;
    mask = 0xffffff00;
    addrid = 0xff00ff00;
    addaid = 0xd00dd00d;
    rtrid = 42;
    ospf_age = 40;
    ooptions = 0x00;
    priority = 0x00;
    dead_int = 30;
    payload = NULL;
    payload_s = 0;
    *options = NULL;
    option_s = 0;
    interval = 2;
    seqnum = 420;
    frag = IP_DF;
    ospftype = -1;
}
