/*
 * THE NEMESIS PROJECT
 * Copyright (C) 1999, 2000 Mark Grimes <obecian@packetninja.net>
 * Copyright (C) 2001 - 2003 Jeff Nathan <jeff@snort.org>
 *
 * nemesis-rip.c (RIP Packet Injector)
 *
 */

#include "nemesis-rip.h"
#include "nemesis.h"

static ETHERhdr etherhdr;
static IPhdr iphdr;
static UDPhdr udphdr;
static RIPhdr riphdr;
static PayloadData pd;
static OptionsData ipod;
static int got_payload;
static char *payloadfile = NULL;       /* payload file name */
static char *ipoptionsfile = NULL;     /* IP options file name */
static char *device = NULL;            /* Ethernet device */

int nemesis_rip(int argc, char **argv)
{
    const char *module = "RIP Packet Injection";

    maketitle(title, module, version);

#if 0
    if (argc < 2)
        rip_usage(argv[0]);
#endif
    if (argc > 1 && !strncmp(argv[1], "help", 4))
        rip_usage(argv[0]);

    if (nemesis_seedrand() < 0)
        fprintf(stderr, "ERROR: Unable to seed random number generator.\n");

    rip_initdata();

    rip_cmdline(argc, argv);    

    rip_validatedata();

    rip_verbose();

    if (got_payload)
    {
        if  ((pd.payload = calloc(RIPBUFFSIZE, sizeof(char))) == NULL)
        {
            perror("ERROR: Unable to allocate packet payload memory");
            exit(1);
        }

        if ((pd.payload_s = readfile(pd.payload, payloadfile, RIPBUFFSIZE, 
                PAYLOADMODE)) < 0)
        {
            fprintf(stderr, "ERROR: readfile() failure.\n");
            exit(1);
        }
    }

    if (got_ipoptions)
    {
        if ((ipod.options = calloc(OPTIONSBUFFSIZE, sizeof(char))) == NULL)
        {
            perror("ERROR: Unable to allocate IP options memory");
            exit(1);
        }

        if ((ipod.options_s = readfile(ipod.options, ipoptionsfile, 
                    OPTIONSBUFFSIZE, OPTIONSMODE)) < 0)
        {
            fprintf(stderr, "ERROR: readfile() failure.\n");
            exit(1);
        }
    }

    if (buildrip(&etherhdr, &iphdr, &udphdr, &riphdr, &pd, &ipod, device) < 0)
    {
        if (verbose)
            printf("\nRIP Injection Failure\n");
        
        if (got_payload)
            free(pd.payload);

        if (got_ipoptions)
            free(ipod.options);

        exit(1);
    }
    else
    {
        if (verbose)
            printf("\nRIP Packet Injected\n");
        
        if (got_payload)
            free(pd.payload);

        if (got_ipoptions)
            free(ipod.options);

        exit(0);
    }
}

void rip_initdata(void)
{
    /* defaults */
    etherhdr.ether_type = ETHERTYPE_IP;     /* Ethernet type IP */
    memset(etherhdr.ether_shost, 0, 6);     /* Ethernet source address */
    memset(etherhdr.ether_dhost, 0xff, 6);  /* Ethernet destination address */
    memset(&iphdr.ip_src.s_addr, 0, 4);     /* IP source address */
    memset(&iphdr.ip_dst.s_addr, 0, 4);     /* IP destination address */
    iphdr.ip_tos = IPTOS_RELIABILITY;       /* IP type of service */
    iphdr.ip_id = (u_int16_t)libnet_get_prand(PRu16);
                                            /* IP ID */
    iphdr.ip_p = IPPROTO_UDP;               /* IP protocol UDP */
    iphdr.ip_off = 0;                       /* IP fragmentation offset */
    iphdr.ip_ttl = 255;                     /* IP TTL */
    udphdr.uh_sport = 520;                  /* UDP source port */
    udphdr.uh_dport = 520;                  /* UDP destination port */
    riphdr.cmd = RIPCMD_REQUEST;            /* RIP command */
    riphdr.ver = 2;                         /* RIP version */
    riphdr.rd = 0;                          /* RIP routing domain */
    riphdr.af = 0;                          /* RIP address family */
    riphdr.rt = 0;                          /* RIP route tage */
    riphdr.addr = 0;                        /* RIP address */
    riphdr.mask = 0;                        /* RIP subnet mask */
    riphdr.next_hop = 0;                    /* RIP next-hop IP address */
    riphdr.metric = 0;                      /* RIP metric */
    pd.payload = NULL;
    pd.payload_s = 0;
    ipod.options = NULL;
    ipod.options_s = 0;
}

void rip_validatedata(void)
{
    struct sockaddr_in sin;

    /* validation tests */
    if (iphdr.ip_src.s_addr == 0)
        iphdr.ip_src.s_addr = (u_int32_t)libnet_get_prand(PRu32);
    if (iphdr.ip_dst.s_addr == 0)
        iphdr.ip_dst.s_addr = (u_int32_t)libnet_get_prand(PRu32);

    /* if the user has supplied a source hardware addess but not a device
     * try to select a device automatically
     */
    if (memcmp(etherhdr.ether_shost, zero, 6) && !got_link && !device)
    {
        if (libnet_select_device(&sin, &device, (char *)&errbuf) < 0)
        {
            fprintf(stderr, "ERROR: Device not specified and unable to "
                    "automatically select\n a device.\n");
            exit(1);
        }
        else
        {
            got_link = 1;
        }
    }

    /* if a device was specified and the user has not specified a source 
     * hardware address, try to determine the source address automatically
     */
    if (got_link)
    {
        if ((nemesis_check_link(&etherhdr, device)) < 0)
        {
            fprintf(stderr, "ERROR: cannot retrieve hardware address of %s.\n",
                    device);
            exit(1);
        }
    }

    /* Attempt to send valid packets if the user hasn't decided to craft an
     * anomolous packet
     */
}

void rip_usage(char *arg)
{
    putchar('\n');
    puts(title);
    putchar('\n');

    printf("RIP usage:\n  %s [-v (verbose)] [options]\n\n", arg);
    printf("RIP options: \n"
           "  -c <RIP Command>\n"
           "  -V <RIP Version>\n"
           "  -r <RIP Route Domain>\n"
           "  -a <RIP Address Family>\n"
           "  -R <RIP Route Tag>\n"
           "  -k <RIP Network Address Mask>\n"
           "  -h <RIP Next Hop>\n"
           "  -m <RIP metric>\n"
           "  -P <Payload file>\n\n");
    printf("UDP options:\n"
           "  -x <Source port>\n"
           "  -y <Destination port>\n\n");
    printf("IP options: \n"
           "  -S <Source IP address>\n"
           "  -D <Destination IP address>\n"
           "  -I <IP ID>\n"
           "  -T <IP TTL>\n"
           "  -t <IP TOS>\n"
           "  -F <IP fragmentation offset>\n"
           "  -O <IP options file>\n\n");
    printf("Data Link Options: \n"
           "  -d <Ethernet device>\n"
           "  -H <Source MAC address>\n"
           "  -M <Destination MAC address>\n\n");
    exit(1);
}

void rip_cmdline(int argc, char **argv)
{   
    int opt, i;
    u_int32_t addr_tmp[6];
    char *rip_options;
    extern char *optarg;
    extern int optind;

#if defined(ENABLE_PCAPOUTPUT)
    rip_options = "a:c:d:D:F:h:H:i:I:k:m:M:O:P:r:R:S:t:T:V:x:y:v?";
#else
    rip_options = "a:c:d:D:F:h:H:i:I:k:m:M:O:P:r:R:S:t:T:V:x:y:v?";
#endif

    while ((opt = getopt(argc, argv, rip_options)) != -1)
    {
        switch (opt)
        {
            case 'a':   /* RIP address family */
                riphdr.af = xgetint16(optarg);
                break;
            case 'c':   /* RIP command */
                riphdr.cmd = xgetint8(optarg);
                break;
            case 'd':    /* Ethernet device */
                if (strlen(optarg) < 256)
                {
                    device = strdup(optarg);
                    got_link = 1;
                }
                else
                {
                    fprintf(stderr, "ERROR: device %s > 256 characters.\n", 
                            optarg);
                    exit(1);
                }
                break;
            case 'D':    /* destination IP address */
                if ((nemesis_name_resolve(optarg, &iphdr.ip_dst.s_addr)) < 0)
                {
                    fprintf(stderr, "ERROR: Invalid destination IP address: "
                            "\"%s\".\n", optarg);
                    exit(1);
                }
                break;
            case 'F':    /* IP fragmentation offset */
                iphdr.ip_off = xgetint16(optarg) & IP_OFFMASK;
                break;
            case 'h':   /* RIP next hop address */
                if ((nemesis_name_resolve(optarg,
                        (u_int32_t *)&riphdr.next_hop)) < 0)
                {
                    fprintf(stderr, "ERROR: Invalid next hop IP address: "
                            "\"%s\".\n", optarg);
                    exit(1);
                }
                break;
            case 'H':    /* Ethernet source address */
                memset(addr_tmp, 0, sizeof(addr_tmp));
                sscanf(optarg, "%02X:%02X:%02X:%02X:%02X:%02X", &addr_tmp[0],
                        &addr_tmp[1], &addr_tmp[2], &addr_tmp[3], &addr_tmp[4],
                        &addr_tmp[5]);
                for (i = 0; i < 6; i++)
                    etherhdr.ether_shost[i] = (u_int8_t)addr_tmp[i];
                break;
            case 'i':    /* RIP IP address */
                if ((nemesis_name_resolve(optarg, 
                        (u_int32_t *)&riphdr.addr)) < 0)
                {
                    fprintf(stderr, "ERROR: Invalid destination IP address: "
                            "\"%s\".\n", optarg);
                    exit(1);
                }
                break;
            case 'I':   /* IP ID */
                iphdr.ip_id = xgetint16(optarg);
                break;
            case 'k':   /* RIP mask IP address */
                if ((nemesis_name_resolve(optarg, 
                        (u_int32_t *)&riphdr.mask)) < 0)
                {
                    fprintf(stderr, "ERROR: Invalid RIP mask IP address: "
                            "\"%s\".\n", optarg);
                    exit(1);
                }
                break;
            case 'm':   /* RIP metric */
                riphdr.metric = xgetint32(optarg);
                break;
            case 'M':    /* Ethernet destination address */
                memset(addr_tmp, 0, sizeof(addr_tmp));
                sscanf(optarg, "%02X:%02X:%02X:%02X:%02X:%02X", &addr_tmp[0],
                        &addr_tmp[1], &addr_tmp[2], &addr_tmp[3], &addr_tmp[4],
                        &addr_tmp[5]);
                for (i = 0; i < 6; i++)
                    etherhdr.ether_dhost[i] = (u_int8_t)addr_tmp[i]; 
                break;
            case 'O':   /* IP options file */
                if (strlen(optarg) < 256)
                {
                    ipoptionsfile = strdup(optarg);
                    got_ipoptions = 1;
                }
                else
                {
                    fprintf(stderr, "ERROR: IP options file %s > 256 "
                            "characters.\n", optarg);
                    exit(1);
                }
                break;
            case 'P':   /* payload file */
                if (strlen(optarg) < 256)
                {
                    payloadfile = strdup(optarg);
                    got_payload = 1;
                }
                else
                {
                    fprintf(stderr, "ERROR: payload file %s > 256 "
                            "characters.\n", optarg);
                    exit(1);
                }
                break;
            case 'r':   /* RIP routing domain */
                riphdr.rd = xgetint16(optarg);
                break;
            case 'R':   /* RIP route tag */
                riphdr.rt = xgetint16(optarg);
                break;
            case 'S':    /* source IP address */
                if ((nemesis_name_resolve(optarg, &iphdr.ip_src.s_addr)) < 0)
                {
                    fprintf(stderr, "ERROR: Invalid source IP address: \"%s\"."
                            "\n", optarg);
                    exit(1);
                }
                break;
            case 't':   /* IP type of service */
                iphdr.ip_tos = xgetint8(optarg);
                break;
            case 'T':   /* IP time to live */
                iphdr.ip_ttl = xgetint8(optarg);
                break;
            case 'v':
                verbose++;
                if (verbose == 1)
                {
                    putchar('\n');
                    puts(title);
                    putchar('\n');
                }
                break;
            case 'V':   /* RIP version */
                riphdr.ver = xgetint8(optarg);
                break;
            case 'x':   /* UDP source port */
                udphdr.uh_sport = xgetint16(optarg);
                break;
            case 'y':   /* UDP destination port */
                udphdr.uh_dport = xgetint16(optarg);
                break;
            case '?':   /* FALLTHROUGH */
            default:
                rip_usage(argv[0]);
                break;
        }
    }
    argc -= optind;
    argv += optind;
}

void rip_verbose(void)
{
    if (verbose)
    {
        if (got_link)
            printmac(etherhdr);

        printip(iphdr);

        printf("         [UDP Ports] %hu > %hu\n", udphdr.uh_sport, 
                udphdr.uh_dport);
        putchar('\n');

        switch(riphdr.cmd)
        {
            case RIPCMD_REQUEST:
                printf("        [RIP Command] Request (%c)\n", riphdr.cmd);
                break;
            case RIPCMD_RESPONSE:
                printf("        [RIP Command] Response (%c)\n", riphdr.cmd);
                break;
            case RIPCMD_TRACEON:
                printf("        [RIP Command] Tracing on (%c)\n", riphdr.cmd);
                break;
            case RIPCMD_TRACEOFF:
                printf("        [RIP Command] Tracing off (%c)\n", riphdr.cmd);
                break;
            case RIPCMD_POLL:
                printf("        [RIP Command] Poll (%c)\n", riphdr.cmd);
                break;
            case RIPCMD_POLLENTRY:
                printf("        [RIP Command] Poll entry (%c)\n", riphdr.cmd);
                break;
            case RIPCMD_MAX:
                printf("        [RIP Command] Max (%c)\n", riphdr.cmd);
                break;
        }
        printf("        [RIP Version] %c\n", riphdr.ver);
        printf(" [RIP Routing Domain] %hu\n", riphdr.rd);
        printf(" [RIP Address Family] %hu\n", riphdr.af);
        printf("      [RIP Route Tag] %hu\n", riphdr.rt);
        printf("        [RIP Address] %s\n", 
                strdup(inet_ntoa(*(struct in_addr *)&riphdr.addr)));
        printf("   [RIP Network Mask] %s\n", 
                strdup(inet_ntoa(*(struct in_addr *)&riphdr.mask)));

        printf("       [RIP Next Hop] %s\n", 
                strdup(inet_ntoa(*(struct in_addr *)&riphdr.next_hop)));
        printf("         [RIP Metric] %u\n", (u_int32_t)riphdr.metric);
    }
}
