/* protocols.c
	Suuport functions for sendip.c
*/

#define __USE_BSD     /* GLIBC */
#define _BSD_SOURCE   /* LIBC5 */
#define __FAVOR_BSD   /* Some of GLIBC */
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip6.h>
#include <netinet/ip_icmp.h>
#include <netinet/icmp6.h>
#include <netinet/udp.h>
#include <netinet/tcp.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "csum.h"
#include "rip.h"

#ifndef __GNU_LIBRARY__
#define __GNU_LIBRARY__ 42
#endif
#if __GNU_LIBRARY__ < 6
#define icmp     icmphdr
#define icmp_type type
#define icmp_code code
#define icmp_cksum checksum
#define uh_sport source
#define uh_dport dest
#define uh_ulen  len
#define uh_sum   check
#endif

/* Handy macros to set the TCP flags */
#define SETRES(tcp,i) tcp->th_flags |= (i)<<6;

/* For the CWR and ECN bits see rfc2481 
   The symbols TH_CWR and TH_ECN haven't been seen in netinet/tcp.h
   yet, but the names are a hopefully good guess.
*/
#ifndef TH_CWR
#define TH_CWR 0x40
#endif
#ifndef TH_ECN
#define TH_ECN 0x80
#endif
#define SETCWR(tcp,i) (i)?(tcp->th_flags|=TH_CWR):(tcp->th_flags&=~TH_CWR);
#define SETECN(tcp,i) (i)?(tcp->th_flags|=TH_ECN):(tcp->th_flags&=~TH_ECN);

#define SETURG(tcp,i) (i)?(tcp->th_flags|=TH_URG):(tcp->th_flags&=~TH_URG);
#define SETACK(tcp,i) (i)?(tcp->th_flags|=TH_ACK):(tcp->th_flags&=~TH_ACK);
#define SETPSH(tcp,i) (i)?(tcp->th_flags|=TH_PUSH):(tcp->th_flags&=~TH_PUSH);
#define SETRST(tcp,i) (i)?(tcp->th_flags|=TH_RST):(tcp->th_flags&=~TH_RST);
#define SETFIN(tcp,i) (i)?(tcp->th_flags|=TH_FIN):(tcp->th_flags&=~TH_FIN);
#define SETSYN(tcp,i) (i)?(tcp->th_flags|=TH_SYN):(tcp->th_flags&=~TH_SYN);

extern void usage_exit(const char *err);

void ip_initialize(struct ip *iph);
void ip6_initialize(struct ip6_hdr *hdr);
void icmp_initialize(struct icmp *icp);
void icmp6_initialize(struct icmp6_hdr *icmp6);
void tcp_initialize(struct tcphdr *tcp);
void udp_initialize(struct udphdr *udp);

void do_ipopt(struct ip *iph, char *opt, char *arg);
void do_ip6opt(struct ip6_hdr *hdr, char *opt, char *arg);
void do_icmpopt(struct icmp *icp, char *opt, char *arg);
void do_icmp6opt(struct icmp6_hdr *icmp6, char *opt, char *arg);
void do_tcpopt(struct tcphdr *tcp, unsigned char *tcpopts, char *opt, char *arg);
void do_udpopt(struct udphdr *udp, char *opt, char *arg);
void do_ripopt(struct rip_options *rippack, char *opt, char **arg);

void ip_set_addr(struct ip *iph, struct sockaddr_in *to);
void ip6_set_addr(struct ip6_hdr *hdr, struct sockaddr_in6 *to);
void ip_finalize(struct ip *iph, int length, struct sockaddr_in *to);
void ip6_finalize(struct ip6_hdr *hdr, int length, struct sockaddr_in6 *to);
void icmp_finalize(struct ip *iph, struct icmp *icp, int length);
void icmp6_finalize(struct ip6_hdr *hdr, struct icmp6_hdr *icmp6, int length);
void tcp_finalize(struct ip *iph, struct tcphdr *tcp, int length);
void udp_finalize(struct ip *iph, struct udphdr *udp, int length);
void rip_finalize(struct rip_options *rippack, unsigned char *dataStart);

void ip_initialize(struct ip *iph) {
	iph->ip_v=     (unsigned int)0x4;
	iph->ip_hl=    (unsigned int)0x5;
	iph->ip_tos=       (u_int8_t)0x00;
	iph->ip_len=      (u_int16_t)0xFFFF;        /* automatic */
	iph->ip_id=       (u_int16_t)rand();
	iph->ip_off=      (u_int16_t)0x0000;
	iph->ip_ttl=       (u_int8_t)0xFF;
	iph->ip_p=         (u_int8_t)0xFF;          /* automatic */
	iph->ip_sum=      (u_int16_t)0xFFFF;        /* automatic */
	iph->ip_src.s_addr=          INADDR_NONE;   /* automatic */
	iph->ip_dst.s_addr=          INADDR_NONE;   /* automatic */
}

void ip6_initialize(struct ip6_hdr *hdr) {
	hdr->ip6_flow = 0;
	hdr->ip6_vfc = 0x6 << 4; /* version */
	hdr->ip6_plen = (u_int16_t)0xFFFF;
	hdr->ip6_nxt = (u_int8_t)0xFF; /* automatic */
	hdr->ip6_hlim = 32;
	hdr->ip6_src = in6addr_any;
	hdr->ip6_dst = in6addr_any;
}

void icmp_initialize(struct icmp *icp) {
	icp->icmp_type=              ICMP_ECHO;
	icp->icmp_code=    (u_int8_t)0x00;
	icp->icmp_cksum=  (u_int16_t)0xFFFF;        /* automatic */
}

void icmp6_initialize(struct icmp6_hdr *icmp6) {
	icmp6->icmp6_type = ICMP6_ECHO_REQUEST;
	icmp6->icmp6_code = (u_int8_t)0x00;
	icmp6->icmp6_cksum = (u_int16_t)0xFFFF;     /* automatic */
}

void tcp_initialize(struct tcphdr *tcp) {
	tcp->th_sport =   (u_int16_t)0x0000;
	tcp->th_dport =   (u_int16_t)0x0000;
	tcp->th_seq =     (u_int32_t)rand();
	tcp->th_ack =     (u_int32_t)0x00000000;
	tcp->th_off =     (u_int16_t)0x5;
	tcp->th_x2 =      (u_int16_t)0x0;
	tcp->th_flags =   (u_int16_t)TH_SYN;
	tcp->th_win =     (u_int16_t)65535;
	tcp->th_sum =     (u_int16_t)0xFFFF;  /* automatic */
	tcp->th_urp =     (u_int16_t)0x0000;
}

void udp_initialize(struct udphdr *udp) {
	udp->uh_sport =     (u_int16_t)0x0000;
	udp->uh_dport =       (u_int16_t)0x0000;
	udp->uh_ulen =        (u_int16_t)0xFFFF;        /* automatic */
	udp->uh_sum =      (u_int16_t)0xFFFF;        /* automatic */
}

void do_ipopt(struct ip *iph, char *opt, char *arg) {
	char rb[32];
	if(*arg == 'r') {
		sprintf(rb,"%d",rand());
		arg = rb;
	}
	switch(opt[2]) {
	case 'h':
		iph->ip_hl = (unsigned int)strtoul(arg, (char **)NULL, 0) & 0xF;
		break;
	case 'y':
		iph->ip_tos = (u_int8_t)strtoul(arg, (char **)NULL, 0);
		break;
	case 'l':
		iph->ip_len = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 'i':
		iph->ip_id = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 't':
		iph->ip_ttl = (u_int8_t)strtoul(arg, (char **)NULL, 0);
		break;
	case 'p':
	   iph->ip_p = (u_int8_t)strtoul(arg, (char **)NULL, 0);
		break;
	case 'c':
		iph->ip_sum = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 's':
		iph->ip_src.s_addr = inet_addr(arg);
		break;
	case 'd':
		iph->ip_dst.s_addr = inet_addr(arg);
		break;
	case 'f':
		switch(opt[3]) {
		case 'r':
			if(arg != rb && *arg != '0' && *arg != '1') {
				usage_exit("IP reserved flag must be 0 or 1\n");
			}
			*arg-='0';
			if(*arg) {
				iph->ip_off |= (u_int16_t)0x8000;
			} else {
				iph->ip_off &= (u_int16_t)0x7FFF;
			}
			break;
		case 'd':
			if(arg != rb && *arg != '0' && *arg != '1') {
				usage_exit("IP don't fragment flag must be 0 or 1\n");
			}
			*arg-='0';
			if(*arg) {
				iph->ip_off |= (u_int16_t)0x4000;
			} else {
				iph->ip_off &= (u_int16_t)0xBFFF;
			}
			break;
		case 'm':
			if(arg != rb && *arg != '0' && *arg != '1') {
				usage_exit("IP more fragments flag must be 0 or 1\n");
			}
			*arg-='0';
			if(*arg) {
				iph->ip_off |= (u_int16_t)0x2000;
			} else {
				iph->ip_off &= (u_int16_t)0xDFFF;
			}
			break;
		case '\0':
			iph->ip_off &= (u_int16_t)0xE000;
			iph->ip_off |= (u_int16_t)strtoul(arg, (char **)NULL, 0) & 
				(u_int16_t)0x1FFF;
			break;
		default:
			usage_exit("unknown IP flag\n");
			break;
		}
		break;
	default:
		usage_exit("unknown IP option\n");
		break;
	}
}

void do_ip6opt(struct ip6_hdr *hdr, char *opt, char *arg) 
{
	struct in6_addr addr;

	switch(opt[2]) {

	case 'f':
		hdr->ip6_flow |= htonl((u_int32_t)strtoul(arg, (char **)NULL, 0) & 0xFFF00000);
		break;
	case 'v':
		hdr->ip6_vfc &= 0x0F;
		hdr->ip6_vfc |= (u_int8_t)strtoul(arg, (char **)NULL, 0) << 4;
		break;
	case 't':
		hdr->ip6_flow |= htonl(((u_int32_t)strtoul(arg, (char **)NULL, 0) << 20) & 0x0F000000);
		break;
	case 'l':
		hdr->ip6_plen = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 'h':
		hdr->ip6_hlim = (u_int8_t)strtoul(arg, (char **)NULL, 0);
		break;
	case 'n':
		hdr->ip6_nxt = (u_int8_t)strtoul(arg, (char **)NULL, 0);
		break;
	case 's':
		if (inet_pton(AF_INET6, arg, &addr)) {
			memcpy(&hdr->ip6_src, &addr, sizeof(struct in6_addr));
		}
		break;
	case 'd':
		if (inet_pton(AF_INET6, arg, &addr)) {
			memcpy(&hdr->ip6_dst, &addr, sizeof(struct in6_addr));
		}
		break;
	default:
		usage_exit("unknown IPv6 option\n");
		break;
	}
}

void do_icmpopt(struct icmp *icp, char *opt, char *arg) {
	char rb[32];
	if(*arg == 'r') {
		sprintf(rb,"%d",rand());
		arg = rb;
	}	switch(opt[2]) {
	case 't':
		icp->icmp_type = (u_int8_t)strtoul(arg, (char **)NULL, 0);
		break;
	case 'd':
		icp->icmp_code = (u_int8_t)strtoul(arg, (char **)NULL, 0);
		break;
	case 'c':
		icp->icmp_cksum = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	default:
		usage_exit("unknown ICMP option\n");
		break;
	}
}

void do_icmp6opt(struct icmp6_hdr *icmp6, char *opt, char *arg) 
{
	char rb[32];
	if(*arg == 'r') {
		sprintf(rb,"%d",rand());
		arg = rb;
	}
	switch (opt[2]) {
	case 't':
		icmp6->icmp6_type = (u_int8_t)strtoul(arg, (char **)NULL, 0);
		break;
	case 'd':
		icmp6->icmp6_code = (u_int8_t)strtoul(arg, (char **)NULL, 0);
		break;
	case 'c':
		icmp6->icmp6_cksum = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	default:
		usage_exit("unknown ICMPv6 option\n");
		break;
	}
}

void do_tcpopt(struct tcphdr *tcp, unsigned char *tcpopts, char *opt, 
char *arg) {
	char rb[32];
	if(*arg == 'r') {
		sprintf(rb,"%d",rand());
		arg = rb;
	}
	switch(opt[2]) {
	case 's':
		tcp->th_sport = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 'd':
		tcp->th_dport = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 'n':
		tcp->th_seq = htonl((u_int32_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 'a':
		tcp->th_ack = htonl((u_int32_t)strtoul(arg, (char **)NULL, 0));
		SETACK(tcp,1);
		break;
	case 't':
		tcp->th_off = (u_int16_t)strtoul(arg, (char **)NULL, 0) &0xF;
		break;
	case 'r':
		tcp->th_x2 = (u_int16_t)(strtoul(arg, (char **)NULL, 0) & 0x003C)>>2;
		SETRES(tcp,(u_int16_t)strtoul(arg, (char **)NULL, 0) & 0x0003);
		break;
	case 'f':
		if(arg != rb && *arg != '0' && *arg != '1') {
			usage_exit("TCP flag must be 0 or 1\n");
		}
		*arg-=0x30;
		switch(opt[3]) {
		case 'e': /* ECN - see rfc2481 */
			SETECN(tcp,(u_int16_t)*arg&1);
			break;
		case 'c': /* CWR - see rfc2481 */
			SETCWR(tcp,(u_int16_t)*arg&1);
			break;
		case 'u':
			SETURG(tcp,(u_int16_t)*arg&1);
			break;
		case 'a':
			SETACK(tcp,(u_int16_t)*arg&1);
			break;
		case 'p':
			SETPSH(tcp,(u_int16_t)*arg&1);
			break;
		case 'r':
			SETRST(tcp,(u_int16_t)*arg&1);
			break;
		case 's':
			SETSYN(tcp,(u_int16_t)*arg&1);
			break;
		case 'f':
			SETFIN(tcp,(u_int16_t)*arg&1);
			break;
		default:
			usage_exit("TCP flage not known\n");
			break;
		}
		break;
	case 'w':
		tcp->th_win = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 'c':
		tcp->th_sum = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 'u':
		tcp->th_urp = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		SETURG(tcp,1);
		break;
        /* @@@@ tcp options options ***/
        case 'o':
                { int optunknown=1;
          	  if (!strcmp(opt+3, "eol"))
                  { /* End of options list RFC 793 kind 0, no length */
                    tcpopts[++*tcpopts]=0; 
                    optunknown=0; }
        	  if (!strcmp(opt+3, "nop"))
                  { /* No op RFC 793 kind 1, no length */
                    tcpopts[++*tcpopts]=1; 
                    optunknown=0; }
        	  if (!strcmp(opt+3, "mss"))
                  { /* Maximum segment size RFC 793 kind 2 */
                    unsigned mss=atoi(arg);
                    tcpopts[++*tcpopts]=2; 
                    tcpopts[++*tcpopts]=4; 
                    tcpopts[++*tcpopts]=(mss>>8)&0xff; 
                    tcpopts[++*tcpopts]=mss&0xff; 
                    optunknown=0;
                  } 
        	  if (!strcmp(opt+3, "wscale"))
                  { /* Window scale rfc1323 */
                    unsigned mss=atoi(arg);
                    tcpopts[++*tcpopts]=3; 
                    tcpopts[++*tcpopts]=3; 
                    tcpopts[++*tcpopts]=mss&0xff; 
                    optunknown=0;
                  } 
        	  if (!strcmp(opt+3, "sackok"))
                  { /* Selective Acknowledge permitted rfc1323 */
                    tcpopts[++*tcpopts]=4; 
                    tcpopts[++*tcpopts]=2; 
                    optunknown=0;
                  }
        	  if (!strcmp(opt+3, "sack"))
                  { /* Selective Acknowledge rfc1323 */
                    unsigned char *lenptr, *next;
                    unsigned long le, re;
                    tcpopts[++*tcpopts]=5; 
                    *(lenptr=tcpopts+(++*tcpopts))=2; 
                    next=arg;
                    while (*next)
                    { 
                      /* get left edge */
                      next=strchr(arg, ':');
                      if (!next) 
                      { fprintf(stderr, 
                         "Value in tcp sack option incorrect. Usage: \n");
                        fprintf(stderr, 
                        " -tosack left:right[,left:right...]\n");
                        exit(2); }
                      *next++=0;
                      le=atoi(arg);
                      arg=next;
                      /* get right edge */
                      next=strchr(arg, ',');
                      if (!next) 
                        next=arg-1; /* Finito - next points to \0 */ 
                      else
                        *next++=0;
                      re=atoi(arg);
                      arg=next;
                      
                      le=htonl(le);
                      re=htonl(re);
                      memcpy(tcpopts+*tcpopts+1, &le, 4);
                      memcpy(tcpopts+*tcpopts+5, &re, 4);
                      *lenptr+=8;
                      *tcpopts+=8;
                    }
                    optunknown=0;
                  } 
        	  if (!strcmp(opt+3, "ts"))
        	  if (!strcmp(opt+3, "ts"))
                  { /* Timestamp rfc1323 */
                    unsigned long tsval=0, tsecr=0;
                    if (2!=sscanf(arg, "%ld:%ld", &tsval, &tsecr))
                    { fprintf(stderr, 
                                "Invalid value for tcp timestamp option.\n");
                      fprintf(stderr, 
                                "Usage: -tots tsval:tsecr\n");
                      exit(2);
                    }
                    tcpopts[++*tcpopts]=8; 
                    tcpopts[++*tcpopts]=10; 
                    tsval=htonl(tsval);
                    memcpy(tcpopts+(++*tcpopts), &tsval, 4);
                    *tcpopts+=3;
                    tsecr=htonl(tsecr);
                    memcpy(tcpopts+(++*tcpopts), &tsecr, 4);
                    *tcpopts+=3;
                    optunknown=0;
                  } 
                  if (optunknown)
                  { fprintf(stderr, "unksupported TCP Option %s val %s\n", 
                                     opt, arg);
                    exit(2); } 
                }
                break;
	default:
		usage_exit("unknown TCP options\n");
		break;
	}
}

void do_udpopt(struct udphdr *udp, char *opt, char *arg) {
	char rb[32];
	if(*arg == 'r') {
		sprintf(rb,"%d",rand());
		arg = rb;
	}
	switch(opt[2]) {
	case 's':
		udp->uh_sport = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 'd':
		udp->uh_dport = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 'l':
		udp->uh_ulen = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	case 'c':
		udp->uh_sum = htons((u_int16_t)strtoul(arg, (char **)NULL, 0));
		break;
	default:
		usage_exit("unknown UDP options\n");
		break;
	}
}

void do_ripopt(struct rip_options *rippack, char *opt, char **arg)
{
	static int visited = 0;
	static int howMany = 0;
	if( !visited)
	{
		rippack->version = 2;
		rippack->command = 1;
		rippack->authenticate = 0x0;
	}
	if( howMany == 25)
		fprintf( stderr, "Warning. A real RIP packet contains no more than 25 entries.\n");

	++visited;
	
	switch( opt[2])
	{
	case 'v': /* version */
		rippack->version = (uint8_t)strtoul( arg[0], (char**)0, 0);
		break;

	case 'c': /* command */
		rippack->command = (uint8_t)strtoul( arg[0], (char**)0, 0);
		break;

	case 'e': /* rip entry */
		rippack->addressFamily = arg[0][0] == '-' ? 2 : (uint16_t)strtoul( arg[0], (char**)0, 0);
		rippack->routeTagOrAuthenticationType = arg[0][0] == '-' ? 0 : (uint16_t)strtoul( arg[1], (char**)0, 0);
		rippack->address = arg[0][0] == '-' ? inet_addr( "0.0.0.0") : inet_addr( arg[2]);
		rippack->subnetMask = arg[0][0] == '-' ? inet_addr( "255.255.255.0") : inet_addr( arg[3]);
		rippack->nextHop = arg[0][0] == '-' ? inet_addr( "0.0.0.0") : inet_addr( arg[4]);
		rippack->metric = arg[0][0] == '-' ? 16 : (uint32_t)strtoul( arg[5], (char**)0, 0);
		++howMany;
		break;

	case 'a': /* authenticate */
		if( howMany != 0)
			fprintf( stderr, "Warning. A real RIP-2 packet only has authentication in the first entry.\n");
		
		rippack->authenticate = 0x1;
		if( ! isalpha( arg[0][0]))
			strcpy( rippack->password, "Brave new world");
		else
			strncpy( rippack->password, arg[0], 16);
		rippack->routeTagOrAuthenticationType = 0x02;
		rippack->addressFamily = 0xFFFF;
		++howMany;
		break;
	case 'd': /* default request */
		break;
	default:
		break;
	}
}


void ip_set_addr(struct ip *iph, struct sockaddr_in *to) {
	if(iph->ip_src.s_addr == INADDR_NONE)
		iph->ip_src.s_addr = inet_addr("127.0.0.1");
	if(iph->ip_dst.s_addr == INADDR_NONE)
		iph->ip_dst.s_addr = to->sin_addr.s_addr;
}

void ip6_set_addr(struct ip6_hdr *hdr, struct sockaddr_in6 *to) {
	if (IN6_IS_ADDR_UNSPECIFIED(&hdr->ip6_src))
		hdr->ip6_src = in6addr_loopback;
	if (IN6_IS_ADDR_UNSPECIFIED(&hdr->ip6_dst))
		memcpy(&hdr->ip6_dst, &to->sin6_addr, sizeof(struct in6_addr));
}

void ip_finalize(struct ip *iph, int length, struct sockaddr_in *to) {
	if(iph->ip_len==(u_int16_t)0xFFFF) iph->ip_len = htons(length);
	if(iph->ip_p == (u_int8_t)0xFF) iph->ip_p=0;
	if(iph->ip_sum == (u_int16_t)0xFFFF) {
		iph->ip_sum = 0;
		iph->ip_sum = csum((unsigned short *)iph, sizeof(struct ip));
	}
}

void ip6_finalize(struct ip6_hdr *hdr, int length, struct sockaddr_in6 *to) {
	if (hdr->ip6_plen == (u_int16_t)0xFFFF) 
		hdr->ip6_plen = htons(length - sizeof(struct ip6_hdr));
	if (hdr->ip6_nxt == (u_int8_t)0xFF)
		hdr->ip6_nxt = IPPROTO_NONE;
}

void icmp_finalize(struct ip *iph, struct icmp *icp, int length) {
	if(iph->ip_p == (u_int8_t)0xFF) iph->ip_p = IPPROTO_ICMP;
	if(icp->icmp_cksum == (u_int16_t)0xFFFF) {
		icp->icmp_cksum = 0;
		icp->icmp_cksum = csum((unsigned short *)icp,
									  length-sizeof(struct ip));
	}
}

void icmp6_finalize(struct ip6_hdr *hdr, struct icmp6_hdr *icmp6, int length) 
{
	if (hdr->ip6_nxt == (u_int8_t)0xFF)
		hdr->ip6_nxt = IPPROTO_ICMPV6;
	if (icmp6->icmp6_cksum == (u_int16_t)0xFFFF) {
		icmp6->icmp6_cksum = 0;
		icmp6->icmp6_cksum = csum((unsigned short *)icmp6,
					  length - sizeof(struct ip6_hdr));
	}
}


void tcp_finalize(struct ip *iph, struct tcphdr *tcp, int length) {
	if(iph->ip_p == (u_int8_t)0xFF) iph->ip_p = IPPROTO_TCP;
	if(tcp->th_sum == (u_int16_t)0xFFFF) {
		tcp->th_sum = 0;
		tcp->th_sum = tcpcsum((unsigned char *)iph, (unsigned short *)tcp,
									 length-sizeof(struct ip));
	}
}

void udp_finalize(struct ip *iph, struct udphdr *udp, int length) {
	if(udp->uh_ulen==(u_int16_t)0xFFFF)
		udp->uh_ulen = htons(length-sizeof(struct ip));
	if(iph->ip_p == (u_int8_t)0xFF) iph->ip_p = IPPROTO_UDP;
	if(udp->uh_sum == (u_int16_t)0xFFFF) {
		udp->uh_sum = 0;
		udp->uh_sum = udpcsum((unsigned char *)iph, (unsigned short *)udp,
									 length-sizeof(struct ip));
	}
}

void rip_finalize( struct rip_options *rippack, unsigned char *dataStart)
{
	uint16_t *us;
	us = (uint16_t*)(dataStart+4);
	*us = htons( rippack->addressFamily);
	us = (uint16_t*)(dataStart+6);
	*us = htons( rippack->routeTagOrAuthenticationType);

	if( !rippack->authenticate)
	{
		uint32_t *ul;
		ul = (uint32_t*)(dataStart+8);
		*ul = rippack->address;
		ul = (uint32_t*)(dataStart+12);
		*ul = rippack->subnetMask;
		ul = (uint32_t*)(dataStart+16);
		*ul = rippack->nextHop;
		ul = (uint32_t*)(dataStart+20);
		*ul = htonl( rippack->metric);
	}
	else
		memcpy( dataStart+8, rippack->password, strlen( rippack->password));	

/*
// Compiler flakiness means that having this line as the first line of this function
// causes parsing errors in the uintxx_t* type declarations!! -- egcs 2.91.66
*/
	sprintf( dataStart, "%c%c%c%c", rippack->command, rippack->version, '\0', '\0');
}


#if 0
void rip_finalize( struct rip_options *rippack, unsigned char *dataStart)
{
	sprintf( dataStart, "%c%c%c%c", rippack->command, rippack->version, '\0', '\0');
	*(dataStart+4) = ((char*)&rippack->addressFamily)[1]; /* little endian .... */
	*(dataStart+5) = ((char*)&rippack->addressFamily)[0];
	*(dataStart+6) = ((char*)&rippack->routeTagOrAuthenticationType)[1];
	*(dataStart+7) = ((char*)&rippack->routeTagOrAuthenticationType)[0];
	if( !rippack->authenticate)
	{
		int i;
  		for( i=0; i<4; ++i)
  		{
/* we are little endian so ... */
  			*(dataStart+8+i) = ((char*)&rippack->address)[i];
  			*(dataStart+12+i) = ((char*)&rippack->subnetMask)[i];
  			*(dataStart+16+i) = ((char*)&rippack->nextHop)[i];
  			*(dataStart+20+i) = ((char*)&rippack->metric)[3-i];
  		} 
	}
	else
		memcpy( dataStart+8, rippack->password, strlen( rippack->password));
}
#endif
