/*********************************************************************
  Hide for PGP:
  a steganographic program that hides files in BMP, WAV or VOC files.
  If the hidden file resembles random noise - e.g. files encrypted with
  Pretty Good Privacy by Phil Zimmermann and without any header - e.g.
  the header information removed by Stealth by Henry Hastur - the hidden
  information cannot be detected.

  copyright 1996 - 2000 by Heinz Repp
  last modified: 02/20/00

  This program has been developed with PGP 2.6 and Stealth in mind so it
  supports the stream mode of Stealth: the file to be hidden or extracted
  must be provided or read through a pipe or stdin/stdout redirection.

  usage:
  Hide4PGP [-v|-q] [-nf] [-nc] [-u|-nu] <mediafile> [<file-to-hide>]
  Hide4PGP -x [-v|-q] <mediafile>  [<extracted-file>]
  Hide4PGP -i <mediafile>
  Hide4PGP -h


  parameters may be in any order (but first file is always the media file)
  and switches combined

  !!  for non UNIX systems either USE_FREOPEN or USE_SETMODE must be
  !!  defined (see Hide4PGP.Doc / Source for details)

*********************************************************************/

/* includes */
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <sys\utime.h>
#if defined (USE_SETMODE)
  #include <fcntl.h>
  #include <io.h>
#endif
#include "hide4pgp.h"
#include "utils.h"

#define BytesForLength(x) (x < 64 ? 8 : x < 41024 ? 16 : x < 1089600 ? 24 : 32)

/*********************************************************************
  hello:  show initial prompt and copyright
*********************************************************************/

void hello (void)
{
  fputs ("Hide4PGP v2.0 - hides secret files in BMP/WAV/VOC\n"
         "(c) 1996-2000 by Heinz Repp\n\n", stderr);
}


/*********************************************************************
  usage:  show syntax
*********************************************************************/

void usage (void)
{
  hello ();
  ErrorExit (NOARGUMENTS,
    "usage:\n"
    "    Hide4PGP [-v|-q] [-nf] [-nc] [-u|-nu] <mediafile> [<file-to-hide>]\n"
    "or: Hide4PGP -x [-v|-q] <mediafile> [<extracted-file>]\n"
    "or: Hide4PGP -i <mediafile>\n"
    "\n"
    "    Hide4PGP -h for help.\n");
}


/*********************************************************************
  help:  show detailed help
*********************************************************************/

void help (void)
{
  hello ();
  ErrorExit (HELPNOACTION,
    " Info about a multimedia file and its hiding capacity:\n"
    "    Hide4PGP -i <mediafile>\n"
    " Hide secret data in a multimedia file:\n"
    "    Hide4PGP [-options] <mediafile> <secret.in>\n"
    "   or using pipes:\n"
    "    <other program> | Hide4PGP [-options] <mediafile>\n"
    " Extract secret data out of a multimedia file:\n"
    "    Hide4PGP -x [-options] <mediafile> <secret.out>\n"
    "   or using pipes:\n"
    "    Hide4PGP -x [-options] <mediafile> | <other program>\n"
    "\n"
    " Options may appear anywhere on the command line and be combined:\n"
    "  -x  eXtract (default is hiding)    -v  Verbose: detailed report\n"
    "  -i  Info about file and capacity   -q  Quiet: only fatal errors\n"
    "  -h  (or ?) Help (this screen)\n"
    "                                         'NOT'-options:\n"
    "                                    -nf  do Not Fake File modification time\n"
    " For 256 color bitmaps (BMP) only:  -nc  do Not Change palette Colors\n"
    "  -u  use more UnUsed pal. entries  -nu  do Not use any UnUsed pal. entries\n");
}


/*********************************************************************
  main:       parses the command line, reads the first byte in the
              StegFile, decides which file type to assume, and then
              does most of the work by vectorized calls to type and
              data width specific routines
  parameters: one filename (required) and optional switches
*********************************************************************/

int main (int argc, char *argv[])
{
  UWORD32 (*NextBlock) (void);
  void    (*EditBlock) (UWORD32);
  int     c, MediaFilePos = 0, SecretFilePos = 0, fakedate = TRUE;
  UWORD32 block;
  struct stat StegStats;

  if (argc < 2)
    usage ();

  /* parse command line */
  while (--argc)
  {
    switch (*argv[argc])
    {
      case '-':
      case '/':
        while (c = *++argv[argc])
        {
          switch (toupper (c))
          {
            case '?':
            case 'H':
              help ();
              break;

            case 'I':
              verbose = 3;
              hiding = FALSE;
              break;

            case 'N':
              c = *++argv[argc];
              switch (toupper (c))
              {
                case 'C':
                  smooth = FALSE;
                  break;

                case 'F':
                  fakedate = FALSE;
                  break;

                case 'U':
                  fillup = 0;
                  break;

                default:
                 if (verbose) hello ();
                  ErrorExit (INVALIDSWITCH,
                             "Error: Not a valid negation switch.\n");
              }
              break;

            case 'U':
              fillup = 2;
              break;

            case 'Q':
              verbose = 0;
              break;

            case 'V':
              verbose = 2;
              break;

            case 'X':
              hiding = FALSE;
              break;

            default:
              if (verbose) hello ();
              ErrorExit (INVALIDSWITCH, "Error: Invalid switch.\n");
          }
        }
        break;

      default:
        if (! MediaFilePos)
          MediaFilePos = argc;
        else                         /* media filename already found */
          if (! SecretFilePos)
          {
            SecretFilePos = MediaFilePos;
            MediaFilePos = argc;
          }
          else                       /* secret filename also already found */
          {
            if (verbose) hello ();
            ErrorExit (THIRDFILENAME,
              "Error: More than two filenames specified.\n");
          }
    }
  }

  if (verbose) hello ();

  if (! MediaFilePos)
    ErrorExit (NOFILENAME, "Error: Required filename missing.\n");

  /* open media file for steganography */
  StegFile = fopen (argv[MediaFilePos], hiding ? "r+b" : "rb");
  if (StegFile == NULL)
  {
    fputs ("Error: Cannot open <", stderr);
    fputs (argv[MediaFilePos], stderr);
    verbose = 0;
    ErrorExit (OPENERROR, "> for steganography.\n");
  }

  if (hiding && fakedate)
    fstat (fileno (StegFile), &StegStats);

  /* determine data file type from first byte in file */
  c = GetStegByte ();
  if (!(c == 'B' && CheckIfBMP (&NextBlock) ||
        c == 'C' && CheckIfVOC (&NextBlock) ||
        c == 'R' && CheckIfWAV (&NextBlock)))
    ErrorExit (UNKNOWNTYPE, "Error: File format not recognized.\n");

  if (verbose == 3)
  {
    StegLength = MediaLength * MaxBitsAllowed / 8;
    fprintf (stderr, "You may hide up to %d bytes in <%s>.\n",
             (MediaLength - BytesForLength (StegLength))
             * MaxBitsAllowed / 8, argv[MediaFilePos]);
    verbose = 0;

    goto SkipStegSection;     /* instead of a huge 'else' section */
  }

  if (SecretFilePos)
  {
    /* open secret file for steganography */
    SecretFile = fopen (argv[SecretFilePos], hiding ? "rb" : "wb");
    if (SecretFile == NULL)
    {
      fputs ("Error: Cannot open secret data file <", stderr);
      fputs (argv[SecretFilePos], stderr);
      verbose = 0;
      ErrorExit (OPENERROR, ">.\n");
    }
  }
  else
  {
  /* set stdin/stdout to binary */
#if ! defined (UNIX)
#if defined (USE_FREOPEN)
    if (hiding)
      freopen ("", "rb", stdin);
    else
      freopen ("", "wb", stdout);
#elif defined (USE_SETMODE)
    setmode (fileno (hiding ? stdin : stdout), O_BINARY);
#else
  #error You MUST define a macro to choose the method \
         of setting stdin/out to binary (see doc)!
#endif
#endif
    if (! hiding)
      SecretFile = stdout;
  }

  if (hiding)
  {
    if (SecretFilePos)
    {
      fseek (SecretFile, 0L, SEEK_END);
      StegLength = ftell (SecretFile);
    }
    else
    {
      if (verbose > 1)
      {
        fputs ("Reading data to hide ... ", stderr);
        fflush (stderr);
      }

      /* open temporary file for secret data read from stdin */
      if ((SecretFile = tmpfile ()) == NULL)
        ErrorExit (TEMPERROR, "Error: Unable to open temporary file");

      /* Read and count secret data from stdin to temporary file */
      while ((c = getchar ()) != EOF)
      {
        putc (c, SecretFile);
        StegLength++;
      }

      if (verbose > 1)
        fputs ("Done.\n", stderr);
    }

    /* check capacity */
    if (StegLength > (MediaLength - BytesForLength (StegLength))
                     * MaxBitsAllowed / 8)
      ErrorExit (CAPACITYEX, "Error: Capacity of media file exhausted.\n");

    StegLeft = StegLength;
    EditBlock = SetBlock;
    NextByte = GetLengthByte;
  }
  else
  {
    EditBlock = GetBlock;
    OutputByte = LengthOut;
  }

  if (verbose > 1)
  {
    fputs (hiding ? "Hiding ... " : "Unveiling ... ", stderr);
    fflush (stderr);
  }

  /* main workhorse loop */
  while (block = (*NextBlock) ())
    (*EditBlock) (block);

  /* clean up and summarize */

  fclose (SecretFile);
SkipStegSection:
  fclose (StegFile);

  if (hiding && fakedate)
    utime (argv[MediaFilePos], (struct utimbuf *) &StegStats.st_atime);

  if (StegLeft)
    UnexpEOF ();

  if (verbose)
  {
    if (verbose > 1)
      fprintf (stderr, "Done.\n%ld Bytes %s file <%s>.\n",
               StegLength,
               hiding ? "hidden in" : "recovered from",
               argv[MediaFilePos]);
    fputs ("Steganographic operation finished.\n", stderr);
  }

  return 0;
}
