/*********************************************************************
  Utility routines

  part of: Hide4PGP by Heinz Repp
    last modified: 02/23/00
*********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include "hide4pgp.h"
#include "utils.h"
#include "scramble.h"


/* globals */
int hiding = TRUE, verbose = 1;              /* command line options */
FILE    *StegFile;                           /* data file to work on */
FILE    *SecretFile;            /* (temporary) file with secret data */
WORD32  StegLength = 0;             /* length of hidden/to hide data */
WORD32  StegLeft = 0;        /* part of StegLength not yet processed */
WORD32  MediaLength = 0;        /* data points available for storage */
size_t  DataSize;                            /* bytes per data point */
int     MaxBitsAllowed;                 /* bits/data point available */
int     LastLength = FALSE;    /* flag: last length byte in progress */
void    (*ModData) (void *, size_t);     /* funct ptr: modify buffer */
unsigned int (*GetBits) (int);
                      /* funct ptr: extracts bit(s) from data points */
UBYTE   (*NextByte) (void);  /* funct ptr: fetches next byte to hide */
void    (*OutputByte) (UBYTE);  /* funct ptr: delivers restored byte */

/* static variables */
        /* initialize bit distribution algorithm to 1 bit/data point */
static long IndicatorSum = 0, BitsNeeded = 1, BytesAvail = 1;


/*********************************************************************
  ErrorExit: called on critical errors; prints error message and aborts
  parameter: Programm Exit Code, Pointer to Message
  returns:   never
*********************************************************************/

void ErrorExit (int ExitCode, char *ErrorMessage)
{
  if (verbose > 1)
    fputs ("\n", stderr);

  fputs (ErrorMessage, stderr);
  exit (ExitCode);
}


/*********************************************************************
  UnexpEOF: is called when read past EOF, outputs an error message
             and aborts
  parameter: none
  returns:   never
*********************************************************************/

void UnexpEOF (void)
{
  ErrorExit (UNEXPEOF,
   "Error: Unexpected end of file encountered - aborting.\n");
}


/*********************************************************************
  NoMemory:  is called when malloc fails or SetBlock ends up with a
             zero byte block
  parameter: none
  returns:   never
*********************************************************************/

void NoMemory (void)
{
  ErrorExit (OUTOFMEMORY, "Error: Not enough memory - aborting.\n");
}


/*********************************************************************
  GetStegByte: reads 8 bit from media file
  parameter: none
  returns:   next byte
*********************************************************************/

UBYTE GetStegByte (void)
{
  int b;

  if ((b = getc (StegFile)) == EOF)
    UnexpEOF ();

  return (UBYTE) b;
}


/*********************************************************************
  GetStegWord: reads one 16 bit word from media file
  parameter: none
  returns:   next two bytes
*********************************************************************/

UWORD16 GetStegWord (void)
{
  UWORD16 value;

  if (fread (&value, 2, 1, StegFile) != 1)
    UnexpEOF ();

  return value;
}


/*********************************************************************
  GetStegLong: reads one 32 bit word from media file
  parameter: none
  returns:   next four bytes
*********************************************************************/

UWORD32 GetStegLong (void)
{
  UWORD32 value;

  if (fread (&value, 4, 1, StegFile) != 1)
    UnexpEOF ();

  return value;
}


/*********************************************************************
  PutStegWord: writes one 16 bit word to media file
  parameter: data word, Pointer to FILE descriptor
  returns:   number of written bytes
*********************************************************************/

size_t PutStegWord (UWORD16 value)
{
  return fwrite (&value, 2, 1, StegFile);
}


/*********************************************************************
  NextIs:   compare string to next bytes in file, allways read the
             whole length
  parameter: pointer to a string to be matched
  returns:   flag, TRUE if equal
             file pointer set behind the characters compared
*********************************************************************/

int NextIs (char *pattern)
{
  int result = TRUE;

  while (*pattern)
    if (GetStegByte () != *pattern++)
      result = FALSE;

  return result;
}


/*********************************************************************
  TurnWheel: shows a little wheel as progress indicator
  parameter: none
  returns:   nothing
*********************************************************************/

void TurnWheel (void)
{
  static int    i = 3;
  static UBYTE  wheel[4] = {'-', '\\', '|', '/'};

  fputc (wheel[i = (i + 1) % 4], stderr);
  putc ('\b', stderr);
  fflush (stderr);
}


/*********************************************************************
  BitsThisTime: calculates the number of bits in current data point;
             uses a 'line draw' algorithm to distribute all bits
             evenly among all available bytes
  parameter: none
  returns:   bit number
*********************************************************************/

int BitsThisTime (void)
{
  ldiv_t btt;

  btt = ldiv (IndicatorSum + BitsNeeded, BytesAvail);
  IndicatorSum = btt.rem;

  return btt.quot;
}


/*********************************************************************
  InitBitDist: initializes bit distribution algorithm to found data
             length and available data points
  parameter: number of length bytes processed
  returns:   nothing
*********************************************************************/

void InitBitDist (int LenByt)
{
  BitsNeeded = StegLength << 3;           /* 8 bit/byte */
  BytesAvail = MediaLength - (LenByt << 3);
  IndicatorSum = BytesAvail >> 1;
}


/*********************************************************************
  NextBits:  delivers the required number of bits from a static shift
             register; when necessary, shift register is filled from
             left to right with next scrambled byte
  parameter: number of bits
  returns:   requested bits as integer
*********************************************************************/

unsigned int NextBits (int HowMany)
{
  static unsigned int ShiftReg = 0;
  static int TopBit = 0;
  unsigned int r;

  if (HowMany > TopBit)      /* not enough bits to satisfy request */
  {
    ShiftReg |= (unsigned int) Scramble ((*NextByte) ()) << TopBit;
    TopBit += 8;
  }

  r = ShiftReg & ~(~0u << HowMany);
  ShiftReg >>= HowMany;
  TopBit -= HowMany;

  return r;
}


/*********************************************************************
  ModDBytes: injects the secret data bits into a block of 8 bit data
             in memory
  parameter: pointer to UBYTE array, size of array
  returns:   nothing
*********************************************************************/

void ModDBytes (UBYTE *block, size_t count)
{
  int n;

  while (count--)
  {
    if (n = BitsThisTime ())
      *block = *block & (~0u << n) | NextBits (n);
    block++;
  }
}


/*********************************************************************
  ModDWords: injects the secret data bits into a block of 16 bit data
             in memory
  parameter: pointer to int-array, size of array
  returns:   nothing
*********************************************************************/

void ModDWords (UWORD16 *block, size_t count)
{
  int n;

  while (count--)
  {
    if (n = BitsThisTime ())
      *block = *block & (~0u << n) | NextBits (n);
    block++;
  }
}


/*********************************************************************
  ModLBytes: injects the length bits into a block of 8 bit data
             in memory, switches to data routine after last byte
  parameter: pointer to UBYTE array, size of array
  returns:   nothing
*********************************************************************/

void ModLBytes (UBYTE *block, size_t count)
{
  static int BitsLeft = 0;

  while (count--)
  {
    *block = *block & ~1u | NextBits (1);
    block++;

    if (BitsLeft)                    /* last seven bits in progress */
    {
      if (--BitsLeft == 0)
      {
        ModData = (void (*) (void *, size_t)) ModDBytes;
        ModDBytes (block, count); /* rest of block with new routine */
        return;
      }
    }
    else if (LastLength)                  /* first bit of last byte */
      BitsLeft = 7;
  }
}


/*********************************************************************
  ModLWords: injects the length bits into a block of 16 bit data
             in memory, switches to data routine after last byte
  parameter: pointer to int-array, size of array
  returns:   nothing
*********************************************************************/

void ModLWords (UWORD16 *block, size_t count)
{
  static int BitsLeft = 0;

  while (count--)
  {
    *block = *block & ~1u | NextBits (1);
    block++;

    if (BitsLeft)                    /* last seven bits in progress */
    {
      if (--BitsLeft == 0)
      {
        ModData = (void (*) (void *, size_t)) ModDWords;
        ModDWords (block, count); /* rest of block with new routine */
        return;
      }
    }
    else if (LastLength)                  /* first bit of last byte */
      BitsLeft = 7;
  }
}


/*********************************************************************
  GetBitsFromByte: get LSB bit(s) from next byte of the disk file
  parameter: number of Least Significant Bits to extract
  returns:   requested bits as integer
*********************************************************************/

unsigned int GetBitsFromByte (int Bits)
{
  return (unsigned int) GetStegByte () & ~(~0u << Bits);
}


/*********************************************************************
  GetBitsFromWord: get LSB bit(s) from next word of the disk file
  parameter: number of Least Significant Bits to extract
  returns:   requested bits as integer
*********************************************************************/

unsigned int GetBitsFromWord (int Bits)
{
  return (unsigned int) GetStegWord () & ~(~0u << Bits);
}


/*********************************************************************
  ExtractBits:  extracts the hidden bits out of previously modified
             data and rearranges them to bytes in a local shift
             register that is emptied to length calculation or
             SecretFile when full
  parameter: none
  returns:   nothing
*********************************************************************/

void ExtractBits (void)
{
  static unsigned int ShiftReg = 0;
  static int TopBit = 0;
  int HowMany;

  HowMany = BitsThisTime ();

  ShiftReg  |= (*GetBits) (HowMany) << TopBit;
  TopBit += HowMany;

  if (TopBit >= 8)      /* another byte ready */
  {
    (*OutputByte) (DeScramble (ShiftReg & 0xFFu));
    ShiftReg >>= 8;
    TopBit -= 8;
  }
}


/*********************************************************************
  GetSecretByte: reads one byte from the temporary intermediate file
  parameter: none
  returns:   read byte
*********************************************************************/

UBYTE GetSecretByte (void)
{
  if (! StegLeft--)
    ErrorExit (OUTOFSYNC, "Error: Reading past end of data to hide.\n");

  return getc (SecretFile);
}


/*********************************************************************
  DataOut:   writes reconstructed byte to SecretFile
  parameter: byte
  returns:   nothing
*********************************************************************/

void DataOut (UBYTE which)
{
  if (StegLeft--)
    putc (which, SecretFile);
  else
    ErrorExit (OUTOFSYNC, "Error: Reading past end of media file.\n");
}


    /* EncodeLength/LengthOut handle Lengths up to 269525055  */
    /* enough, as the 8-fold amount of bits is necessary and  */
    /* must fit into a 'long' value; avoids leading zeros     */


/*********************************************************************
  EncodeLength: encodes the length of secret data (Secretlength) into
             a buffer of up to 4 bytes
  parameter: pointer to buffer
  returns:   number of bytes used
*********************************************************************/

int EncodeLength (UBYTE *LenghtBytes)
{
  UWORD32 Len;

  Len = StegLength;

  if (Len < 64)
  {
    LenghtBytes[0] = (UBYTE) Len;
    return 1;
  }

  Len -= 64;
  LenghtBytes[0] = Len & 0xFF;
  Len >>= 8;

  if (Len < 160)
  {
    LenghtBytes[1] = (UBYTE) Len + 64;
    return 2;
  }

  Len -= 160;
  LenghtBytes[1] = Len & 0xFF;
  Len >>= 8;

  if (Len < 16)
  {
    LenghtBytes[2] = (UBYTE) Len + 224;
    return 3;
  }

  Len -= 16;
  LenghtBytes[2] = Len & 0xFF;
  Len >>= 8;
  LenghtBytes[3] = (UBYTE) Len + 240;
  return 4;
}


/*********************************************************************
  GetLengthByte: returns one byte of the encoded length kept in a
             static buffer; on 1st invocation, encode length; on last
             invocation, switch input to read from intermediate file
  parameter: none
  returns:   one length byte
*********************************************************************/

UBYTE GetLengthByte (void)
{
  static UBYTE LengthBytes [4];
  static int BytesLeft = 0;

  /* Load byte array on first call */
  if (BytesLeft == 0)
  {
    BytesLeft = EncodeLength (LengthBytes);
    InitBitDist (BytesLeft);      /* prepare Bitdistribution */
  }

  /* when this is the last length byte */
  if (--BytesLeft == 0)
  {
    LastLength = TRUE;                       /* signal: last byte */
    fseek (SecretFile, 0L, SEEK_SET);
    NextByte = GetSecretByte;    /* switch to start of tempfile */
  }

  return LengthBytes [BytesLeft];
}


/*********************************************************************
  LengthOut: decodes the length of hidden data by interpreting 1 to 4
             successive bytes; switches to data output after last one
  parameter: one byte of the encoded length
  returns:   nothing
*********************************************************************/

void LengthOut (UBYTE AddByte)
{
  static int ByteNo = 0;

  StegLength += AddByte;
  switch (++ByteNo)
  {
    case 1:
      if (StegLength >= 64)
      {
        StegLength <<= 8;
        StegLength -= 16320;
        return;
      }
      break;
    case 2:
      if (StegLength >= 41024)
      {
        StegLength <<= 8;
        StegLength -= 10461120;
        return;
      }
      break;
    case 3:
      if (StegLength >= 1089600)
      {
        StegLength <<= 8;
        StegLength -= 277848000;
        return;
      }
      break;
  }

  if (StegLength > (MediaLength - ByteNo) * MaxBitsAllowed / 8)
    ErrorExit (INVALIDLENGTH, "Error: File seems not to contain hidden data.\n");

  StegLeft = StegLength;
  InitBitDist (ByteNo);
  OutputByte = DataOut;   /* send following data to stdout */
}


/*********************************************************************
  SetBlock:  set the secret bits in a block of the disk file in
             portions of 10 kB max;
  parameter: length of the continous block in data points, file
             pointer set to first byte
  returns:   nothing
*********************************************************************/

void SetBlock (UWORD32 count)
{
  void   *block;
  long   position;
  size_t subcount;

  subcount = 10240 / DataSize;
  while (count)
  {
    if (subcount > count)
      subcount = count;
    count -= subcount;

    if (block = malloc (subcount * DataSize))
    {
      if (verbose > 1)
        TurnWheel ();

      position = ftell (StegFile);
      fseek (StegFile, position, SEEK_SET);
      if (fread (block, DataSize, subcount, StegFile) != subcount)
        UnexpEOF ();

      (*ModData) (block, subcount);

      fseek (StegFile, position, SEEK_SET);
      fwrite (block, DataSize, subcount, StegFile);
      free (block);
    }
    else
      NoMemory ();
  }
}


/*********************************************************************
  GetBlock:  get bits from a block of the disk file in 10 kB portions
  parameter: count of the continous block, read pointer set to first
             byte
  returns:   nothing
*********************************************************************/

void GetBlock (UWORD32 count)
{
  size_t subcount;

  while (count)
  {
    subcount = 10240 / DataSize;
    if (subcount > count)
      subcount = count;
    count -= subcount;

    if (verbose > 1)
      TurnWheel ();

    while (subcount--)
      ExtractBits ();
  }
}
