/*********************************************************************
  Routines for Creative Labs VOC files
  support for 8 bit sound data

  part of: Hide4PGP by Heinz Repp
  last modified: 12/13/99
*********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "hide4pgp.h"
#include "utils.h"


static long position;


/*********************************************************************
  CheckPByte: checks the 'packbyte' encoding the number of bits one
             data point uses; aborts if not zero i.e. 8 bit
  parameter: none
  returns:   nothing
*********************************************************************/

void CheckPByte (void)
{
  if (GetStegByte ())
    ErrorExit (VOCNOT8BIT, "Error (VOC): resolution not 8 bit.\n");
}


/*********************************************************************
  ModeOut:   outputs mode and sample frequency once if verbose > 1
  parameter: stereo flag, sample rate
  returns:   nothing
*********************************************************************/

void ModeOut (int mode, long rate)
{
  static int shown = FALSE;

  if (verbose > 1 && (shown ? FALSE : (shown = TRUE)))
    fprintf (stderr, "Mode: %so, Sample frequency: %li Hz.\n",
             mode ? "Stere" : "Mon", rate);
}


/*********************************************************************
  VOCNextBlock: positions read pointer to the next data block (if any)
             optionally outputs sound characteristics
  parameter: none
  returns:   length of continous data block, 0L if none
*********************************************************************/

UWORD32 VOCNextBlock (void)
{
  int   id, extended = FALSE;
  long  length;

  while (!fseek (StegFile, position, SEEK_SET) &&
         (id = GetStegByte ()) != EOF && id)
  {
    length =  GetStegByte ();
    length += (long) GetStegWord () << 8;
    position += length + 4;     /* block length + length bytes */
    if (extended && id != 1)
    {
      if (verbose > 1)
        fputs ("VOC: Missing data block after extended block.\n", stderr);
      extended = FALSE;
    }
    switch (id)
    {
      case 1:                          /* voice data */
        if (extended)
        {
          extended = FALSE;
          GetStegWord ();          /* dummy */
        }
        else
        {
          ModeOut (0, 1000000L / (256 - GetStegByte ()));
          CheckPByte ();
        }
        return length - 2;

      case 2:                          /* voice continuation */
        return length;

      case 8:                          /* extended block */
        length = GetStegWord ();
        CheckPByte ();
        if (GetStegByte ())
          ModeOut (1, 128000000L / (65536L - length));
        else
          ModeOut (0, 256000000L / (65536L - length));
        extended = TRUE;
    }
  }
  return 0;
}


/*********************************************************************
  CheckIfVOC: checks file characteristics and optionally outputs
             version number
  parameter: none
  returns:   data width flag - always FALSE meaning 8 bit data
*********************************************************************/

int CheckIfVOC (UWORD32 (**NextBlockP) (void))
{
  int     vers;
  long    first;
  UWORD32 blklen;

  if (! NextIs ("reative Voice File\x1A"))
    return FALSE;

  first = position = GetStegWord ();
  if (verbose)
  {
    fputs ("Creative Labs VOC file", stderr);
    if (verbose > 1)
    {
      vers = GetStegByte ();
      fprintf (stderr, " version %i.%i",
               GetStegByte (), vers);
    }
    fputs (" detected.\n", stderr);
  }

  DataSize = 1;
  MaxBitsAllowed = 1;
  if (hiding)
    ModData = (void (*) (void *, size_t)) ModLBytes;
  else
    GetBits = GetBitsFromByte;
  *NextBlockP = VOCNextBlock;

  /* sum up usable bytes in all blocks */
  while (blklen = VOCNextBlock ())
    MediaLength += blklen;
  position = first;

  return TRUE;
}
