#include "mjackson.h"

/* general inet wrapper functions and other numerical things */

const char *
ipv4_ntop (int family, const void *addrptr, char *strptr, size_t len)
{
  const u_char *p = (const u_char *) addrptr;

  if (family == AF_INET)
    {
      if (IPV4_ADDRLEN > len)
	{
	  errno = ENOSPC;
	  return (NULL);
	}

      sprintf (strptr, "%d.%d.%d.%d", p[0], p[1], p[2], p[3]);
      return (strptr);
    }

  errno = EAFNOSUPPORT;
  return (NULL);
}

const char *
ipv4_showname (int family, const void *addrptr, char *strptr, size_t len)
{
  struct hostent *hentry;

  size_t aflen;

  if (family == AF_INET)
    aflen = 4;
  else
    {
      errno = EAFNOSUPPORT;
      return (NULL);
    }
    
  if ((hentry = gethostbyaddr (addrptr, aflen, family)) != NULL)
    {
    alarm(0);
      if (strlen (hentry->h_name) < len)
	{
	  strcpy (strptr, hentry->h_name);
	  return (strptr);
	}
    }
  return (ipv4_ntop(family, addrptr, strptr, len));
}

int 
ipv4_rslvhost (char *hostname, SAin * hostinfo)
{
  struct hostent *hentry;

  if (!inet_aton (hostname, &hostinfo->sin_addr))
    {
      if ((hentry = gethostbyname (hostname)) != NULL)
	{
	  memcpy (&hostinfo->sin_addr, hentry->h_addr, hentry->h_length);
	}
      else
	return (0);
    }

  return (1);
}

u_short 
in_cksum (u_short * addr, int len) {
  int nleft = len;
  int sum = 0;

  u_short *w = addr;
  u_short answer = 0;

  /*
   * Our algorithm is simple, using a 32 bit accumulator (sum), we add
   * sequential 16 bit words to it, and at the end, fold back all the
   * carry bits from the top 16 bits into the lower 16 bits.
   */

  while (nleft > 1)
    {
      sum += *w++;
      nleft -= 2;
    }

  /* mop up an odd byte, if necessary */
  if (nleft == 1)
    {
      *(unsigned char *) (&answer) = *(unsigned char *) w;
      sum += answer;
    }

  /* 4add back carry outs from top 16 bits to low 16 bits */
  sum = (sum >> 16) + (sum & 0xffff);	/* add hi 16 to low 16 */
  sum += (sum >> 16);		/* add carry */
  answer = ~sum;		/* truncate to 16 bits */
  return (answer);
}


u_short tcp_cksum (char *packet, int len,
                   struct in_addr src_addr,
                   struct in_addr dst_addr)
{
   char *temppkt;
   u_short cksum;

   struct temphdr {
      struct in_addr src;
      struct in_addr dst;
      unsigned char zero;
      unsigned char pr;
      u_short len;
   } temphdr;

   temphdr.pr = IPPROTO_TCP;
   temphdr.len = htons (len);
   temphdr.zero = 0;

   temphdr.src = src_addr;
   temphdr.dst = dst_addr;

   if((temppkt = malloc (sizeof (temphdr) + len)) == NULL) {
      perror ("cannot allocate memory\n");
      exit (EXIT_FAILURE);
   }

   memcpy (temppkt, &temphdr, sizeof(struct temphdr));
   memcpy ((temppkt + sizeof(temphdr)), packet, len);

   cksum = in_cksum((unsigned short *)temppkt, (len + sizeof(temphdr)));
   free (temppkt);
   return cksum;
}

int 
hamming (u_long mask)
{
  u_short i, j;
  for (i = j = 0; i < 32; i++)
    if (mask & (1 << i))
      j++;
  return (j >> 1);
}
