/*
 * This file is part of the coreboot project.
 *
 * Copyright 2013 Google Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef __SOC_NVIDIA_TEGRA_I2C_H__
#define __SOC_NVIDIA_TEGRA_I2C_H__

#include <stdint.h>

void i2c_init(unsigned bus);

#define IOHEADER_BITFIELD(name, shift, mask) \
	IOHEADER_##name##_SHIFT = shift, \
	IOHEADER_##name##_MASK = \
		mask << IOHEADER_##name##_SHIFT

#define IOHEADER_BITFIELD_VAL(field, name, val) \
	IOHEADER_##field##_##name = \
		val << IOHEADER_##field##_SHIFT

enum {
	/* Word 0 */
	IOHEADER_BITFIELD(WORD0_PROTHDRSZ, 28, 0x3),

	IOHEADER_BITFIELD(WORD0_PKTID, 16, 0xff),

	IOHEADER_BITFIELD(WORD0_CONTROLLER_ID, 12, 0xf),

	IOHEADER_BITFIELD(WORD0_PROTOCOL, 4, 0xf),
	IOHEADER_BITFIELD_VAL(WORD0_PROTOCOL, I2C, 1),

	IOHEADER_BITFIELD(WORD0_PKTTYPE, 0, 0x7),
	IOHEADER_BITFIELD_VAL(WORD0_PKTTYPE, REQUEST, 0),
	IOHEADER_BITFIELD_VAL(WORD0_PKTTYPE, RESPONSE, 1),
	IOHEADER_BITFIELD_VAL(WORD0_PKTTYPE, INTERRUPT, 2),
	IOHEADER_BITFIELD_VAL(WORD0_PKTTYPE, STOP, 3),

	/* Word 1 */
	IOHEADER_BITFIELD(WORD1_PAYLOADSIZE, 0, 0xfff)
};

enum {
	IOHEADER_BITFIELD(I2C_REQ_RESP_PKT_FREQ_SHIFT, 25, 0x1),
	IOHEADER_BITFIELD_VAL(I2C_REQ_RESP_PKT_FREQ_SHIFT, END, 0),
	IOHEADER_BITFIELD_VAL(I2C_REQ_RESP_PKT_FREQ_SHIFT, EACH, 1),

	IOHEADER_BITFIELD(I2C_REQ_RESP_PKT_ENABLE, 24, 0x1),

	IOHEADER_BITFIELD(I2C_REQ_HS_MODE, 22, 0x1),

	IOHEADER_BITFIELD(I2C_REQ_CONTINUE_ON_NACK, 21, 0x1),

	IOHEADER_BITFIELD(I2C_REQ_SEND_START_BYTE, 20, 0x1),

	IOHEADER_BITFIELD(I2C_REQ_READ_WRITE, 19, 0x1),
	IOHEADER_BITFIELD_VAL(I2C_REQ_READ_WRITE, WRITE, 0),
	IOHEADER_BITFIELD_VAL(I2C_REQ_READ_WRITE, READ, 1),

	IOHEADER_BITFIELD(I2C_REQ_ADDRESS_MODE, 18, 0x1),
	IOHEADER_BITFIELD_VAL(I2C_REQ_ADDRESS_MODE, 7BIT, 0),
	IOHEADER_BITFIELD_VAL(I2C_REQ_ADDRESS_MODE, 10BIT, 1),

	IOHEADER_BITFIELD(I2C_REQ_IE, 17, 0x1),

	IOHEADER_BITFIELD(I2C_REQ_REPEAT_START_STOP, 16, 0x1),
	IOHEADER_BITFIELD_VAL(I2C_REQ_REPEAT_START_STOP, STOP, 0),
	IOHEADER_BITFIELD_VAL(I2C_REQ_REPEAT_START_STOP, START, 1),

	IOHEADER_BITFIELD(I2C_REQ_CONTINUE_XFER, 15, 0x1),

	IOHEADER_BITFIELD(I2C_REQ_HS_MASTER_ADDR, 12, 0x7),

	IOHEADER_BITFIELD(I2C_REQ_SLAVE_ADDR, 0, 0x3ff)
};

enum {
	TEGRA_I2C_CNFG_MSTR_CLR_BUS_ON_TIMEOUT_SHIFT = 15,
	TEGRA_I2C_CNFG_MSTR_CLR_BUS_ON_TIMEOUT_MASK =
		0x1 << TEGRA_I2C_CNFG_MSTR_CLR_BUS_ON_TIMEOUT_SHIFT,
	TEGRA_I2C_CNFG_DEBOUNCE_CNT_SHIFT = 12,
	TEGRA_I2C_CNFG_DEBOUNCE_CNT_MASK =
		0x7 << TEGRA_I2C_CNFG_DEBOUNCE_CNT_SHIFT,
	TEGRA_I2C_CNFG_NEW_MASTER_FSM_SHIFT = 11,
	TEGRA_I2C_CNFG_NEW_MASTER_FSM_MASK =
		0x1 << TEGRA_I2C_CNFG_NEW_MASTER_FSM_SHIFT,
	TEGRA_I2C_CNFG_PACKET_MODE_EN_SHIFT = 10,
	TEGRA_I2C_CNFG_PACKET_MODE_EN_MASK =
		0x1 << TEGRA_I2C_CNFG_PACKET_MODE_EN_SHIFT,
	TEGRA_I2C_CNFG_SEND_SHIFT = 9,
	TEGRA_I2C_CNFG_SEND_MASK = 0x1 << TEGRA_I2C_CNFG_SEND_SHIFT,
	TEGRA_I2C_CNFG_NOACK_SHIFT = 8,
	TEGRA_I2C_CNFG_NOACK_MASK = 0x1 << TEGRA_I2C_CNFG_NOACK_SHIFT,
	TEGRA_I2C_CNFG_CMD2_SHIFT = 7,
	TEGRA_I2C_CNFG_CMD2_MASK = 0x1 << TEGRA_I2C_CNFG_CMD2_SHIFT,
	TEGRA_I2C_CNFG_CMD1_SHIFT = 6,
	TEGRA_I2C_CNFG_CMD1_MASK = 0x1 << TEGRA_I2C_CNFG_CMD1_SHIFT,
	TEGRA_I2C_CNFG_START_SHIFT = 5,
	TEGRA_I2C_CNFG_START_MASK = 0x1 << TEGRA_I2C_CNFG_START_SHIFT,
	TEGRA_I2C_CNFG_SLV2_SHIFT = 4,
	TEGRA_I2C_CNFG_SLV2_MASK = 0x1 << TEGRA_I2C_CNFG_SLV2_SHIFT,
	TEGRA_I2C_CNFG_LENGTH_SHIFT = 1,
	TEGRA_I2C_CNFG_LENGTH_MASK = 0x7 << TEGRA_I2C_CNFG_LENGTH_SHIFT,
	TEGRA_I2C_CNFG_A_MOD_SHIFT = 0,
	TEGRA_I2C_CNFG_A_MOD_MASK = 0x1 << TEGRA_I2C_CNFG_A_MOD_SHIFT
};

enum {
	TEGRA_I2C_PKT_STATUS_COMPLETE_SHIFT = 24,
	TEGRA_I2C_PKT_STATUS_COMPLETE_MASK =
		0x1 << TEGRA_I2C_PKT_STATUS_COMPLETE_SHIFT,
	TEGRA_I2C_PKT_STATUS_PKT_ID_SHIFT = 16,
	TEGRA_I2C_PKT_STATUS_PKT_ID_MASK =
		0xff << TEGRA_I2C_PKT_STATUS_PKT_ID_SHIFT,
	TEGRA_I2C_PKT_STATUS_BYTENUM_SHIFT = 4,
	TEGRA_I2C_PKT_STATUS_BYTENUM_MASK =
		0xfff << TEGRA_I2C_PKT_STATUS_BYTENUM_SHIFT,
	TEGRA_I2C_PKT_STATUS_NOACK_ADDR_SHIFT = 3,
	TEGRA_I2C_PKT_STATUS_NOACK_ADDR_MASK =
		0x1 << TEGRA_I2C_PKT_STATUS_NOACK_ADDR_SHIFT,
	TEGRA_I2C_PKT_STATUS_NOACK_DATA_SHIFT = 2,
	TEGRA_I2C_PKT_STATUS_NOACK_DATA_MASK =
		0x1 << TEGRA_I2C_PKT_STATUS_NOACK_DATA_SHIFT,
	TEGRA_I2C_PKT_STATUS_ARB_LOST_SHIFT = 1,
	TEGRA_I2C_PKT_STATUS_ARB_LOST_MASK =
		0x1 << TEGRA_I2C_PKT_STATUS_ARB_LOST_SHIFT,
	TEGRA_I2C_PKT_STATUS_BUSY_SHIFT = 0,
	TEGRA_I2C_PKT_STATUS_BUSY_MASK =
		0x1 << TEGRA_I2C_PKT_STATUS_BUSY_SHIFT,
};

enum {
	TEGRA_I2C_FIFO_STATUS_TX_FIFO_EMPTY_CNT_SHIFT = 4,
	TEGRA_I2C_FIFO_STATUS_TX_FIFO_EMPTY_CNT_MASK =
		0xf << TEGRA_I2C_FIFO_STATUS_TX_FIFO_EMPTY_CNT_SHIFT,

	TEGRA_I2C_FIFO_STATUS_RX_FIFO_FULL_CNT_SHIFT = 0,
	TEGRA_I2C_FIFO_STATUS_RX_FIFO_FULL_CNT_MASK =
		0xf << TEGRA_I2C_FIFO_STATUS_RX_FIFO_FULL_CNT_SHIFT
};

extern void * const tegra_i2c_bases[];

struct tegra_i2c_regs {
	uint32_t cnfg;
	uint32_t cmd_addr0;
	uint32_t cmd_addr1;
	uint32_t cmd_data1;
	uint32_t cmd_data2;
	uint8_t _rsv0[8];
	uint32_t status;
	uint32_t sl_cnfg;
	uint32_t sl_rcvd;
	uint32_t sl_status;
	uint32_t sl_addr1;
	uint32_t sl_addr2;
	uint32_t tlow_sext;
	uint8_t _rsv1[4];
	uint32_t sl_delay_count;
	uint32_t sl_int_mask;
	uint32_t sl_int_source;
	uint32_t sl_int_set;
	uint8_t _rsv2[4];
	uint32_t tx_packet_fifo;
	uint32_t rx_fifo;
	uint32_t packet_transfer_status;
	uint32_t fifo_control;
	uint32_t fifo_status;
	uint32_t interrupt_mask;
	uint32_t interrupt_status;
	uint32_t clk_divisor;
	uint32_t interrupt_source;
	uint32_t interrupt_set;
	uint32_t slv_tx_packet_fifo;
	uint32_t slv_rx_fifo;
	uint32_t slv_packet_status;
	uint32_t bus_clear_config;
	uint32_t bus_clear_status;
	uint32_t spare;
};

#endif	/* __SOC_NVIDIA_TEGRA_I2C_H__ */
