#!/usr/bin/perl

use IPC::Open3;
use Sys::Hostname;
use Getopt::Std;
use POSIX;

my @devices;
my %results;

# WARNING!! Do not add code bewteen "#BEGIN_VERSION_GENERATION" and
# "#END_VERSION_GENERATION"  It is generated by the Makefile

#BEGIN_VERSION_GENERATION
$FENCE_RELEASE_NAME="";
$REDHAT_COPYRIGHT="";
$BUILD_DATE="";
#END_VERSION_GENERATION

sub get_key
{
    my $addr = gethostbyname($name) or die "$!\n";

    return unpack("H*", $addr);
}

sub register_device
{
    my $func = (caller(0))[3];
    my ($dev, $key) = @_;

    print "DEBUG: $func ($dev, $key)\n" if ($opt_d);

    my ($in, $out, $err);
    my $cmd = "sg_persist $dev -o -G -S $key";

    my $pid = open3($in, $out, $err, $cmd) or die "$!\n";

    waitpid($pid, 0);

    my $rval = WEXITSTATUS($?);

    $results{$dev}[0] = $rval;

    print "DEBUG: [$rval] $cmd\n" if $opt_d;

    close($in);
    close($out);
    close($err);

    return $rval;
}

sub unregister_device
{
    my $func = (caller(0))[3];
    my ($dev, $key) = @_;

    print "DEBUG: $func ($dev, $key)\n" if $opt_d;

    my ($in, $out, $err);
    my $cmd = "sg_persist $dev -o -G -K $key -S 0";

    my $pid = open3($in, $out, $err, $cmd) or die "$!\n";

    waitpid($pid, 0);

    my $rval = WEXITSTATUS($?);

    $results{$dev}[1] = $rval;

    print "DEBUG: [$rval] $cmd\n" if $opt_d;

    close($in);
    close($out);
    close($err);

    return $rval;
}

sub get_block_devices
{
    my $block_dir = "/sys/block";

    opendir(DIR, $block_dir) or die "Error: $! $block_dir\n";

    my @block_devices = grep { /^sd*/ } readdir(DIR);

    closedir(DIR);

    for $dev (@block_devices)
    {
	push @devices, "/dev/" . $dev;
    }
}

sub get_cluster_devices
{
    my ($in, $out, $err);
    my $cmd = "lvs --noheadings --separator : -o vg_attr,devices";
    my $pid = open3($in, $out, $err, $cmd) or die "$!\n";

    waitpid($pid, 0);

    die "Error: unable to exec lvs command.\n" if WEXITSTATUS($?);

    while (<$out>)
    {
	chomp;

	my ($vg_attr, $dev) = split /:/, $_, 3;

	if ($vg_attr =~ /.*c$/)
	{
	    $dev =~ s/\(.*\)//;
	    push @devices, $dev;
	}
    }

    close($in);
    close($out);
    close($err);
}

sub test_devices
{
    my $name = hostname() or die "$!\n";
    my $addr = gethostbyname($name) or die "$!\n";
    my $key = unpack("H*", $addr);

    foreach $dev (@devices)
    {
	if (register_device($dev, $key) != 0)
	{
	}
	if (unregister_device($dev, $key) != 0)
	{
	}
    }
}

sub print_results
{
    my $device_count = scalar(@devices);

    my $failure_count = 0;
    my $success_count = 0;

    print "\nAttempted to register with devices:\n";
    print "-------------------------------------\n";

    for $dev (@devices)
    {
	print "\t$dev\t";
	if ($results{$dev}[0] == 0)
	{
	    $success_count++;
	    print "Success\n";
	}
	else
	{
	    $failure_count++;
	    print "Failure\n";
	}
    }

    print "-------------------------------------\n";
    print "Number of devices tested: $device_count\n";
    print "Number of devices passed: $success_count\n";
    print "Number of devices failed: $failure_count\n\n";

    if ($failure_count != 0)
    {
	exit(1);
    }
}

sub print_usage
{
    print "\nUsage: scsi_test [-c|-s] [-d] [-h]\n\n";

    print "Options:\n\n";

    print "  -c     Cluster mode. This mode is intended to test\n";
    print "         SCSI persistent reservation capabilties for\n";
    print "         devices that are part of existing clustered\n";
    print "         volumes. Only devices in LVM cluster volumes\n";
    print "         will be tested.\n\n";
    print "  -s     SCSI mode. This mode is intended to test SCSI\n";
    print "         persistent reservation capabilities for all SCSI\n";
    print "         devices visible on a node.\n\n";
    print "  -d     Debug flag. This will print debugging information\n";
    print "         such as the actual commands being run to register\n";
    print "         and unregister a device.\n\n";
    print "  -h     Help. Prints out this usage information.\n\n";
}

### main ###

if (getopts("cdhsv") == 0)
{
    print_usage;
    exit(1);
}

if ($opt_h)
{
    print_usage;
    exit(0);
}

if ($opt_c)
{
    print "\nTesting devices in cluster volumes...\n";
    get_cluster_devices;
}

if ($opt_s)
{
    print "\nTesting all SCSI block devices...\n";
    get_block_devices;
}

if (!$opt_c && !$opt_s)
{
    print "\nPlease specify either cluster or SCSI mode.\n";
    print_usage;
    exit(1);
}

test_devices;

print_results;

