// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
// ----------------------------------------------------------------------
// FieldBCondBase: the base class for all Field-based boundary conditions.
//-----------------------------------------------------------------------------

#ifndef POOMA_BCONDS_FIELDBCONDBASE_H
#define POOMA_BCONDS_FIELDBCONDBASE_H

#include "Field/Field.h"
#include "BConds/BCondBase.h"

//-----------------------------------------------------------------------------
// FieldBCondBase is an abstract base class for all boundary conditions. 
// It extends BCondBase to work specifically with Fields.
//-----------------------------------------------------------------------------

template<class Subject>
class FieldBCondBase : 
  public BCondBase<typename Subject::Domain_t, Subject> {
public:

  //---------------------------------------------------------------------------
  // Convenient typedefs. Make this class, the base class, the domain type, 
  // and the subject public.

  typedef Subject Subject_t;
  typedef typename Subject_t::Domain_t Domain_t;
  typedef BCondBase<Domain_t, Subject_t> Base_t;  
  typedef FieldBCondBase<Subject> This_t;  

  //---------------------------------------------------------------------------
  // Constructors. We do not support a default constructor since we always have
  // to have a subject. However, we can rely on the default constructors for
  // domains to make empty sets. We rely on shallow copy semantics for the 
  // subject.

  FieldBCondBase(const Subject_t &s)
  : Base_t(s, DontCopyBoundaryConditions()) { }

  FieldBCondBase(const Subject_t &s, const Domain_t &dest)
  : Base_t(s, DontCopyBoundaryConditions(), dest) { }

  FieldBCondBase(const Subject_t &s, const Domain_t &dest, const Domain_t &src)
  : Base_t(s, DontCopyBoundaryConditions(), dest, src) { }

  //---------------------------------------------------------------------------
  // Copy constructor. We rely on shallow copy semantics for the subject.

  FieldBCondBase(const This_t &model)
  : Base_t(model) { }

  //---------------------------------------------------------------------------
  // Clone function. Return a new version of this boundary condition with
  // the new subject.

  virtual This_t *retarget(const Subject_t &newSubject) const = 0;
  
  //---------------------------------------------------------------------------
  // Trivial destructor, but virtual since we will be subclassing.

  virtual ~FieldBCondBase() { }    
};

#endif // POOMA_BCONDS_FIELDBCONDBASE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldBCondBase.h,v $   $Author: swhaney $
// $Revision: 1.4 $   $Date: 2000/03/07 13:16:11 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
