// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// MultiPatch-Engine non-inline template definitions.
//-----------------------------------------------------------------------------

#include "Engine/MultiPatchEngine.h"
#include "Evaluator/EngineTraits.h"
#include "Engine/CompressedFraction.h"
#include "Array/Array.h"
#include "Tulip/ReduceOverContexts.h"
#include "Threads/PoomaCSem.h"

///////////////////////////////////////////////////////////////////////////////
//
// MultiPatch-Engine Member Functions
//
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
//
// Engine<Dim,T,MultiPatch>::Engine()
//
// Default constructor ... you should use operator= to initialize this
// engine after using this constructor.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
Engine()
  : pDirty_m(0)
{
  // This object must be initialized later via operator=.  If it is not,
  // this is not a useful object.  Do not attach to the layout, since we
  // don't have one yet.
}


//-----------------------------------------------------------------------------
//
// Engine<Dim,T,MultiPatch>::Engine(const Layout_t &)
//
// Initialize with a layout - we take the total domain from this,
// and register as a user of the layout.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
Engine(const Layout_t &layout)
  : layout_m(layout),
    data_m(layout.sizeGlobal()),
    pDirty_m(new bool(true))
{
  typedef typename Layout_t::Value_t Node_t;

  // check for correct match of PatchTag and the mapper used to make the
  // layout.
  // THIS IS A HACK! we test on the context of the first patch, and if it
  // is -1, we have a Layout made with the LocalMapper.

#if POOMA_CHEETAH

  if( layout_m.nodeListGlobal().size() > 0)
  {    
    int fpID = layout_m.nodeListGlobal()[0]->context();

    bool compatible =
      DistributionTraits<PatchTag>::remote ? (fpID != -1) : (fpID == -1);
      
    PInsist(compatible,
	    "PatchTag is incompatible with the ContextMapper");
    }

#endif  
  
  // Initialize the patches. We iterate through the Nodes in the layout, which
  // are supposed to match up with the patches as they are stored in our
  // vector. For each patch, we pass Node information to enable initialization.
  
  int sz = data().size();

  typedef Pooma::CountingSemaphore CountingSemaphore_t;
  typedef typename Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
    PatchAllocator<Node_t,CountingSemaphore_t> PatchAllocator_t;
  CountingSemaphore_t csem;
  
  csem.height(sz);
  
  typename Layout_t::const_iterator p = layout_m.beginGlobal();

  for (int i = 0; i < sz; ++i, ++p)
    {
      // Have threads perform data()[i] = PatchEngine_t(*p);
      
      PatchAllocator_t *spot = new PatchAllocator_t(data()[i], *p, csem);
      Pooma::addRunnable(spot);  // See spot run!
    }
  
  // Wait for all of the runnables to complete...
  
  csem.wait();
      
  // Attach ourself to the layout so we can receive messages.
  
  layout_m.attach(*this);
}


//-----------------------------------------------------------------------------
//
// Engine<Dim,T,MultiPatch>::Engine(const Engine_t &model)
//
// Copy constructor.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
Engine(const Engine_t &modelEngine)
  : layout_m(modelEngine.layout_m),
    data_m(modelEngine.data_m),
    pDirty_m(modelEngine.pDirty_m)
{
  // Make sure the RHS is initialized.

  PAssert(modelEngine.initialized());

  // Attach ourself to the layout so we can receive messages.
  
  layout_m.attach(*this);  
}


//-----------------------------------------------------------------------------
//
// Engine<Dim,T,MultiPatch>::~Engine()
//
// Destructor ... detach from layout.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
~Engine()
{
  if (initialized())
    {
      // Detach ourselves from the layout.
  
      layout_m.detach(*this);
  
      // If this is the last reference to our data, delete the dirty flag.
  
      if (!data().isShared())
	delete pDirty_m;
    }
}


//-----------------------------------------------------------------------------
//
// Engine_t &Engine<Dim,T,MultiPatch>::operator=(const Engine_t &)
//
// Assignment operator.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> > &
Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
operator=(const Engine_t &model)
{
  // Check for self-assignment.
  
  if (&model == this)
    return *this;
 
  // Make sure the RHS is initialized.

  PAssert(model.initialized());

  // If we have been previously initialized, clean up ...

  if (initialized())
    {
      // If this is the last copy of our data, delete the old dirty flag.
  
      if (!data().isShared())
	delete pDirty_m;
    
      // Detach ourselves from the old layout.

      layout_m.detach(*this);
    }

  // Assign our data.

  data_m = model.data();
  pDirty_m = model.pDirty_m;

  // Copy and attach ourself to the layout so we can receive messages.

  layout_m = model.layout_m;
  layout_m.attach(*this);
  
  return *this;  
}


//-----------------------------------------------------------------------------
//
// Gets a private copy of this engine's data. Just loop through all of our
// engines and ask them to make their own copies.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
Engine<Dim, T, MultiPatch<LayoutTag,PatchTag> > &
Engine<Dim, T, MultiPatch<LayoutTag,PatchTag> >::
makeOwnCopy()
{
  int sz = data().size();
  for (int i = 0; i < sz; ++i)
    data()[i].makeOwnCopy();
    
  return *this;
}


//-----------------------------------------------------------------------------
//
// Fill the internal guard cells if needed, and clear the dirty flag.
// Current implementation is LOCAL ONLY!!!
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
void Engine<Dim, T, MultiPatch<LayoutTag,PatchTag> >::
fillGuardsHandler(const WrappedInt<true> &) const
{
  if (!isDirty()) return;
  
#if POOMA_PURIFY

  // This is here to remove spurious UMRs that result when un-initialized
  // guards are copied in the following loop. All of the unitialized data
  // is ultimately overwritten with good data, so I don't see why purify
  // calls these UMRs in stead of unitialized memory copies, but it does.
  // I don't do this in general since it would be slow and since T(0) is
  // not generally valid. This does mean that fillGuards() will fail
  // with purify for types that do not know what to do with T(0).

  setGuards(T(0));

#endif

  typename Layout_t::FillIterator_t p = layout_m.beginFillList();
   
  while (p != layout_m.endFillList())
    {
      int src  = p->ownedID_m;
      int dest = p->guardID_m;
      
      // Create patch arrays that see the entire patch:
                  
      Array<Dim, T, PatchTag> lhs(data()[dest]), rhs(data()[src]);
      
      // Now do assignment from the subdomains.

      lhs(p->domain_m) = rhs(p->domain_m);
      
      ++p;
    }
    
  *pDirty_m = false;
}


//-----------------------------------------------------------------------------
//
// Set the internal guard cells to a particular value (default zero).
// Current implementation is LOCAL ONLY!!!
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
void Engine<Dim, T, MultiPatch<LayoutTag,PatchTag> >::
setGuards(const T &val) const
{
  typename Layout_t::FillIterator_t p = layout_m.beginFillList();
   
  while (p != layout_m.endFillList())
    {
      int dest = p->guardID_m;
      
      // Create patch arrays that see the entire patch:
                  
      Array<Dim, T, PatchTag> lhs(data()[dest]);
      
      // Now do assignment from the subdomains.

      lhs(p->domain_m) = val;
      
      ++p;
    }

  *pDirty_m = true;
}


//-----------------------------------------------------------------------------
//
// Accumulate from the internal guards into the owned domain.
// Current implementation is LOCAL ONLY!!!
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
void Engine<Dim, T, MultiPatch<LayoutTag,PatchTag> >::
accumulateFromGuards() const
{
  typename Layout_t::FillIterator_t p = layout_m.beginFillList();
     
  while (p != layout_m.endFillList())
    {
      // This time we're going from the guards to the owned.
      
      int dest = p->ownedID_m;
      int src  = p->guardID_m;
      
      // Create patch arrays that see the entire patch:
                  
      Array<Dim, T, PatchTag> lhs(data()[dest]), rhs(data()[src]);
      
      // Now accumulate values from the guards.

      lhs(p->domain_m) += rhs(p->domain_m);
      
      ++p;
    }

  *pDirty_m = true;
}


//-----------------------------------------------------------------------------
//
// void Engine<Dim,T,MultiPatch>::
// dynamicHandler<true>(Observable_t, ObserverEvent)
//
// Handler for dynamic events for patch engines that have dynamic capabilities.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
void Engine<Dim, T, MultiPatch<LayoutTag,PatchTag> >::
dynamicHandler(Observable_t &, 
               const ObserverEvent &event, 
               const WrappedInt<true> &)
{ 
  // From the event code, figure out what we should do.
  // (There has to be a better way to do this!!! [JAC])

  switch (event.event())
  {
    case DynamicEvents::create:
      {
        // Create new elements at the end of our block of data
        typedef const CreateEvent &EventRef_t;
        EventRef_t e = dynamic_cast<EventRef_t>(event);
        performCreate(e.amount(), e.patch(), e.ID());
      }
      break;
    case DynamicEvents::destroyInterval:
      { 
        // Delete elements in our patch of data using an Interval  
      
        typedef const DestroyEvent< Interval<1> > &EventRef_t;
        EventRef_t e = dynamic_cast<EventRef_t>(event);
	      
        switch (e.method())
        {
          case DynamicEvents::backfill:
            performDestroy(e.domain(), e.patch(), BackFill(), e.ID());
	    break;
	  case DynamicEvents::shiftup:
	    performDestroy(e.domain(), e.patch(), ShiftUp(), e.ID());
	    break;
	  default:
	    PInsist(false,"Unsupported delete method MultiPatchEngine::destroy");
        }
      }
      break;
    case DynamicEvents::destroyRange:
      {
        // Delete elements in our patch of data using a Range
        
        typedef const DestroyEvent< Range<1> > &EventRef_t;
        EventRef_t e = dynamic_cast<EventRef_t>(event);

        switch (e.method())
        {
          case DynamicEvents::backfill:
            performDestroy(e.domain(), e.patch(), BackFill(), e.ID());
	    break;
	  case DynamicEvents::shiftup:
	    performDestroy(e.domain(), e.patch(), ShiftUp(), e.ID());
	    break;
	  default:
	    PInsist(false,"Unsupported delete method MultiPatchEngine::destroy");
        }
      }
      break;
    case DynamicEvents::destroyList:
      {
        // Delete elements in our patch of data using an IndirectionList
    
        typedef const DestroyEvent< IndirectionList<int> > &EventRef_t;
        EventRef_t e = dynamic_cast<EventRef_t>(event);
    
        switch (e.method())
        {
          case DynamicEvents::backfill:
            performDestroy(e.domain(), e.patch(), BackFill(), e.ID());
	    break;
	  case DynamicEvents::shiftup:
	    performDestroy(e.domain(), e.patch(), ShiftUp(), e.ID());
	    break;
	  default:
	    PInsist(false,"Unsupported delete method MultiPatchEngine::destroy");
        }
      }
      break;
    case DynamicEvents::copyInterval:
      {
        // Copy elements in a specific Interval
        typedef const CopyEvent< Interval<1> > &EventRef_t;
        EventRef_t e = dynamic_cast<EventRef_t>(event);
        performCopy(e.domain(), e.fromPatch(), e.toPatch(), e.ID());
      }
      break;
    case DynamicEvents::copyRange:
      {
        // Copy elements in a specific Range
        typedef const CopyEvent< Range<1> > &EventRef_t;
	EventRef_t e = dynamic_cast<EventRef_t>(event);
	performCopy(e.domain(), e.fromPatch(), e.toPatch(), e.ID());
      }
      break;
    case DynamicEvents::copyList:
      {
        // Copy a list of elements
        typedef const CopyEvent< IndirectionList<int> > &EventRef_t;
	EventRef_t e = dynamic_cast<EventRef_t>(event);
	performCopy(e.domain(), e.fromPatch(), e.toPatch(), e.ID());
      }
      break;
    case DynamicEvents::copyPatchList:
      {
        // Copy from a list of lists.
        typedef const CopyPatchEvent &EventRef_t;
        EventRef_t e = dynamic_cast<EventRef_t>(event);
        performPatchCopy(e.domainLists(), e.fromPatch(), e.toPatch(), 
                         e.create(), e.ID());
      }
      break;
    case DynamicEvents::sync:
      {
        // loop across all patch engines, and change their (?)
        // domain layout domains. 

        for (int i=0; i < layout().sizeGlobal(); ++i)
	  data()[i].sync(layout().nodeListGlobal()[i]->domain());
      }
      break;
    default:
      PInsist(0,"Invalid dynamic op???");
  }
}

//-----------------------------------------------------------------------------
//
// void Engine<Dim,T,MultiPatch>::notify(Observable_t, ObserverEvent)
//
// Be notified of various events from the layout, including
// when the layout is deleted or when dynamic operations occur.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
void Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
notify(Observable_t &observed, const ObserverEvent &event)
{
  // Make sure this is an event for us.

  PAssert(observed.ID() == layout().ID());

  // If the event is for partitioning, do that

  if(event.event() == Layout_t::repartitionEvent)
    {
      typedef typename Layout_t::Value_t Node_t;
  
      // Reinitialize the patches. We need to make a new ref-counted
      // pointer since the number of nodes could have changed.
      // We then replace our data pointer with this new one.
      
      int sz = layout().sizeGlobal();
      
      PatchContainer_t newData(sz);
      
      typedef Pooma::CountingSemaphore CountingSemaphore_t;
      typedef typename Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
        PatchAllocator<Node_t,CountingSemaphore_t> PatchAllocator_t;
      CountingSemaphore_t csem;
  
      csem.height(sz);
  
      typename Layout_t::const_iterator p = layout().beginGlobal();
      
      for (int i = 0; i < sz; ++i, ++p)
	{
	  // Have threads perform newData[i] = PatchEngine_t(*p);
	  
          PatchAllocator_t *spot = new PatchAllocator_t(newData[i], *p, csem);
          Pooma::addRunnable(spot);  // See spot run!
        }
        
      csem.wait();
      
      data_m = newData;
    }
  else
    {
      // The event is either dynamic, or unknown.
      // Dynamic events are deferred to dynamicHandler:

      if (DynamicEvents::isDynamic(event.event()))
        {
          dynamicHandler(observed, event, 
            WrappedInt<PatchEngine_t::dynamic>());
	} 
      else
	{
	  // an event we don't care about, do nothing
	}
    }
}


//-----------------------------------------------------------------------------
//
// void Engine<Dim,T,MultiPatch>::performCreate(CreateSize_t n, PatchID_t p)
//
// Carry out a request to perform a create operation in a particular
// patch.  The layout is responsible for figuring out what patch to do
// this in, so the patch number must be a valid index into our local
// patch list.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
void Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
performCreate(CreateSize_t num, PatchID_t localPatchID, DynamicID_t did)
{
  PAssert(num >= 0);
  PAssert(localPatchID >= 0 && localPatchID < layout().sizeLocal());

  // Check if this has been performed before.  If so, skip it.
  // Use the specialized routine "checkDynamicID(obj, ID)" to
  // check the ID, which should return true if the operation
  // should proceed (and also set the ID of the obj).

  int globalID = layout().nodeListLocal()[localPatchID]->globalID();

  if (! checkDynamicID(data()[globalID], did))
    return;

  // Ask the individual patch to do the create, since it has not yet.

  data()[globalID].create(num);
}

//-----------------------------------------------------------------------------
//
// void Engine<Dim,T,MultiPatch>::
// performDestroy(Dom &killlist, PatchID_t p, DelMethod method, DynamicID_t id)
//
// Carry out the work to perform a destroy operation on a particular
// patch.  The layout is responsible for figuring out what patch to do
// this in, so the patch number must be a valid index into our local
// patch list.  Also, the domain must be a "relative" domain, with zero-
// based values.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
template <class Dom, class DeleteMethod>
void Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
performDestroy(const Dom &killlist, PatchID_t localPatchID,
	       const DeleteMethod &method, DynamicID_t did)
{
  // Only the patch specific performDestroy is implemented, as the layout will
  // take care of breaking down a cross patch destroy call into a set
  // of patch specific destroy calls.

  PAssert(localPatchID >= 0 && localPatchID < layout().sizeLocal());

  // Check if this has been performed before.  If so, skip it.
  // Use the specialized routine "checkDynamicID(obj, ID)" to
  // check the ID, which should return true if the operation
  // should proceed (and also set the ID of the obj).

  int globalID = layout().nodeListLocal()[localPatchID]->globalID();

  if (! checkDynamicID(data()[globalID], did))
    return;

  // Ask the individual patch to do the destroy, since it has not yet.
  // Set the offsetFlag to true here, since kill list is zero-based

  data()[globalID].destroy(killlist, method, true);
}


//-----------------------------------------------------------------------------
//
// void Engine<Dim,T,MultiPatch>::performCopy(Dom &copylist, PatchID_t fpatch,
//                                            PatchID_t tpatch)
//
// Carry out the work to perform a copy of values from one patch
// to another (or perhaps to the same patch).
// The layout is responsible for figuring out what patch to do
// this in, so the patch number must be a valid index into our local
// patch list.  Also, the domain must be a "relative" domain, with zero-
// based values.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
template <class Dom>
void Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
performCopy(const Dom &copylist, PatchID_t frompatch, PatchID_t topatch,
	    DynamicID_t did)
{
  PAssert(Dim == 1);
  PAssert(frompatch >= 0 && frompatch < layout().sizeLocal());
  PAssert(topatch >= 0 && topatch < layout().sizeLocal());

  // Check if this has been performed before.  If so, skip it.
  // Use the specialized routine "checkDynamicID(obj, ID)" to
  // check the ID, which should return true if the operation
  // should proceed (and also set the ID of the obj).

  int from_gID =  layout().nodeListLocal()[frompatch]->globalID();
  int to_gID =  layout().nodeListLocal()[topatch]->globalID();


  bool chk1 = checkDynamicID(data()[from_gID], did);
  bool chk2 = chk1;
  if (frompatch != topatch)
    chk2 = checkDynamicID(data()[to_gID], did);
  PAssert(chk1 == chk2);
  if (!chk1 || !chk2)
    return;

  // We have to copy elements from one patch to another here (instead
  // of calling a routine in the single-patch engine) because the
  // data might span multiple patches.  The algorithm is the same
  // regardless of whether frompatch is the same as topatch.

  PAssert(copylist[0].max() < data()[from_gID].domain().size());

  // Create storage for copied elements, and note where we start
  // putting copied values in.

  int offs = data()[from_gID].domain()[0].first();
  int i = data()[to_gID].domain()[0].last() + 1;
  int num = copylist.size();
  data()[to_gID].create(num);

  // Copy over values from one patch to another.

  for (int n = 0; n < num; ++n, ++i)
  {
    localPatchEngine(data()[to_gID])(i) =
      localPatchEngine(data()[from_gID])(copylist[0](n) + offs);
  }
}


//-----------------------------------------------------------------------------
//
// void Engine<Dim,T,MultiPatch>::performPatchCopy(...)
//
// Do the actual work of a multiple-list copy.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class LayoutTag, class PatchTag>
void Engine<Dim, T, MultiPatch<LayoutTag, PatchTag> >::
performPatchCopy(const IndirectionList< IndirectionList<int> > &domlists,
		 const IndirectionList< int > &fromlist,
		 PatchID_t topatch,
		 bool docreate,
		 DynamicID_t did)
{
  PAssert(Dim == 1);
  PAssert(topatch >= 0 && topatch < layout().sizeLocal());

  // Check if this has been performed before.  If so, skip it.
  // Use the specialized routine "checkDynamicID(obj, ID)" to
  // check the ID, which should return true if the operation
  // should proceed (and also set the ID of the obj).

  int to_gID =  layout().nodeListLocal()[topatch]->globalID();

  if (! checkDynamicID(data()[to_gID], did))
    return;

  // We have to copy elements from one patch to another here (instead
  // of calling a routine in the single-patch engine) because the
  // data might span multiple patches.  The algorithm is the same
  // regardless of whether frompatch is the same as topatch.
  // Go through all the lists, and copy data to our end.  First make
  // sure we're not going to overflow anything.

  int i, p, fill, created = 0;
  int np = domlists.size();
  PAssert(fromlist.size() == np);
  for (p = 0; p < np; ++p)
    {
      int frompatch = fromlist(p);
      int from_gID = layout().nodeListLocal()[frompatch]->globalID();
      
      PAssert(frompatch >= 0 && frompatch < layout().sizeLocal());
      PAssert(domlists(p).first() >= 0);
      PAssert(domlists(p).last() <= data()[from_gID].domain()[0].last());
      created += domlists(p).size();
    }

  // First, create space at the end if necessary; otherwise, overwrite
  // storage at the end.

  if (docreate)
    {
      fill = data()[to_gID].domain()[0].last() + 1;
      data()[to_gID].create(created);
    }
  else
    {
      PAssert(created <= data()[to_gID].domain()[0].length());
      fill = data()[to_gID].domain()[0].last() + 1 - created;
    }

  // Now, copy elements from the given domain into the new storage, from
  // each of the indirection lists.

  for (p = 0; p < np; ++p)
    {
      int sz = domlists(p).size();
      int frompatch = fromlist(p);
      int from_gID = layout().nodeListLocal()[frompatch]->globalID();
      int offs = data()[from_gID].domain()[0].first();
      for (i = 0; i < sz; ++i, ++fill)
      {
	localPatchEngine(data()[to_gID])(fill) =
	  localPatchEngine(data()[from_gID])(domlists(p)(i) + offs);
      }
    }
}

//-----------------------------------------------------------------------------
//
// long elementsCompressed()
//
// Compute the number of elements that are currently compressed. Compute
// with the local patches and then do a cross-context reduction.
//
//-----------------------------------------------------------------------------

template<int Dim, class T, class LTag, class PatchTag>
long elementsCompressed(const Engine<Dim, T, MultiPatch<LTag, PatchTag> > 
  &engine)
{
  int size = engine.layout().sizeLocal();

  bool distributed = true;
  if (size > 0 && engine.layout().beginLocal()->context() == -1)
    distributed = false;
    
  long num = 0L;
  for (int i = 0 ; i < size ; ++i )
    num += elementsCompressed(engine.localPatch(i));
  
  if (distributed)
    {
      ReduceOverContexts<long, OpAddAssign> total(num);
      total.broadcast(num);
    }

  return num;
}

//-----------------------------------------------------------------------------
//
// long elementsCompressed()
//
// Compute the number of elements that are currently compressed. Compute
// with the local patches and then do a cross-context reduction. This is
// a little tricky since we must iterate over Nodes here since patch indices
// don't really mean anything for views.
//
//-----------------------------------------------------------------------------

template<int Dim, class T, class LTag, class PatchTag, int Dim2>
long elementsCompressed(const 
  Engine<Dim, T, MultiPatchView<LTag, PatchTag, Dim2> > &engine)
{
  typedef Engine<Dim, T, MultiPatchView<LTag, PatchTag, Dim2> > Engine_t;
  typedef typename Engine_t::Layout_t Layout_t;
  
  int size = engine.layout().sizeLocal();
  bool distributed = true;
  if (size > 0 && engine.layout().beginLocal()->context() == -1)
    distributed = false;

  typename Layout_t::const_iterator i = engine.layout().beginLocal();
  long num = 0L;
  while (i != engine.layout().endLocal())
    {
      num += elementsCompressed(engine.globalPatch(*i));
      ++i;
    }

  if (distributed)
    {  
      ReduceOverContexts<long, OpAddAssign> total(num);
      total.broadcast(num);
    }
    
  return num;
}

//-----------------------------------------------------------------------------
//
// void compress()
//
// (Try to) compress all the local patches.
//
//-----------------------------------------------------------------------------

template<int Dim, class T, class LTag, class PatchTag>
void compress(Engine<Dim, T, MultiPatch<LTag, PatchTag> > &engine)
{
  // Iterate through patches and try to compress them all.
  
  for (int i = 0; i < engine.layout().sizeLocal(); ++i)
    compress(engine.localPatch(i));
}

//-----------------------------------------------------------------------------
//
// void uncompress()
//
// Manually uncompress all the local patches.
//
//-----------------------------------------------------------------------------

template<int Dim, class T, class LTag, class PatchTag>
void uncompress(Engine<Dim, T, MultiPatch<LTag, PatchTag> > &engine)
{
  // Iterate through patches and try to uncompress them all.
  
  for (int i = 0; i < engine.layout().sizeLocal(); ++i)
    uncompress(engine.localPatch(i));
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: MultiPatchEngine.cpp,v $   $Author: sa_smith $
// $Revision: 1.46 $   $Date: 2000/07/12 19:32:00 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
