// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// IO functions to support Pooma Arrays
//-----------------------------------------------------------------------------

#ifndef POOMA_IO_ARRAY_IO_H
#define POOMA_IO_ARRAY_IO_H

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "Pooma/Arrays.h"
#include "IO/CTTI.h"
#include "IO/Serializers.h"
#include "IO/DomainIO.h"
#include "IO/EngineIO.h"
#include "DynamicArray/DynamicArray.h"
#include <string>


//-----------------------------------------------------------
// Functions for Array<Dim,T,EngineTag>
//-----------------------------------------------------------
// CTTI function for Array<Dim,T,EngineTag>
//-----------------------------------------------------------

template <int Dim, class T, class EngineTag>
std::string PoomaCTTI(const Array<Dim, T, EngineTag>&);
template <int Dim, class T, class EngineTag>
std::string PoomaCTTI(const Array<Dim, T, EngineTag>&){
  char numbuf[10];
  sprintf(numbuf,"%d,\0",Dim);
  std::string strng= "Array<"+
    std::string(numbuf)+ PoomaCTTI(T())
    + ","+ PoomaCTTI(EngineTag())+ ">";
  return strng;
}

//-----------------------------------------------------------
// Serializers for Array<Dim,T,EngineTag>
//-----------------------------------------------------------
template <class Stream, int Dim, class T, class EngineTag>
int serialize(Stream& s, const Array<Dim, T, EngineTag>& array);
template <class Stream, int Dim, class T, class EngineTag>
int serialize(Stream& s, const Array<Dim, T, EngineTag>& array){

  typedef Array<Dim,T,EngineTag> Array_t;
  int nBytes=0;

  // Obtain the attributes of the array.
  // Get the array engine.
  Array_t::Engine_t engine= array.engine();

  // Serialize the engine. This is essentially all the data.
  nBytes+= serialize(s,engine);

  return nBytes;
}

template <class Stream, int Dim, class T, class EngineTag>
int deserialize(Array<Dim, T, EngineTag>& array, Stream& s);
template <class Stream, int Dim, class T, class EngineTag>
int deserialize(Array<Dim, T, EngineTag>& array, Stream& s){

  typedef Array<Dim,T,EngineTag> Array_t;
  int nBytes=0;

  // Retrieve the engine from storage.
  // This is essentially all the data.
  Array_t::Engine_t engine;
  nBytes+= deserialize(engine,s);

  // Initialize the array with the restored engine.
  array.initialize(engine);

  return nBytes;
}

template <int Dim, class T, class EngineTag>
int serialSizeof(const Array<Dim, T, EngineTag>& array);
template <int Dim, class T, class EngineTag>
int serialSizeof(const Array<Dim, T, EngineTag>& array){

  typedef Array<Dim,T,EngineTag> Array_t;
  int nBytes=0;

  // Get the array engine.
  Array_t::Engine_t engine= array.engine();

  // Size the engine. This contains essentially all the data.
  nBytes+= serialSizeof(engine);

  return nBytes;
}

//-----------------------------------------------------------
// Functions for DynamicArray<T,EngineTag>
//-----------------------------------------------------------
// CTTI function for DynamicArray<T,EngineTag>
//-----------------------------------------------------------

template <class T, class EngineTag>
std::string PoomaCTTI(const DynamicArray<T, EngineTag>&);
template <class T, class EngineTag>
std::string PoomaCTTI(const DynamicArray<T, EngineTag>&){
  std::string strng= "DynamicArray<"+
    + PoomaCTTI(T())+ ","+ PoomaCTTI(EngineTag())+ ">";
  return strng;
}

//-----------------------------------------------------------
// Serializers for DynamicArray<T,EngineTag>
//-----------------------------------------------------------
template <class Stream, class T, class EngineTag>
int serialize(Stream& s, const DynamicArray<T, EngineTag>& array);
template <class Stream, class T, class EngineTag>
int serialize(Stream& s, const DynamicArray<T, EngineTag>& array){

  typedef DynamicArray<T,EngineTag> Array_t;
  typedef typename Array_t::Base_t Base_t;
  int nBytes=0;

  // This serializer simply casts the dynamic array to its
  // base 1D Array object and serializes it.
  Base_t &baseArray= array.array();
  nBytes+= serialize(s,baseArray);

  return nBytes;
}

template <class Stream, class T, class EngineTag>
int deserialize(DynamicArray<T, EngineTag>& array, Stream& s);
template <class Stream, class T, class EngineTag>
int deserialize(DynamicArray<T, EngineTag>& array, Stream& s){

  typedef DynamicArray<T,EngineTag> Array_t;
  typedef typename Array_t::Base_t Base_t;
  int nBytes=0;

  // This serializer simply casts the dynamic array to its
  // base 1D Array object and deserializes it.
  Base_t &baseArray= array.array();
  nBytes+= deserialize(baseArray,s);

  return nBytes;
}

template <class T, class EngineTag>
int serialSizeof(const DynamicArray<T, EngineTag>& array);
template <class T, class EngineTag>
int serialSizeof(const DynamicArray<T, EngineTag>& array){

  typedef DynamicArray<T,EngineTag> Array_t;
  typedef typename Array_t::Base_t Base_t;
  int nBytes=0;

  // Simply casts the dynamic array to its
  // base 1D Array object and sizes it.
  Base_t &baseArray= array.array();
  nBytes+= serialSizeof(baseArray);

  return nBytes;

}

#endif // POOMA_IO_ARRAY_IO_HY_IO_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ArrayIO.h,v $   $Author: ambro $
// $Revision: 1.11 $   $Date: 2000/07/26 21:53:20 $
// ----------------------------------------------------------------------
// ACL:rcsinfo


