// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
// ----------------------------------------------------------------------
// Updater: intermediate templated base class for all updaters.
// UpdaterCategory: base class for updater categories.
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_UPDATER_UPDATER_H
#define POOMA_NEWFIELD_UPDATER_UPDATER_H

#include "Domain/Loc.h"
#include "NewField/Updater/UpdaterListItem.h"

//-----------------------------------------------------------------------------
// Updater is a base template for all automatic updaters, which are constructed
// as partial specializations to this class. These partial specializations
// have the following interface requirements:
//
// (1) They must inherit publicly from UpdaterBase<Subject, Category>.
// (2) They must define a constructor that takes a const reference to the
//     subject and a const reference to a Category object.
// (3) They must define an update() function that actually does some work.
//-----------------------------------------------------------------------------

template<class Subject, class Category>
class Updater;


//-----------------------------------------------------------------------------
// UpdaterCategory is a base class for the updater category tags.
// It only contains a factory method for creating an updater given
// a subject, which is typically a Field or Array. 
//
// Users wanting to make a particular sort of updater (e.g., 
// ExtrapolateBC, EurekaBC) should inherit from this class using the 
// Glommable pattern; that is,
//
//   class ExtrapolateBC: public UpdaterCategory<ExtrapolateBC> { ... }
//
// They should also define data members in the sub-class for storing 
// information the updater needs to evaluate itself.
//-----------------------------------------------------------------------------

template<class Category>
class UpdaterCategory
{
public:

  //---------------------------------------------------------------------------
  // Default and copy constructors. Trivial since we have no data. Supplied
  // to keep purify from griping.
  
  UpdaterCategory() { }
  UpdaterCategory(const UpdaterCategory<Category> &) { }

  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  UpdaterCategory<Category> &operator=(const UpdaterCategory<Category> &)
  {
    return *this;
  }
  
  //---------------------------------------------------------------------------
  // Factory method. Actually makes the updater.
  
  template<class Subject>
  UpdaterListItem *create(const Subject &s) const
  {
    return new Updater<Subject, Category>
      (s, static_cast<const Category &>(*this));
  }
};

#endif // POOMA_NEWFIELD_UPDATER_UPDATER_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Updater.h,v $   $Author: swhaney $
// $Revision: 1.2 $   $Date: 2000/07/20 21:03:51 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
