// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_DOMAIN_DOMAIN_ITERATOR_H
#define POOMA_DOMAIN_DOMAIN_ITERATOR_H

//-----------------------------------------------------------------------------
// Classes: 
//   DomainIterator<Dom>
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
//
//   DomainIterator<Dom> - Iterates through domain data (of type Dom)
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Utilities/PAssert.h"

#include <stddef.h> // ptrdiff_t
#include <iterator>

//-----------------------------------------------------------------------------
//
// Full Description:
// DomainIterator<Dom>
//
// A simple iterator class to iterate through all of the points
// in a given domain of type Dom. The individual points are returned
// as Loc<Dim>s or Region<Dim,T>'s when the iterator is dereferenced.
//
// This is an input-iterator, in the STL sense.  It only defines deref,
// ->, and ++ operators.
//
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Open POOMA namespace:
//-----------------------------------------------------------------------------

// namespace POOMA {

template <class Dom>
class DomainIterator
{
public:

  //============================================================
  // Typedefs and enumerations
  //============================================================

  typedef DomainIterator<Dom>            This_t;
  typedef Dom                            Domain_t;
  typedef typename Domain_t::AskDomain_t Value_t;

  typedef std::input_iterator_tag        iterator_category;
  typedef Value_t                        value_type;
  typedef ptrdiff_t                      difference_type;
  typedef const Value_t*                 pointer;
  typedef const Value_t&                 reference;
  
  enum { dimensions = Domain_t::dimensions };

  //============================================================
  // Constructors
  //============================================================

  // The main DomainIterator stores the given domain, and sets all its 1D
  // iterators to the start. This constructor sets up a "begin" iterator.

  DomainIterator(const Dom &d, int size = 0)
    : index_m(size), domain_m(d), loc_m(d.firsts())
    {
      PAssert(index_m >= 0 && index_m <= domain_m.size());
      for (int i=0; i < dimensions; ++i)
	current_m[i] = 0;
    }

  // Copy constructor.

  DomainIterator(const This_t &model)
    : index_m(model.index_m), domain_m(model.domain_m), loc_m(model.loc_m)
    {
      PAssert(index_m >= 0 && index_m <= domain_m.size());
      for (int i=0; i < dimensions; ++i)
	current_m[i] = model.current_m[i];
    }

  // The default constructor constructs an end iterator for an empty
  // domain.

  DomainIterator()
    : index_m(0)
    {
      for (int i=0; i < dimensions; ++i)
	current_m[i] = 0;
    }

  //============================================================
  // Accessors
  //============================================================

  // Dereference operator. Returns const ref to internal Loc.

  inline const Value_t &operator*() const
    {
      PAssert(!done());
      return loc_m;
    }

  // Member selection operator. Allows calling const Loc
  // member functions. Not too useful, but it is part of
  // the required input iterator interface. 

  inline const Value_t *operator->() const
    {
      PAssert(!done());
      return &loc_m;
    }

  // Equality tests.
  // Note that any two iterators that are both marked
  // as being at the end of iteration will compare equal.

  inline bool operator==(const This_t &rhs) const
    {
      return (index_m == rhs.index_m);
    }

  inline bool operator!=(const This_t &rhs) const
    {
      return (index_m != rhs.index_m);
    }

  // At-end (false) test.
  // Returns true if this iterator is at-end.

  bool done() const
    {
      return (index_m >= domain_m.size());
    }

  //============================================================
  // Mutators
  //============================================================

  // Assignment operator.

  This_t &operator=(const This_t &model)
    {
      index_m   = model.index_m;
      domain_m  = model.domain_m;
      loc_m     = model.loc_m;
      for (int i=0; i < dimensions; ++i)
	current_m[i] = model.current_m[i];
      return *this;
    }

  // Pre-increment operator. 
  // Takes us to the next point in the Interval<Dim> space of
  // points. This is done in Fortran (column-major) order.

  This_t &operator++()
    { 
      increment();
      return *this;
    }

  // Post-increment operator.
  // This has to make a copy, so prefer the above if possible.

  This_t operator++(int)
    {
      DomainIterator<Dom> save(*this);
      increment();
      return save;
    }

private:

  //============================================================
  // Data members.
  //============================================================

  // The domain we're iterating over.

  Domain_t domain_m;

  // Our current value, stored as a point domain.

  Value_t loc_m;

  // Our current position in each dimension.

  int current_m[dimensions];

  // Our current total index.

  int index_m;

  //============================================================
  // Implementation functions
  //============================================================

  // Increment iterator.

  void increment()
    {
      PAssert(!done());

      for (int i = 0; i < dimensions; ++i)
	{
	  if (++(current_m[i]) >= domain_m[i].length())
	    {
	      if (i < dimensions-1)
		{
		  current_m[i] = 0;
		  loc_m[i] = domain_m[i].first();
		}
	    }
	  else
	    {
	      loc_m[i] = domain_m[i](current_m[i]);
	      break;
	    }
	}

      // increase our total index

      ++index_m;
    }
};


// } // namespace POOMA

#endif // POOMA_DOMAIN_DOMAIN_ITERATOR_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: DomainIterator.h,v $   $Author: jac $
// $Revision: 1.7 $   $Date: 2000/04/01 00:26:22 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
